"use client";

import { useEffect, useMemo, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import type { ActionItem, ActionItemStatus } from "@/api/routes/action-items";
import {
  Card,
  CardBody,
  CardMeta,
  Chip,
  ToolbarRow,
  TextInput,
  Button,
  SecondaryButton,
  DangerButton,
  Divider,
  EmptyState,
} from "@/components/ui/CardBits";
import { useRouter } from "next/navigation";
import AssigneesPicker from "@/components/cases/AssigneesPicker";
import type {
  ActionItemAssignee,
  ActionItemAssigneeRole,
} from "@/api/routes/action-items";

function fmt(iso?: string | null) {
  if (!iso) return "-";
  try {
    return new Date(iso).toLocaleString();
  } catch {
    return iso ?? "-";
  }
}

function isOverdue(due?: string | null) {
  if (!due) return false;
  return new Date(due).getTime() < Date.now();
}

function AssigneesEditor(props: {
  actionItemId: string;
  current: ActionItemAssignee[];
  disabled?: boolean;
  onSave: (
    payload: Array<{ userId: string; role: ActionItemAssigneeRole }>
  ) => void;
}) {
  const { current, onSave, disabled } = props;

  // a picker Sel struktúrájához alakítjuk
  const [draft, setDraft] = useState<
    Array<{
      user: {
        id: string;
        name?: string | null;
        email?: string | null;
        username?: string | null;
      };
      role: ActionItemAssigneeRole;
    }>
  >([]);

  useEffect(() => {
    setDraft(
      (current ?? []).map((x) => ({
        user: {
          id: x.user_id,
          name: x.name,
          email: x.email,
          username: x.username,
        },
        role: x.role,
      }))
    );
  }, [current]);

  const payload = useMemo(
    () => draft.map((x) => ({ userId: x.user.id, role: x.role })),
    [draft]
  );

  return (
    <div style={{ display: "grid", gap: 10, opacity: disabled ? 0.6 : 1 }}>
      <AssigneesPicker value={draft as any} onChange={setDraft as any} />

      <ToolbarRow>
        <Button onClick={() => onSave(payload)} disabled={disabled}>
          Mentés
        </Button>
      </ToolbarRow>
    </div>
  );
}

export default function ActionItemsPanel(props: {
  caseId: string; // kötelező
  caseRef?: string;
  threadId?: string | null;
  decisionId?: string | null;
  meetingId?: string | null;

  title?: string; // UI cím a panel felett (ha a page-en SectionTitle-ozol, nem muszáj)
  mineDefault?: 1 | 0; // default 0 (case szinten általában mindenki lássa)
  includeArchivedDefault?: 0 | 1; // default 0

  showCreate?: boolean; // default true
}) {
  const router = useRouter();

  const {
    caseId,
    caseRef,
    threadId,
    decisionId,
    meetingId,
    mineDefault = 0,
    includeArchivedDefault = 0,
    showCreate = true,
  } = props;

  const [items, setItems] = useState<ActionItem[]>([]);
  const [loading, setLoading] = useState(false);

  const [mine, setMine] = useState<1 | 0>(mineDefault);
  const [includeArchived, setIncludeArchived] = useState<0 | 1>(
    includeArchivedDefault
  );

  const [newTitle, setNewTitle] = useState("");
  const [newDue, setNewDue] = useState("");
  const [creating, setCreating] = useState(false);

  // --- assignees per action item (cache)
  const [assigneesById, setAssigneesById] = useState<
    Record<string, ActionItemAssignee[]>
  >({});
  const [assigneesLoadingById, setAssigneesLoadingById] = useState<
    Record<string, boolean>
  >({});
  const [assigneesSavingById, setAssigneesSavingById] = useState<
    Record<string, boolean>
  >({});
  const [assigneesOpenById, setAssigneesOpenById] = useState<
    Record<string, boolean>
  >({});

  const filtered = useMemo(() => {
    return (items ?? []).filter((x) => {
      if (threadId && x.thread_id !== threadId) return false;
      if (decisionId && x.decision_id !== decisionId) return false;
      if (meetingId && x.meeting_id !== meetingId) return false;
      return true;
    });
  }, [items, threadId, decisionId, meetingId]);

  async function load() {
    if (!caseId) return;
    setLoading(true);
    try {
      const r = await ApiCall.actionItems.list({
        caseId,
        ...(mine ? { mine: 1 } : {}),
        ...(includeArchived ? { includeArchived: 1 } : {}),
      });

      setItems(r.data.items ?? []);
    } catch (e: any) {
      toast.error(
        e?.response?.data?.message ?? "Feladatok betöltése sikertelen"
      );
      setItems([]);
    } finally {
      setLoading(false);
    }
  }

  useEffect(() => {
    void load();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [caseId, mine, includeArchived, threadId, decisionId, meetingId]);

  async function create() {
    if (!caseId) return;
    if (!newTitle.trim()) return;

    setCreating(true);
    try {
      await ApiCall.actionItems.create({
        caseId,
        title: newTitle.trim(),
        dueAt: newDue ? new Date(newDue).toISOString() : undefined,
        threadId: threadId ?? undefined,
        decisionId: decisionId ?? undefined,
        meetingId: meetingId ?? undefined,
      });

      toast.success("Feladat létrehozva");
      setNewTitle("");
      setNewDue("");
      await load();
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Létrehozás sikertelen");
    } finally {
      setCreating(false);
    }
  }

  async function markDone(id: string) {
    try {
      await ApiCall.actionItems.patch(id, {
        status: "done" as ActionItemStatus,
      });
      toast.success("Készre állítva");
      await load();
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Mentés sikertelen");
    }
  }

  async function archive(id: string) {
    if (!confirm("Archiválod ezt a feladatot?")) return;
    try {
      await ApiCall.actionItems.archive(id);
      toast.success("Archiválva");
      await load();
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Archiválás sikertelen");
    }
  }

  async function loadAssignees(actionItemId: string) {
    // ha már van cache, ne spameld
    if (assigneesById[actionItemId]) return;

    setAssigneesLoadingById((p) => ({ ...p, [actionItemId]: true }));
    try {
      const r = await ApiCall.actionItems.assignees.get(actionItemId);
      setAssigneesById((p) => ({
        ...p,
        [actionItemId]: r.data.assignees ?? [],
      }));
    } catch (e: any) {
      toast.error(
        e?.response?.data?.message ?? "Hozzárendeltek betöltése sikertelen"
      );
      setAssigneesById((p) => ({ ...p, [actionItemId]: [] }));
    } finally {
      setAssigneesLoadingById((p) => ({ ...p, [actionItemId]: false }));
    }
  }

  async function saveAssignees(
    actionItemId: string,
    next: Array<{ userId: string; role: ActionItemAssigneeRole }>
  ) {
    setAssigneesSavingById((p) => ({ ...p, [actionItemId]: true }));
    try {
      await ApiCall.actionItems.assignees.patch(actionItemId, {
        assignees: next,
      });
      toast.success("Hozzárendeltek mentve");
      // frissítsük a cache-t úgy, hogy a UI a szerkesztett állapotot mutassa
      await refreshAssignees(actionItemId);
    } catch (e: any) {
      toast.error(
        e?.response?.data?.message ?? "Hozzárendeltek mentése sikertelen"
      );
    } finally {
      setAssigneesSavingById((p) => ({ ...p, [actionItemId]: false }));
    }
  }

  async function refreshAssignees(actionItemId: string) {
    setAssigneesLoadingById((p) => ({ ...p, [actionItemId]: true }));
    try {
      const r = await ApiCall.actionItems.assignees.get(actionItemId);
      setAssigneesById((p) => ({
        ...p,
        [actionItemId]: r.data.assignees ?? [],
      }));
    } finally {
      setAssigneesLoadingById((p) => ({ ...p, [actionItemId]: false }));
    }
  }

  function toggleAssignees(actionItemId: string) {
    setAssigneesOpenById((p) => {
      const next = !p[actionItemId];
      // ha most nyitjuk ki, töltsük be
      if (next) void loadAssignees(actionItemId);
      return { ...p, [actionItemId]: next };
    });
  }

  return (
    <Card>
      <CardBody className="space-y-3">
        <ToolbarRow>
          <Chip tone="neutral">
            {loading ? "..." : `${filtered.length} db`}
          </Chip>

          <select
            value={mine}
            onChange={(e) => setMine(Number(e.target.value) as any)}
            style={{
              border: "1px solid rgba(148,163,184,.35)",
              borderRadius: 10,
              padding: "8px 10px",
              background: "transparent",
            }}>
            <option value={0}>Minden (jogosultság szerint)</option>
            <option value={1}>Csak az enyém</option>
          </select>

          <select
            value={includeArchived}
            onChange={(e) => setIncludeArchived(Number(e.target.value) as any)}
            style={{
              border: "1px solid rgba(148,163,184,.35)",
              borderRadius: 10,
              padding: "8px 10px",
              background: "transparent",
            }}>
            <option value={0}>Csak aktív</option>
            <option value={1}>Archiváltal együtt</option>
          </select>

          <SecondaryButton onClick={() => void load()} disabled={loading}>
            {loading ? "..." : "Frissít"}
          </SecondaryButton>
        </ToolbarRow>

        {showCreate ? (
          <>
            <Divider />
            <ToolbarRow>
              <TextInput
                value={newTitle}
                onChange={(e) => setNewTitle(e.target.value)}
                placeholder="Új feladat címe…"
                style={{ flex: 1, minWidth: 260 }}
              />
              <TextInput
                type="datetime-local"
                value={newDue}
                onChange={(e) => setNewDue(e.target.value)}
                placeholder="Határidő"
              />
              <Button
                onClick={() => void create()}
                disabled={creating || !newTitle.trim()}>
                {creating ? "..." : "Hozzáad"}
              </Button>
            </ToolbarRow>
          </>
        ) : null}

        <Divider />

        {filtered.length ? (
          <div className="grid gap-3">
            {filtered.map((t) => {
              const overdue = isOverdue(t.due_at);
              const tone =
                t.status === "done"
                  ? "good"
                  : t.status === "blocked"
                  ? "bad"
                  : overdue
                  ? "warn"
                  : "neutral";

              return (
                <Card key={t.id}>
                  <CardBody className="space-y-2">
                    <div
                      style={{
                        display: "flex",
                        justifyContent: "space-between",
                        gap: 12,
                      }}>
                      <div style={{ minWidth: 0 }}>
                        <div style={{ fontWeight: 800 }}>{t.title}</div>
                        <CardMeta>
                          status: {t.status} · due: {fmt(t.due_at)}
                        </CardMeta>
                      </div>

                      <div
                        style={{ display: "flex", gap: 8, flexWrap: "wrap" }}>
                        <Chip tone={tone as any}>{t.status}</Chip>
                        {t.archived_at ? (
                          <Chip tone="warn">archived</Chip>
                        ) : null}
                      </div>
                    </div>

                    <ToolbarRow>
                      <SecondaryButton
                        onClick={() => {
                          const ref = t.code ?? t.id;
                          const url = caseRef
                            ? `/admin/cases/${caseRef}/action-items?focus=${ref}`
                            : `/admin/action-items?q=${ref}`;
                          router.push(url);
                        }}>
                        Megnyitás
                      </SecondaryButton>

                      {/* <SecondaryButton
                        onClick={() =>
                          //   router.push(`/admin/action-items?q=${t.code ?? t.id}`)
                          router.push(
                            `/admin/cases/${caseRef}/action-items?focus=${
                              t.code ?? t.id
                            }`
                          )
                        }>
                        Megnyitás
                      </SecondaryButton> */}

                      <Button
                        onClick={() => void markDone(t.id)}
                        disabled={t.status === "done" || !!t.archived_at}>
                        Kész
                      </Button>

                      <DangerButton
                        onClick={() => void archive(t.id)}
                        disabled={!!t.archived_at}>
                        Archivál
                      </DangerButton>
                    </ToolbarRow>
                    <Divider />

                    <ToolbarRow>
                      <SecondaryButton onClick={() => toggleAssignees(t.id)}>
                        {assigneesOpenById[t.id]
                          ? "Hozzárendeltek elrejtése"
                          : "Hozzárendeltek"}
                      </SecondaryButton>

                      {assigneesSavingById[t.id] ? (
                        <Chip tone="neutral">mentés...</Chip>
                      ) : null}
                      {assigneesLoadingById[t.id] ? (
                        <Chip tone="neutral">betöltés...</Chip>
                      ) : null}

                      {/* quick summary */}
                      <Chip tone="neutral">
                        {assigneesById[t.id]?.length ?? 0
                          ? `${assigneesById[t.id]!.length} fő`
                          : "nincs"}
                      </Chip>
                    </ToolbarRow>

                    {assigneesOpenById[t.id] ? (
                      <div style={{ marginTop: 8 }}>
                        <CardMeta>Hozzárendeltek (több ember + role)</CardMeta>

                        <AssigneesEditor
                          actionItemId={t.id}
                          current={assigneesById[t.id] ?? []}
                          onSave={(payload) =>
                            void saveAssignees(t.id, payload)
                          }
                          disabled={
                            !!t.archived_at || !!assigneesSavingById[t.id]
                          }
                        />
                      </div>
                    ) : null}
                  </CardBody>
                </Card>
              );
            })}
          </div>
        ) : (
          <EmptyState
            title="Nincs feladat"
            hint="Adj hozzá teendőt — így a döntés előkészítés és a végrehajtás is követhető lesz."
          />
        )}
      </CardBody>
    </Card>
  );
}
