import {
  parse,
  isValidNumber,
  parsePhoneNumberFromString,
  formatInternational,
  parsePhoneNumber,
  AsYouType,
} from "libphonenumber-js";

const VALIDATOR_TYPE_REQUIRE = "REQUIRE";
const VALIDATOR_TYPE_MINLENGTH = "MINLENGTH";
const VALIDATOR_TYPE_MAXLENGTH = "MAXLENGTH";
const VALIDATOR_TYPE_MINWORDCOUNT = "MINWORDCOUNT";
const VALIDATOR_TYPE_MIN = "MIN";
const VALIDATOR_TYPE_MAX = "MAX";
const VALIDATOR_TYPE_EMAIL = "EMAIL";
const VALIDATOR_TYPE_FILE = "FILE";
const VALIDATOR_TYPE_NUMBER = "NUMBER";
const VALIDATOR_TYPE_NUMRANGE = "NUMRANGE";
const VALIDATOR_TYPE_PASSWORD = "PASSWORD";
const VALIDATOR_TYPE_PHONENUMBER_HUN = "PHONE_HU";
const VALIDATOR_TYPE_PHONENUMBER_AT = "PHONE_AT";
const VALIDATOR_TYPE_TAXNUMBER_HUN = "TAXNUMBER_HU";
const VALIDATOR_TYPE_ATU = "ATU";

export const VALIDATOR_REQUIRE = () => ({ type: VALIDATOR_TYPE_REQUIRE });
export const VALIDATOR_FILE = () => ({ type: VALIDATOR_TYPE_FILE });
export const VALIDATOR_ATU = () => ({ type: VALIDATOR_TYPE_ATU });

export const VALIDATOR_MINLENGTH = (val) => ({
  type: VALIDATOR_TYPE_MINLENGTH,
  val: val,
});
export const VALIDATOR_MAXLENGTH = (val) => ({
  type: VALIDATOR_TYPE_MAXLENGTH,
  val: val,
});
export const VALIDATOR_MINWORDCNT = () => ({
  type: VALIDATOR_TYPE_MINWORDCOUNT,
});
export const VALIDATOR_MIN = (val) => ({ type: VALIDATOR_TYPE_MIN, val: val });
export const VALIDATOR_MAX = (val) => ({ type: VALIDATOR_TYPE_MAX, val: val });
export const VALIDATOR_EMAIL = () => ({ type: VALIDATOR_TYPE_EMAIL });
export const VALIDATOR_NUMBER = () => ({ type: VALIDATOR_TYPE_NUMBER });
export const VALIDATOR_NUMRANGE = () => ({ type: VALIDATOR_TYPE_NUMRANGE });
export const VALIDATOR_PASSWORD = () => ({
  type: VALIDATOR_TYPE_PASSWORD,
  min: 8,
  lcase: true,
  ucase: true,
  num: true,
  message: "Min. 8 karakter, kis- és nagybetű, vagy szám szükséges!",
});
export const VALIDATOR_PHONE_HU = () => ({
  type: VALIDATOR_TYPE_PHONENUMBER_HUN,
});
export const VALIDATOR_PHONE_AT = () => ({
  type: VALIDATOR_TYPE_PHONENUMBER_AT,
});
export const VALIDATOR_TAXNUMBER_HU = () => ({
  type: VALIDATOR_TYPE_TAXNUMBER_HUN,
});

export const validate = (value, validators) => {
  let isValid = true;
  for (const validator of validators) {
    if (validator.type === VALIDATOR_TYPE_REQUIRE) {
      isValid = isValid && value.trim().length > 0;
    }
    if (validator.type === VALIDATOR_TYPE_MINLENGTH) {
      isValid = isValid && value.trim().length >= validator.val;
    }
    if (validator.type === VALIDATOR_TYPE_MAXLENGTH) {
      isValid = isValid && value.trim().length <= validator.val;
    }
    if (validator.type === VALIDATOR_TYPE_MINWORDCOUNT) {
      isValid = isValid && /(?=.{2,30}$)(\w.+\s).+/.test(value);
    }
    if (validator.type === VALIDATOR_TYPE_MIN) {
      isValid = isValid && +value >= validator.val;
    }
    if (validator.type === VALIDATOR_TYPE_MAX) {
      isValid = isValid && +value <= validator.val;
    }
    if (validator.type === VALIDATOR_TYPE_NUMRANGE) {
      isValid = isValid && /^(?:[0-9]|[1-4][0-9]|50)$/.test(value);
    }
    if (validator.type === VALIDATOR_TYPE_NUMBER) {
      isValid = isValid && /^-?\d+(\.\d+)?$/.test(value);
      // isValid = isValid && /^[0-9]+(\.[0-9]+)?$/.test(value); // this allows character "e" because it is a valid notation for scientific notation, where e represent exponentiation.
      // isValid = isValid && /^[0-9]+(\.[0-9]+)?$/.test(value);
    }
    if (validator.type === VALIDATOR_TYPE_EMAIL) {
      isValid =
        isValid && /^\w+([-+.']\w+)*@\w+([-.]\w+)*\.\w+([-.]\w+)*$/.test(value);
    }
    if (validator.type === VALIDATOR_TYPE_PASSWORD) {
      isValid = true;

      if (value.trim().length < validator.min) {
        isValid = false;
      }

      if (validator.lcase && !/(?=.*[a-z])/.test(value)) {
        isValid = false;
      }

      if (validator.ucase && !/(?=.*[A-Z])/.test(value)) {
        isValid = false;
      }

      if (validator.num && !/(?=.*\d)/.test(value)) {
        isValid = false;
      }

      return isValid;
    }
    if (validator.type === VALIDATOR_TYPE_PHONENUMBER_HUN) {
      try {
        const phoneNumber = parsePhoneNumber(value, "HU");
        if (!phoneNumber) {
          return (isValid = false);
        }
        isValid = isValid && phoneNumber.isValid();
      } catch (error) {
        isValid = false;
      }
    }
    if (validator.type === VALIDATOR_TYPE_PHONENUMBER_AT) {
      try {
        const phoneNumber = parsePhoneNumber(value, "AT");
        if (!phoneNumber) {
          return (isValid = false);
        }
        isValid = isValid && phoneNumber.isValid();
      } catch (error) {
        isValid = false;
      }
    }
    if (validator.type === VALIDATOR_TYPE_TAXNUMBER_HUN) {
      isValid = isValid && /^\d{8}-\d{1}-\d{2}$/.test(value);
    }
    if (validator.type === VALIDATOR_TYPE_ATU) {
      isValid =
        isValid && value.trim().length > 0 && /^ATU[A-Z0-9]{8}$/.test(value);
      if (!isValid) return validator.message;
    }
  }
  return isValid;
};
