// redux/slices/pageEditorSlice.js
import { createSlice } from "@reduxjs/toolkit";
import {
  containerInitialState,
  getContainerReducers,
} from "../factory/containerSliceFactory";
import {
  columnInitialState,
  getColumnReducers,
} from "../factory/columnSliceFactory";
import {
  elementInitialState,
  getElementReducers,
} from "../factory/elementSliceFactory";
import {
  galleryInitialState,
  getGalleryReducers,
} from "../factory/gallerySliceFactory";
import {
  commonPropsInitialState,
  getCommonPropsReducers,
} from "../factory/commonPropsSliceFactory";

// Custom initial state
const customInitialState = {
  selectedProduct: {},
  selectedArticle: {},
  selectedForm: {},
  selectedBlogs: [],
  selectedCategories: [],
  isPublishedPage: true,
  hasParentPage: false,
  showPageInMenu: true,
  selectedParentPage: [],
  currentChildPages: [],
};

const initialState = {
  ...containerInitialState,
  ...columnInitialState,
  ...elementInitialState,
  ...galleryInitialState,
  ...commonPropsInitialState,
  ...customInitialState,
};

const clamp = (n, min, max) => Math.max(min, Math.min(max, n));

const moveItem = (arr, from, to) => {
  if (from === to) return arr;
  const item = arr.splice(from, 1)[0];
  arr.splice(to, 0, item);
  return arr;
};

// Remap innerId (column indexes) for all blocks in a container when a column moves
const remapInnerIdsOnColumnMove = (content, fromIdx, toIdx) => {
  for (const blk of content ?? []) {
    if (blk?.innerId === undefined) continue;
    const inner = Number(blk.innerId);
    if (inner === fromIdx) {
      blk.innerId = toIdx;
    } else if (fromIdx < toIdx && inner > fromIdx && inner <= toIdx) {
      blk.innerId = inner - 1;
    } else if (fromIdx > toIdx && inner >= toIdx && inner < fromIdx) {
      blk.innerId = inner + 1;
    }
  }
};

// Ensure innerElementIndex is continuous 0..n-1
const reindexElements = (block) => {
  (block?.content ?? []).forEach((el, i) => (el.innerElementIndex = i));
};

// Find or create a column block (by parentId/innerId)
const getOrCreateColumnBlock = (container, parentIdx, columnIdx) => {
  let blk = (container.content ?? []).find(
    (b) => Number(b.parentId) === parentIdx && Number(b.innerId) === columnIdx
  );
  if (!blk) {
    blk = { parentId: parentIdx, innerId: columnIdx, content: [] };
    if (!container.content) container.content = [];
    container.content.push(blk);
  }
  return blk;
};

const pageEditorSlice = createSlice({
  name: "pageEditor",
  initialState,
  reducers: {
    resetPageEditorState: () => initialState,
    resetCurrent: (state, action) => {
      state.current = {};
    },
    // Custom additions
    setPublishedPage: (state, action) => {
      state.isPublishedPage = action.payload.isPublishedPage;
    },
    setHasParentPage: (state, action) => {
      state.hasParentPage = action.payload.hasParentPage;
    },
    setShowPageInMenu: (state, action) => {
      state.showPageInMenu = action.payload.showPageInMenu;
    },
    setSelectedParentPage: (state, action) => {
      state.selectedParentPage = [action.payload.selectedParentPage];
    },
    setCurrentChildPages: (state, action) => {
      state.currentChildPages = action.payload.currentChildPages;
    },
    //
    selectSingleProduct: (state, action) => {
      state.selectedProduct = action.payload.product;
    },
    clearSelectSingleProduct: (state) => {
      state.selectedProduct = null;
    },
    selectSingleForm: (state, action) => {
      state.selectedForm = action.payload.form;
    },
    clearSelectSingleForm: (state) => {
      state.selectedForm = {};
    },
    selectSingleArticle: (state, action) => {
      state.selectedArticle = action.payload.article;
    },
    clearSelectSingleArticle: (state) => {
      state.selectedArticle = {};
    },
    displayedBlogs: (state, action) => {
      state.selectedBlogs.push(action.payload.blog);
    },
    clearDisplayedBlogs: (state) => {
      state.selectedBlogs = [];
    },
    displayedCategories: (state, action) => {
      state.selectedCategories.push(action.payload.category);
    },
    clearDisplayedCategories: (state) => {
      state.selectedCategories = [];
    },
    loadPageEditorInitialData: (state, action) => {
      const { data } = action.payload;

      if (data) {
        state.isPublishedPage = data.isPublished ?? true;
        state.hasParentPage = data.hasParent ?? false;
        state.showPageInMenu = data.showInMenu ?? false;
        state.selectedParentPage = data.parent ?? [];
        state.currentChildPages = data.children ?? [];
      }
    },
    reorderContainersByPosition: (state, action) => {
      const { fromIndex, toPos1Based } = action.payload;
      const arr = state.containerData ?? [];
      if (!arr.length) return;
      const from = clamp(fromIndex, 0, arr.length - 1);
      const to = clamp((toPos1Based ?? 1) - 1, 0, arr.length - 1);
      moveItem(arr, from, to);

      // After reordering containers, fix every block.parentId to match its new container index
      arr.forEach((container, idx) => {
        (container.content ?? []).forEach((blk) => (blk.parentId = idx));
      });
    },

    // 2) COLUMNS: move a column within same container to position (1-based)
    reorderColumnsByPosition: (state, action) => {
      const { containerIndex, fromIndex, toPos1Based } = action.payload;
      const container = state.containerData?.[containerIndex];
      if (!container) return;

      const size = container.size ?? [];
      if (!size.length) return;

      const from = clamp(fromIndex, 0, size.length - 1);
      const to = clamp((toPos1Based ?? 1) - 1, 0, size.length - 1);

      // Move visual slot (widths)
      moveItem(size, from, to);

      // Remap blocks' innerId to follow the index change
      remapInnerIdsOnColumnMove(container.content ?? [], from, to);
    },

    // 3) ELEMENTS: move element anywhere by container/column/position (1-based position)
    moveElementByPosition: (state, action) => {
      const {
        fromContainerIndex,
        fromColumnIndex,
        fromElementIndex,
        toContainerIndex,
        toColumnIndex,
        toPos1Based,
      } = action.payload;

      const fromC = state.containerData?.[fromContainerIndex];
      const toC = state.containerData?.[toContainerIndex];
      if (!fromC || !toC) return;

      const fromBlock = getOrCreateColumnBlock(
        fromC,
        fromContainerIndex,
        fromColumnIndex
      );
      const toBlock = getOrCreateColumnBlock(
        toC,
        toContainerIndex,
        toColumnIndex
      );

      const fromArr = fromBlock.content ?? [];
      const toArr = toBlock.content ?? [];

      if (
        !fromArr.length ||
        fromElementIndex < 0 ||
        fromElementIndex >= fromArr.length
      )
        return;

      const [moved] = fromArr.splice(fromElementIndex, 1);
      // Update its parent/inner ids
      moved.parentId = toContainerIndex;
      moved.innerId = toColumnIndex;

      const insertAt = clamp(
        (toPos1Based ?? toArr.length + 1) - 1,
        0,
        toArr.length
      );
      toArr.splice(insertAt, 0, moved);

      // Reindex
      reindexElements(fromBlock);
      reindexElements(toBlock);
    },
    ...getContainerReducers(),
    ...getColumnReducers(),
    ...getElementReducers(),
    ...getGalleryReducers(),
    ...getCommonPropsReducers(),
  },
});

export const {
  resetPageEditorState,
  selectSingleProduct,
  clearSelectSingleProduct,
  selectSingleForm,
  clearSelectSingleForm,
  selectSingleArticle,
  clearSelectSingleArticle,
  displayedBlogs,
  clearDisplayedBlogs,
  displayedCategories,
  clearDisplayedCategories,
  setPublishedPage,
  setHasParentPage,
  setShowPageInMenu,
  setSelectedParentPage,
  setCurrentChildPages,
  loadPageEditorInitialData,
  reorderContainersByPosition,
  reorderColumnsByPosition,
  moveElementByPosition,
  // Gallery-related actions injected via factory
  setFeaturedImage,
  resetFeaturedImageState,
  setSelectedGalleryImages,
  removeImageFromGallery,
  updateGalleryImage,
  updateGalleryImageOrder,
  resetGalleryState,
  setSelectedSliderImages,
  updateSliderImage,
  removeSingleSliderImage,
  updateSliderImageOrder,
  cloneSliderImage,
  resetSliderState,
  setSelectedImage,
  resetImageState,
  cloneGalleryImage,
  removeSingleGalleryImage,
  selectedImage,
  // Container-related actions injected via factory
  addContainer,
  removeContainer,
  setSelectedContainer,
  setContainerList,
  updateContainerContent,
  updateContainerOrder,
  selectedContainerOptions,
  setContainerTitle,
  resetContainerState,
  cleanCurrentlySelectedContent,
  cloneContainer,
  updateContainerOptions,
  // Column-related actions injected via factory
  addColumn,
  setSelectedColumn,
  updateColumnOrder,
  updateColumnOptions,
  changeColumnSize,
  cloneColumn,
  removeColumn,
  moveColumnAcrossContainers,
  // Element-related actions injected via factory
  setSelectCurrentElement,
  setEditedElement,
  selectedElement,
  cloneElement,
  updateColumnElementOrder,
  moveColumnElement,
  removeElement,
  addcurrentElementToColumn,
  resetElementState,
  editElement,
  resetCurrent,
  moveElementAcrossContainers,
  // Common props actions injected via factory
  publishedPage,
  activeLayout,
  staticPage,
  hasParent,
  showInMenu,
  resetCommonPropsState,
} = pageEditorSlice.actions;

export default pageEditorSlice.reducer;
