import fs from "fs-extra";
import path from "path";
import ExcelJS from "exceljs";

// ----------------- Helper: Detect company code -----------------
function detectCompanyCode(fileName) {
  const nameWithoutExt = fileName.replace(/\.pdf$/i, "");
  const match = nameWithoutExt.match(/^([A-Z]{2,4})/);
  return match ? match[1] : "";
}

// ----------------- Helper: Extract partner company -----------------
const CONTRACT_KEYWORDS = [
  "megb",
  "szerz",
  "vizsgálat",
  "értesítés",
  "adó",
  "igazolás",
  "számla",
  "hozzájárulás",
  "vizsga",
  "bérleti",
  "befizetés",
  "hosszabítás",
  "értesít",
  "módosítás",
  "önellenőrzés",
  "beszerzés",
].map((kw) => kw.toLowerCase());

function extractPartnerCompany(fileName, companyCode) {
  let nameWithoutExt = fileName.replace(/\.pdf$/i, "");
  let nameWithoutCompany = nameWithoutExt.replace(
    new RegExp(`^${companyCode}[-_ ]?`, "i"),
    ""
  );

  nameWithoutCompany = nameWithoutCompany.replace(/[_-]?\d{8}$/, "");

  const words = nameWithoutCompany.split(/[-_ ]+/);

  const partnerWords = [];
  for (const word of words) {
    const lower = word.toLowerCase();
    if (CONTRACT_KEYWORDS.some((kw) => lower.includes(kw))) {
      break;
    }
    partnerWords.push(word);
  }

  return partnerWords.join(" ").trim() || "Unknown";
}

// ----------------- Helper: Parse filename -----------------
function parseFileName(fileName) {
  const baseName = fileName.replace(/\.pdf$/i, "");
  const companyCode = detectCompanyCode(baseName);
  const parts = baseName.split("_");
  const lastPart = parts[parts.length - 1];

  let date = "";
  let cleanedName = baseName;

  if (/^\d{8}$/.test(lastPart)) {
    date = `${lastPart.slice(0, 4)}-${lastPart.slice(4, 6)}-${lastPart.slice(
      6,
      8
    )}`;
    cleanedName = parts.slice(0, -1).join("_");
  }

  return {
    companyCode,
    cleanedName,
    date,
    hasValidCompany: !!companyCode,
    hasValidDate: !!date,
  };
}

// ----------------- Export to Excel -----------------
export default async function exportContractsToExcel(rootPath, outputPath) {
  const workbook = new ExcelJS.Workbook();
  const sheet = workbook.addWorksheet("Contracts");

  const rows = [];
  let maxLevels = 0;

  async function walk(dirPath, levels = []) {
    const items = fs.readdirSync(dirPath, { withFileTypes: true });

    for (const item of items) {
      const fullPath = path.join(dirPath, item.name);

      if (item.isDirectory()) {
        await walk(fullPath, [...levels, item.name]);
      } else {
        const {
          companyCode,
          cleanedName,
          date,
          hasValidCompany,
          hasValidDate,
        } = parseFileName(item.name);

        const partnerCompany = extractPartnerCompany(item.name, companyCode);

        rows.push({
          levels,
          companyCode,
          partnerCompany,
          fileName: cleanedName,
          date,
          originalFileName: item.name,
          companyOk: hasValidCompany,
          dateOk: hasValidDate,
        });

        if (levels.length > maxLevels) maxLevels = levels.length;
      }
    }
  }

  await walk(rootPath);

  // Build headers dynamically
  const headers = [];
  for (let i = 1; i <= maxLevels; i++) {
    headers.push({ header: `Level ${i}`, key: `level${i}` });
  }
  headers.push(
    { header: "Company Code", key: "companyCode" },
    { header: "Partner Company", key: "partnerCompany" },
    { header: "File Name", key: "fileName" },
    { header: "Date", key: "date" },
    { header: "Original File Name", key: "originalFileName" }
  );

  sheet.columns = headers;

  // Write the rows to the Excel sheet
  rows.forEach(
    ({
      levels,
      companyCode,
      partnerCompany,
      fileName,
      date,
      originalFileName,
      companyOk,
      dateOk,
    }) => {
      const rowObject = {};

      // Add folder levels dynamically
      levels.forEach((value, idx) => {
        rowObject[`level${idx + 1}`] = value;
      });

      rowObject.companyCode = companyCode;
      rowObject.partnerCompany = partnerCompany;
      rowObject.fileName = fileName;
      rowObject.date = date;
      rowObject.originalFileName = originalFileName;

      const excelRow = sheet.addRow(rowObject);

      const companyCell = excelRow.getCell(maxLevels + 1);
      const dateCell = excelRow.getCell(maxLevels + 4);

      if (!companyOk) {
        companyCell.fill = {
          type: "pattern",
          pattern: "solid",
          fgColor: { argb: "FFFFC7CE" },
        };
      }

      if (!dateOk) {
        dateCell.fill = {
          type: "pattern",
          pattern: "solid",
          fgColor: { argb: "FFFFC7CE" },
        };
      }
    }
  );

  await workbook.xlsx.writeFile(outputPath);
}
