import axios from "axios";
import Cookies from "js-cookie";
import { safeToast } from "@/lib/safeToast";
// Create instance
const apiClient = axios.create({
  baseURL: process.env.SERVER_URL,
  withCredentials: true,
});

// Used in server contexts to inject token manually
let injectedServerToken = null;

export const setServerToken = (token) => {
  injectedServerToken = token;
};

let isRefreshing = false;
let refreshSubscribers = [];

const onRefreshed = (newToken) => {
  refreshSubscribers.forEach((callback) => callback(newToken));
};

const addSubscriber = (callback) => {
  refreshSubscribers.push(callback);
};

// Request interceptor
apiClient.interceptors.request.use(
  (config) => {
    const token =
      typeof window !== "undefined"
        ? Cookies.get("edSId")
        : injectedServerToken;

    if (token) {
      config.headers.Authorization = `Bearer ${token}`;
      config.headers["x-sid"] = token;
    }

    return config;
  },
  (err) => Promise.reject(err)
);

// Response interceptor with token refresh logic
apiClient.interceptors.response.use(
  (response) => response,
  async (error) => {
    const originalRequest = error.config;

    if (
      error.response?.status === 401 &&
      ["jwt expired", "Authorization header missing"].includes(
        error.response?.data?.message
      ) &&
      !originalRequest._retry
    ) {
      originalRequest._retry = true;
      safeToast.error("Session expired. Refreshing token...");

      if (isRefreshing) {
        return new Promise((resolve) => {
          addSubscriber((newToken) => {
            originalRequest.headers.Authorization = `Bearer ${newToken}`;
            resolve(apiClient(originalRequest));
          });
        });
      }

      isRefreshing = true;

      try {
        const res = await axios.get(
          `${apiClient.defaults.baseURL}/api/admin/auth/refresh-token`,
          { withCredentials: true }
        );
        const newToken = res.data?.accessToken;

        if (!newToken) {
          safeToast.error("Hiányzó refresh token...");
          throw new Error("No access token in refresh");
        }

        if (typeof window !== "undefined") {
          Cookies.set("edSId", newToken);
        } else {
          injectedServerToken = newToken;
        }

        onRefreshed(newToken);
        refreshSubscribers = [];

        originalRequest.headers.Authorization = `Bearer ${newToken}`;
        return apiClient(originalRequest);
      } catch (refreshError) {
        if (typeof window !== "undefined") {
          Cookies.remove("edSId");
          window.dispatchEvent(new CustomEvent("auth-failed"));
        }
        safeToast.error(
          "A token frissítése közben autentikációs hiba történt..."
        );
        return Promise.reject(refreshError);
      } finally {
        isRefreshing = false;
      }
    }

    // Generic error handler
    const status = error.response?.status;
    const message =
      error.response?.data?.message ||
      error.message ||
      "Ismeretlen hiba történt";

    if (status === 400) {
      safeToast.error(`Hibás kérés: ${message}`);
    } else if (status === 403) {
      safeToast.error("Nincs jogosultságod ehhez a művelethez");
    } else if (status === 500) {
      safeToast.error("Szerverhiba történt. Próbáld újra később.");
    } else {
      safeToast.error(`Hiba (${status}): ${message}`);
    }

    return Promise.reject(error);
  }
);

export default apiClient;
