// i18n.ts
import Cookies from "universal-cookie";

import en from "./public/locales/en.json";
import de from "./public/locales/de.json";
import hu from "./public/locales/hu.json";

const langObj = {
  en,
  de,
  hu,
} as const;

type LangKey = keyof typeof langObj;

function getLangFromCookies(cookieHeader?: string): string | undefined {
  // On the server, pass the cookie header string; in the browser, pass null
  const cookies = new Cookies(
    typeof window === "undefined" ? cookieHeader ?? "" : null,
    { path: "/" }
  );
  return cookies.get("i18nextLng");
}
function getByPath(obj: any, path: string) {
  return path
    .split(".")
    .reduce((acc, k) => (acc && acc[k] != null ? acc[k] : undefined), obj);
}

export function getTranslation(cookieHeader?: string) {
  const lang = (getLangFromCookies(cookieHeader) as LangKey) || "hu";
  const data = langObj[lang] ?? langObj.en;

  const t = (key: string, opts?: { defaultValue?: string }) => {
    const v = getByPath(data, key);
    if (typeof v === "string") return v;
    if (opts?.defaultValue != null) return opts.defaultValue;
    return key;
  };

  const i18n = {
    language: lang as string,
    changeLanguage(nextLang: LangKey) {
      const cookies = new Cookies(null, { path: "/" });
      cookies.set("i18nextLng", nextLang);
      this.language = nextLang;
    },
  };

  const initLocale = (fallback: LangKey) => {
    i18n.changeLanguage(
      (getLangFromCookies(cookieHeader) as LangKey) || fallback
    );
  };

  return { t, i18n, initLocale };
}
// export function getTranslation(cookieHeader?: string) {
//   // cookieHeader is only used on the server
//   const lang = (getLangFromCookies(cookieHeader) as LangKey) || "hu";
//   const data = langObj[lang] ?? langObj.en;

//   const t = (key: string) => (key in data ? (data as any)[key] : key);

//   const i18n = {
//     language: lang as string,
//     // NOTE: Only sets the cookie in the browser. For server-side changes,
//     // set the cookie via your Next.js response (see examples below).
//     changeLanguage(nextLang: LangKey) {
//       const cookies = new Cookies(null, { path: "/" });
//       cookies.set("i18nextLng", nextLang);
//       this.language = nextLang;
//     },
//   };

//   const initLocale = (fallback: LangKey) => {
//     i18n.changeLanguage(
//       (getLangFromCookies(cookieHeader) as LangKey) || fallback
//     );
//   };

//   return { t, i18n, initLocale };
// }
