"use client";

import { useEffect, useMemo, useState, useCallback } from "react";
import orgApi from "@/api/org-structure";
import toast from "react-hot-toast";

export default function useOrgStructure(initial = {}) {
  const [sites, setSites] = useState([]);
  const [departments, setDepartments] = useState([]);
  const [locations, setLocations] = useState([]);

  const [siteId, setSiteId] = useState(initial.siteId || null);
  const [departmentId, setDepartmentId] = useState(
    initial.departmentId || null
  );

  // load sites on mount
  useEffect(() => {
    (async () => {
      try {
        const s = await orgApi.listSites();
        setSites(s.data || []);
      } catch {
        toast.error("Telephelyek betöltése sikertelen");
      }
    })();
  }, []);

  // load departments when site changes
  useEffect(() => {
    if (!siteId) {
      setDepartments([]);
      setDepartmentId(null);
      setLocations([]);
      return;
    }
    (async () => {
      try {
        const d = await orgApi.listDepartmentsBySite(siteId);
        setDepartments(d.data || []);
        // reset department/location if no longer valid
        if (!d.data?.some((x) => x.id === departmentId)) {
          setDepartmentId(null);
          setLocations([]);
        }
      } catch {
        toast.error("Szervezeti egységek betöltése sikertelen");
      }
    })();
  }, [siteId]);

  // load locations when department changes
  useEffect(() => {
    if (!departmentId) {
      setLocations([]);
      return;
    }
    (async () => {
      try {
        const l = await orgApi.listLocationsByDepartment(departmentId);
        setLocations(l.data || []);
      } catch {
        toast.error("Helyek betöltése sikertelen");
      }
    })();
  }, [departmentId]);

  const siteOptions = useMemo(
    () => sites.map((s) => ({ value: s.id, label: s.name })),
    [sites]
  );
  const departmentOptions = useMemo(
    () => departments.map((d) => ({ value: d.id, label: d.name })),
    [departments]
  );
  const locationOptions = useMemo(
    () => locations.map((l) => ({ value: l.id, label: l.name })),
    [locations]
  );

  const resetAll = useCallback(() => {
    setSiteId(null);
    setDepartmentId(null);
    setLocations([]);
  }, []);

  return {
    // raw data
    sites,
    departments,
    locations,
    // selected ids
    siteId,
    setSiteId,
    departmentId,
    setDepartmentId,
    // select-friendly options
    siteOptions,
    departmentOptions,
    locationOptions,
    resetAll,
  };
}
