import { useCallback, useEffect, useRef, useState } from "react";
import useDynamicManager from "@/store/managers/useDynamicManager";

/**
 * This custom hook handles the logic and state for the EditorGallery functionality.
 */
export const useEditorGallery = ({ toggleModal, modalName, type }) => {
  const {
    state,
    cloneGalleryImage,
    selectedElement,
    addcurrentElementToColumn,
    removeSingleGalleryImage,
    updateGalleryImageOrder,
    resetGalleryState,
    selectedGalleryImages,
    setSelectedGalleryImages,
  } = useDynamicManager(type);
  // Local states
  const [images, setImages] = useState(null);
  const [selectImageModal, setSelectImageModal] = useState(false);
  const [editImageModal, setEditImageModal] = useState(false);
  const [editedImage, setEditedImage] = useState({});

  // Refs
  const inputRefs = useRef([]);
  // If needed, to store references for editing
  const editImageRefs = useRef([]);

  // Clean up and re-select images if needed
  useEffect(() => {
    if (state.current.content?.length > 0) {
      setSelectedGalleryImages({
        selectedImages: state.current.content?.[0]?.content?.[0]?.images,
      });
    }
  }, []);

  // Handlers

  const handleEditSingleImage = useCallback((image) => {
    setEditedImage(image);
    setEditImageModal(true);
  }, []);

  const handleRemoveSingleImage = useCallback((image, idx) => {
    removeSingleGalleryImage({
      id: image.image._id,
      index: idx,
    });
  }, []);

  const handleCloneGalleryImage = useCallback((i) => {
    cloneGalleryImage({ selected: i });
  }, []);

  const handleUpdateGalleryImageOrder = useCallback((start, end) => {
    updateGalleryImageOrder({ start, end });
  }, []);

  const handleGallerySubmit = useCallback(() => {
    const content = {
      type: "gallery",
      title: "Galéria",
      images: state.galleryImages,
    };

    Object.entries(inputRefs.current).forEach(([key, ref]) => {
      if (ref) {
        if (ref.props) {
          // Handle cases where props contain value data
          if (Array.isArray(ref.props.value)) {
            // Handle arrays of values
            content[key] = ref.props.value.map((item) => item.value);
          } else {
            // Handle single values
            content[key] = ref.props.value?.value ?? "";
          }
        } else {
          // Handle cases where ref itself holds value data
          content[key] = ref?.value ?? "";
        }
      }
    });

    selectedElement({
      content: content,
    });

    addcurrentElementToColumn();
    // resetGalleryState();
    // resetElementState();

    toggleModal(modalName);
    toggleModal("showSelectElementModal");
  }, [state, toggleModal, modalName]);

  return {
    selectImageModal,
    editImageModal,
    editedImage,

    // Setters
    setSelectImageModal,
    setEditImageModal,
    setEditedImage,

    // Handlers
    handleEditSingleImage,
    handleRemoveSingleImage,
    handleCloneGalleryImage,
    handleUpdateGalleryImageOrder,
    handleGallerySubmit,
  };
};
