import React, { useCallback, useEffect, useState } from "react";
import {
  VALIDATOR_REQUIRE,
  VALIDATOR_MINLENGTH,
  VALIDATOR_MAXLENGTH,
  VALIDATOR_EMAIL,
  VALIDATOR_PASSWORD,
  VALIDATOR_PHONE_HU,
  VALIDATOR_TAXNUMBER_HU,
  VALIDATOR_NUMBER,
} from "@/utils/validators";
import Input from "@/components/shared/input/Input";
import { useForm } from "@/hooks/custom-form-hook";
import Select from "react-select";
import icons from "@/assets/icons/icons";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";

const UserData = ({ user, onUserUpdated }) => {
  const [formState, handleInput] = useForm([], false);
  const [showPassword, setShowPassword] = useState(false);

  const [selectedUserType, setSelectedUserType] = useState(
    user?.registrationType?.value
  );
  const userTypes = [
    { id: 0, label: "Magánszemély", value: "private" },
    { id: 1, label: "Cég", value: "company" },
    { id: 2, label: "Lovarda", value: "lovarda" },
  ];

  const handleUpdateUserData = useCallback(async () => {
    const ret = {
      ...formState.inputs,
      registrationType: { value: selectedUserType },
    };
    const updateUser = await ApiCall.users.postEdit(user._id, ret);
    if (updateUser.status === 200) {
      toast.success("Felhasználói adatok sikeresen frissítve!");
      onUserUpdated?.();
    }
    console.log("ret", ret);
  }, [formState, selectedUserType]);

  return (
    <>
      {user && (
        <div className="w-full grid grid-cols-2 gap-4">
          {/* user type */}
          <Select
            options={userTypes}
            className="w-full col-span-2"
            menuPosition={"fixed"}
            placeholder={"Felhasználó típusának kiválasztása"}
            backspaceRemovesValue={false}
            name={"selectedRight"}
            defaultValue={
              user?.registrationType?.value
                ? userTypes?.find(
                    (t) => t.value === user?.registrationType?.value
                  )
                : []
            }
            onChange={(val, e) => {
              console.log("val.value", val.value);
              // second parameter shows action triggered
              setSelectedUserType(val.value);
            }}
            required
          />
          {/* firstname */}
          <Input
            element="input"
            id="firstname"
            type="text"
            name="firstname"
            placeholder="Vezetéknév"
            label="Vezetéknév"
            inputcls="w-full border-2 text-center py-4 px-2 outline-blue-500"
            className="max-md:w-full w-[49.5%] min-h-[6rem]"
            validators={[
              VALIDATOR_REQUIRE(),
              VALIDATOR_MINLENGTH(2),
              VALIDATOR_MAXLENGTH(254),
            ]}
            errorText={"Nem megfelelő formátum"}
            onInput={handleInput}
            initialValid={true}
            style={{
              fontSize: "0.85rem",
              border: "1px solid rgb(209, 213, 219)",
              color: "#080a2c",
            }}
            labelStyle={{
              fontSize: "0.70rem",
              color: "#080a2c",
              opacity: "0.6",
            }}
            errorStyle={{ fontSize: "0.70rem", padding: "0.25rem" }}
            value={user?.firstname?.value}
            autoComplete="off"
          />
          {/* lastname */}
          <Input
            element="input"
            id="lastname"
            type="text"
            name="lastname"
            placeholder="Keresztnév"
            label="Keresztnév"
            inputcls="w-full border-2 text-center py-4 px-2 outline-blue-500"
            className="max-md:w-full w-[49.5%] min-h-[6rem]"
            validators={[
              VALIDATOR_REQUIRE(),
              VALIDATOR_MINLENGTH(2),
              VALIDATOR_MAXLENGTH(254),
            ]}
            errorText={"Nem megfelelő formátum"}
            onInput={handleInput}
            initialValid={true}
            style={{
              fontSize: "0.85rem",
              border: "1px solid rgb(209, 213, 219)",
              color: "#080a2c",
            }}
            labelStyle={{
              fontSize: "0.70rem",
              color: "#080a2c",
              opacity: "0.6",
            }}
            errorStyle={{ fontSize: "0.70rem", padding: "0.25rem" }}
            value={user?.lastname?.value}
            autoComplete="off"
          />
          {/* telefon */}
          <Input
            element="input"
            id="telefon"
            type="text"
            name="telefon"
            placeholder="Telefonszám"
            label="Telefonszám (06/+36)"
            inputcls="w-full border-2 text-center py-4 px-2 outline-blue-500"
            className="w-full col-span-2 min-h-[6rem]"
            validators={[VALIDATOR_REQUIRE(), VALIDATOR_PHONE_HU()]}
            errorText={"Nem megfelelő formátum"}
            onInput={handleInput}
            initialValid={true}
            style={{
              fontSize: "0.85rem",
              border: "1px solid rgb(209, 213, 219)",
              color: "#080a2c",
            }}
            labelStyle={{
              fontSize: "0.70rem",
              color: "#080a2c",
              opacity: "0.6",
            }}
            errorStyle={{ fontSize: "0.70rem", padding: "0.25rem" }}
            value={user?.telefon?.value}
            autoComplete="off"
          />
          {/* username */}
          <Input
            element="input"
            id="username"
            type="text"
            name="username"
            placeholder="E-mail cím"
            label="E-mail cím"
            inputcls="w-full border-2 text-center py-4 px-2 outline-blue-500"
            className="w-full min-h-[6rem]"
            validators={[VALIDATOR_REQUIRE(), VALIDATOR_EMAIL]}
            errorText={"Nem megfelelő formátum"}
            onInput={handleInput}
            initialValid={true}
            style={{
              fontSize: "0.85rem",
              border: "1px solid rgb(209, 213, 219)",
              color: "#080a2c",
            }}
            labelStyle={{
              fontSize: "0.70rem",
              color: "#080a2c",
              opacity: "0.6",
            }}
            errorStyle={{ fontSize: "0.70rem", padding: "0.25rem" }}
            value={user?.username.value}
            autoComplete="off"
            readOnly
          />
          {/* password */}
          <div className="relative">
            <input
              type="password"
              name="fakepass"
              autoComplete="new-password"
              style={{ display: "none" }}
            />
            <Input
              element="input"
              id="newPassword"
              type={`${showPassword ? "text" : "password"}`}
              name="newPassword"
              placeholder="Új jelszó (minimum 8 karakter, kisbetű, nagybetű, szám)"
              label="Új jelszó (minimum 8 karakter, kisbetű, nagybetű, szám)"
              inputcls="w-full border-2 text-center py-4 px-2 outline-blue-500 "
              className="w-full"
              style={{
                fontSize: "0.75rem",
                border: "1px solid rgb(209, 213, 219)",
              }}
              labelStyle={{ fontSize: "0.65rem", color: "#080a2c" }}
              errorStyle={{ fontSize: "0.65rem", padding: "0.25rem" }}
              validators={[
                VALIDATOR_REQUIRE(),
                VALIDATOR_PASSWORD(),
                VALIDATOR_MINLENGTH(8),
                VALIDATOR_MAXLENGTH(255),
              ]}
              errorText={"Nem lehet üres"}
              onInput={handleInput}
              initialValid={true}
              value={""}
              autoComplete="new-password"
            />
            <span
              onClick={() => setShowPassword(!showPassword)}
              className="w-[28px] h-[28px] absolute top-[2.3rem] right-1 cursor-pointer flex flex-row justify-center items-end fill-[#080a2c]">
              {showPassword ? icons.closedEye : icons.eye}
            </span>
          </div>
          {/* cegnev */}
          <Input
            element="input"
            id="cegnev"
            type="text"
            name="cegnev"
            placeholder="Cégnév"
            label="Cégnév"
            inputcls="w-full border-2 text-center py-4 px-2 outline-blue-500"
            className="max-md:w-full w-[49.5%] min-h-[6rem]"
            validators={[
              VALIDATOR_REQUIRE(),
              VALIDATOR_MINLENGTH(2),
              VALIDATOR_MAXLENGTH(254),
            ]}
            errorText={"Nem megfelelő formátum"}
            onInput={handleInput}
            initialValid={true}
            style={{
              fontSize: "0.85rem",
              border: "1px solid rgb(209, 213, 219)",
              color: "#080a2c",
            }}
            labelStyle={{
              fontSize: "0.70rem",
              color: "#080a2c",
              opacity: "0.6",
            }}
            errorStyle={{ fontSize: "0.70rem", padding: "0.25rem" }}
            value={user?.cegnev?.value}
            autoComplete="off"
          />
          {/* adoszam */}
          <Input
            element="input"
            id="adoszam"
            type="text"
            name="adoszam"
            placeholder="Adószám"
            label="Adószám (xxxxxxxx-x-xx)"
            inputcls="w-full border-2 text-center py-4 px-2 outline-blue-500"
            className="max-md:w-full w-[49.5%] min-h-[6rem]"
            validators={[VALIDATOR_REQUIRE(), VALIDATOR_TAXNUMBER_HU()]}
            errorText={"Nem megfelelő formátum"}
            onInput={handleInput}
            initialValid={true}
            style={{
              fontSize: "0.85rem",
              border: "1px solid rgb(209, 213, 219)",
              color: "#080a2c",
            }}
            labelStyle={{
              fontSize: "0.70rem",
              color: "#080a2c",
              opacity: "0.6",
            }}
            errorStyle={{ fontSize: "0.70rem", padding: "0.25rem" }}
            value={user?.adoszam?.value}
            autoComplete="off"
          />

          {/* iranyitoszam */}
          <Input
            element="input"
            id="iranyitoszam"
            type="text"
            name="iranyitoszam"
            placeholder="Irányítószám"
            label="Irányítószám"
            inputcls="w-full border-2 text-center py-4 px-2 outline-blue-500"
            className="max-md:w-full w-[30%] min-h-[6rem]"
            validators={[
              VALIDATOR_REQUIRE(),
              VALIDATOR_MINLENGTH(4),
              VALIDATOR_MAXLENGTH(4),
              VALIDATOR_NUMBER(),
            ]}
            errorText={"Nem megfelelő formátum"}
            onInput={handleInput}
            initialValid={true}
            style={{
              fontSize: "0.85rem",
              border: "1px solid rgb(209, 213, 219)",
              color: "#080a2c",
            }}
            labelStyle={{
              fontSize: "0.70rem",
              color: "#080a2c",
              opacity: "0.6",
            }}
            errorStyle={{ fontSize: "0.70rem", padding: "0.25rem" }}
            value={user?.iranyitoszam?.value}
            autoComplete="off"
          />
          {/* telepules */}
          <Input
            element="input"
            id="telepules"
            type="text"
            name="telepules"
            placeholder="Település"
            label="Település"
            inputcls="w-full border-2 text-center py-4 px-2 outline-blue-500"
            className="max-md:w-full w-[69%] min-h-[6rem]"
            validators={[
              VALIDATOR_REQUIRE(),
              VALIDATOR_MINLENGTH(2),
              VALIDATOR_MAXLENGTH(254),
            ]}
            errorText={"Nem megfelelő formátum"}
            onInput={handleInput}
            initialValid={true}
            style={{
              fontSize: "0.85rem",
              border: "1px solid rgb(209, 213, 219)",
              color: "#080a2c",
            }}
            labelStyle={{
              fontSize: "0.70rem",
              color: "#080a2c",
              opacity: "0.6",
            }}
            errorStyle={{ fontSize: "0.70rem", padding: "0.25rem" }}
            value={user?.telepules?.value}
            autoComplete="off"
          />
          {/* utca */}
          <Input
            element="input"
            id="utca"
            type="text"
            name="utca"
            placeholder="Utca, házszám"
            label="Utca, házszám"
            inputcls="w-full border-2 text-center py-4 px-2 outline-blue-500"
            className="w-full min-h-[6rem]"
            validators={[
              VALIDATOR_REQUIRE(),
              VALIDATOR_MINLENGTH(2),
              VALIDATOR_MAXLENGTH(254),
            ]}
            errorText={"Nem megfelelő formátum"}
            onInput={handleInput}
            initialValid={true}
            style={{
              fontSize: "0.85rem",
              border: "1px solid rgb(209, 213, 219)",
              color: "#080a2c",
            }}
            labelStyle={{
              fontSize: "0.70rem",
              color: "#080a2c",
              opacity: "0.6",
            }}
            errorStyle={{ fontSize: "0.70rem", padding: "0.25rem" }}
            value={user?.utca?.value}
            autoComplete="off"
          />
        </div>
      )}
      <button
        onClick={() => handleUpdateUserData()}
        className="w-full px-4 py-2 bg-green-500/75 hover:bg-green-500 text-white">
        Számlázási adatok frissítése
      </button>
    </>
  );
};

export default UserData;
