"use client";

import { useEffect, useState } from "react";
import Link from "next/link";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";

type DocForm = {
  id: string;
  code: string;
  name: string;
  description: string | null;
  schema: any;
  is_active: boolean;
  created_at: string;
  updated_at: string;
};

type OnlyActiveFilter = "all" | "yes" | "no";

export function FormTemplateList() {
  const [forms, setForms] = useState<DocForm[]>([]);
  const [loading, setLoading] = useState(true);
  const [deletingId, setDeletingId] = useState<string | null>(null);

  const [search, setSearch] = useState("");
  const [onlyActive, setOnlyActive] = useState<OnlyActiveFilter>("all");

  // small debounce so we don't spam the API on every keystroke
  const [searchInput, setSearchInput] = useState("");

  useEffect(() => {
    const t = setTimeout(() => {
      setSearch(searchInput);
    }, 300);
    return () => clearTimeout(t);
  }, [searchInput]);

  const loadForms = async () => {
    try {
      setLoading(true);
      const res = await ApiCall.templateDocuments.getFormEditorList({
        search: search || "",
        onlyActive,
      });
      setForms(res.data ?? []);
    } catch (e) {
      console.error(e);
      toast.error("Űrlap sablonok betöltése sikertelen");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    loadForms();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [search, onlyActive]);

  const handleDelete = async (id: string) => {
    const form = forms.find((f) => f.id === id);
    const name = form?.name ?? id;

    if (
      !confirm(
        `Biztosan törlöd az alábbi űrlap sablont?\n\n${name}\n\nEz a művelet nem visszavonható.`
      )
    ) {
      return;
    }

    try {
      setDeletingId(id);
      await ApiCall.templateDocuments.removeFormEditor(id);
      setForms((prev) => prev.filter((f) => f.id !== id));
      toast.success("Űrlap sablon törölve");
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ??
          "A törlés közben hiba történt – lehet, hogy az űrlap használatban van."
      );
    } finally {
      setDeletingId(null);
    }
  };

  return (
    <main className="max-w-6xl mx-auto p-4 space-y-4 bg-white">
      <header className="flex flex-wrap gap-3 items-center justify-between">
        <div className="space-y-1">
          <h1 className="text-xl font-semibold">Űrlap sablonok</h1>
          <p className="text-sm text-gray-600">
            Itt kezelheted a menthető / újrahasználható űrlapokat (doc_forms),
            amelyeket a dokumentum sablonokhoz tudsz kapcsolni.
          </p>
        </div>

        <Link
          href="/admin/template-documents/forms/new"
          className="inline-flex items-center px-3 py-2 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white text-sm">
          + Új űrlap sablon
        </Link>
      </header>

      {/* Filters */}
      <section className="border border-gray-200 rounded p-3 flex flex-wrap gap-3 items-end">
        <div className="flex flex-col text-xs text-gray-600 min-w-[220px]">
          <label className="mb-1">Keresés (kód / név)</label>
          <input
            type="text"
            className="border rounded px-2 py-1 text-sm"
            placeholder="pl. it-belepteto vagy IT beléptető"
            value={searchInput}
            onChange={(e) => setSearchInput(e.target.value)}
          />
        </div>

        <div className="flex flex-col text-xs text-gray-600 min-w-[160px]">
          <label className="mb-1">Aktivitás szűrő</label>
          <select
            className="border rounded px-2 py-1 text-sm"
            value={onlyActive}
            onChange={(e) => setOnlyActive(e.target.value as OnlyActiveFilter)}>
            <option value="all">Összes</option>
            <option value="yes">Csak aktív</option>
            <option value="no">Csak inaktív</option>
          </select>
        </div>

        <button
          type="button"
          onClick={loadForms}
          className="ml-auto px-3 py-2 rounded border border-gray-300 text-sm hover:bg-gray-50">
          Frissítés
        </button>
      </section>

      {/* List */}
      <section className="border border-gray-200 rounded">
        {loading ? (
          <p className="p-4 text-sm text-gray-600">Betöltés…</p>
        ) : !forms.length ? (
          <p className="p-4 text-sm text-gray-600">
            Nincs találat a megadott szűrőkkel. Hozz létre egy új űrlap sablont
            a jobb felső sarokban.
          </p>
        ) : (
          <div className="overflow-x-auto">
            <table className="min-w-full text-sm">
              <thead className="bg-gray-50 border-b border-gray-200">
                <tr>
                  <th className="text-left px-3 py-2 font-semibold">Név</th>
                  <th className="text-left px-3 py-2 font-semibold">Kód</th>
                  <th className="text-left px-3 py-2 font-semibold">
                    Mezők száma
                  </th>
                  <th className="text-left px-3 py-2 font-semibold">Aktív</th>
                  <th className="text-left px-3 py-2 font-semibold">
                    Módosítva
                  </th>
                  <th className="text-right px-3 py-2 font-semibold">
                    Műveletek
                  </th>
                </tr>
              </thead>
              <tbody>
                {forms.map((f) => {
                  const fieldsCount = Array.isArray(f.schema)
                    ? f.schema.length
                    : (() => {
                        try {
                          const parsed = JSON.parse((f.schema as any) ?? "[]");
                          return Array.isArray(parsed) ? parsed.length : 0;
                        } catch {
                          return 0;
                        }
                      })();

                  const updated = f.updated_at ?? f.created_at ?? "";

                  const updatedShort = updated
                    ? new Date(updated).toLocaleString("hu-HU", {
                        dateStyle: "short",
                        timeStyle: "short",
                      })
                    : "";

                  return (
                    <tr
                      key={f.id}
                      className="border-b border-gray-100 hover:bg-gray-50">
                      <td className="px-3 py-2">
                        <div className="flex flex-col">
                          <span className="font-medium">{f.name}</span>
                          {f.description && (
                            <span className="text-[11px] text-gray-500">
                              {f.description}
                            </span>
                          )}
                        </div>
                      </td>
                      <td className="px-3 py-2 text-xs text-gray-700">
                        <code className="bg-gray-100 px-1 py-0.5 rounded">
                          {f.code}
                        </code>
                      </td>
                      <td className="px-3 py-2 text-xs text-gray-700">
                        {fieldsCount}
                      </td>
                      <td className="px-3 py-2 text-xs">
                        {f.is_active ? (
                          <span className="inline-flex items-center px-2 py-0.5 rounded-full bg-emerald-50 text-emerald-700 border border-emerald-200">
                            Aktív
                          </span>
                        ) : (
                          <span className="inline-flex items-center px-2 py-0.5 rounded-full bg-gray-50 text-gray-600 border border-gray-200">
                            Inaktív
                          </span>
                        )}
                      </td>
                      <td className="px-3 py-2 text-xs text-gray-600">
                        {updatedShort}
                      </td>
                      <td className="px-3 py-2 text-xs">
                        <div className="flex justify-end gap-2">
                          <Link
                            href={`/admin/template-documents/forms/${f.id}`}
                            className="px-2 py-1 border rounded hover:bg-gray-50">
                            Szerkesztés
                          </Link>
                          <button
                            type="button"
                            onClick={() => handleDelete(f.id)}
                            disabled={deletingId === f.id}
                            className="px-2 py-1 border border-red-300 text-red-700 rounded hover:bg-red-50 disabled:opacity-60">
                            {deletingId === f.id ? "Törlés…" : "Törlés"}
                          </button>
                        </div>
                      </td>
                    </tr>
                  );
                })}
              </tbody>
            </table>
          </div>
        )}
      </section>
    </main>
  );
}
