import React, { useEffect, useReducer, useRef } from "react";
import { validate } from "@/utils/validators";

const inputReducer = (state, action) => {
  switch (action.type) {
    case "INIT":
      return {
        ...state,
        value: action.val,
        isChecked: !!action.isChecked,
        isValid: action.hasValidators
          ? validate(action.val, action.validators)
          : state.isValid ?? true,
      };
    case "CHANGE": {
      const hasValidators =
        Array.isArray(action.validators) && action.validators.length > 0;
      return {
        ...state,
        value: action.val,
        isChecked: !!action.isChecked,
        isValid: hasValidators ? validate(action.val, action.validators) : true,
        isTouched: true,
      };
    }
    default:
      return state;
  }
};

const Input = (props) => {
  const hasValidators =
    Array.isArray(props.validators) && props.validators.length > 0;
  const isControlled = props.value !== undefined; // controlled ONLY if value is provided

  const [inputState, dispatch] = useReducer(inputReducer, {
    id: props.id,
    value: isControlled ? props.value ?? "" : props.initialValue ?? "",
    isChecked: isControlled ? !!props.isChecked : !!props.initialChecked,
    isValid: hasValidators
      ? validate(
          isControlled ? props.value ?? "" : props.initialValue ?? "",
          props.validators
        )
      : props.initialValid ?? true,
    isTouched: false,
  });

  const prevInitRef = useRef(props.initialValue);

  // Notify parent on change
  useEffect(() => {
    // if (typeof props.onInput === "function") {
    //   props.onInput(
    //     props.id,
    //     inputState.value,
    //     inputState.isValid,
    //     inputState.isChecked
    //   );
    // }
    const cb = props.onInput || props.onChange;
    if (typeof cb === "function") {
      cb(props.id, inputState.value, inputState.isValid, inputState.isChecked);
    }
  }, [
    props.id,
    inputState.value,
    inputState.isValid,
    inputState.isChecked,
    props.onInput,
  ]);

  // Re-init ONLY if initialValue actually changes (after async load, etc.)
  useEffect(() => {
    if (!isControlled && prevInitRef.current !== props.initialValue) {
      prevInitRef.current = props.initialValue;
      dispatch({
        type: "INIT",
        val: props.initialValue ?? "",
        isChecked: !!props.initialChecked,
        hasValidators,
        validators: props.validators,
      });
    }
  }, [
    isControlled,
    props.initialValue,
    props.initialChecked,
    hasValidators,
    props.validators,
  ]);

  const changeHandler = (event) => {
    const nextVal = event.target.value;
    const nextChecked =
      event.target.type === "checkbox"
        ? event.target.checked
        : inputState.isChecked;

    // Update local state (still dispatch even if controlled so onInput runs via effect)
    dispatch({
      type: "CHANGE",
      val: nextVal,
      isChecked: nextChecked,
      validators: props.validators,
    });

    // Special VAT calc kept
    if (
      event.target.id === "productVat" ||
      event.target.id === "productPieceNettoBuyPrice"
    ) {
      const nettEl = document.querySelector(
        "[name='productPieceNettoBuyPrice']"
      );
      const vatEl = document.querySelector("[name='productVat']");
      const bruttoEl = document.querySelector(
        "[name='productPieceBruttoBuyPrice']"
      );
      if (nettEl && vatEl && bruttoEl) {
        const nett = parseFloat(nettEl.value || "0");
        const vat = parseFloat(vatEl.value || "27");
        const brutto = parseFloat((nett + (nett / 100) * vat).toFixed(2));
        bruttoEl.value = Number.isFinite(brutto) ? brutto.toFixed(2) : "0.00";
      }
    }
  };

  const invalidClass =
    !inputState.isValid && inputState.isTouched ? "form-control--invalid" : "";

  const commonInputClasses = `bg-white cursor-pointer border border-gray-200 h-max p-2 ${
    props.inputcls || ""
  } ${!inputState.isValid && inputState.isTouched ? "error" : ""} ${
    props.readOnly
      ? "!bg-gray-600 outline-transparent text-white !cursor-not-allowed"
      : "outline-blue-300"
  }`;

  // slimmer classes for checkbox
  const checkboxClasses = `cursor-pointer ${props.inputcls || ""}`;

  // Controlled vs uncontrolled wiring
  const valueProp = isControlled
    ? { value: props.value ?? "" }
    : { value: inputState.value };
  const checkedProp = isControlled
    ? { checked: !!props.isChecked }
    : { checked: !!inputState.isChecked };

  return (
    <div
      className={`form-group flex flex-col my-2 w-full ${
        props.className || ""
      } ${invalidClass}`}>
      <label
        htmlFor={props.id}
        className={`${
          props.labelcls || ""
        } font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2`}>
        {props.label ?? props.placeholder ?? ""}
      </label>

      {/* input (text/number/etc.) + checkbox via type */}
      {props.element === "input" && (
        <input
          id={props.id}
          type={props.type}
          name={props.name}
          className={
            props.type === "checkbox" ? checkboxClasses : commonInputClasses
          }
          placeholder={props.placeholder}
          onChange={changeHandler}
          onKeyPress={props.onKeyPress}
          {...(props.type === "checkbox" ? checkedProp : valueProp)}
          style={props.style}
          readOnly={!!props.readOnly}
          autoComplete={props.autoComplete ?? "on"}
        />
      )}

      {/* explicit checkbox variant so <Input element="checkbox" .../> works */}
      {props.element === "checkbox" && (
        <input
          id={props.id}
          type="checkbox"
          name={props.name}
          className={checkboxClasses}
          onChange={changeHandler}
          {...checkedProp}
          readOnly={!!props.readOnly}
        />
      )}

      {props.element === "select" && (
        <select
          id={props.id}
          name={props.name}
          className={`bg-white cursor-pointer ${props.inputcls || ""} ${
            !inputState.isValid && inputState.isTouched ? "error" : ""
          } ${props.isOpen ? "active" : ""}`}
          onChange={changeHandler}
          {...(isControlled
            ? { value: props.value ?? "" }
            : { value: inputState.value || "" })}>
          <option value=""></option>
          {props.options?.map((o, i) => {
            const val =
              typeof o === "string"
                ? o
                : o?.value ?? o?.[props.optionProperty]?.value ?? "";
            const label =
              typeof o === "string"
                ? o
                : o?.label ?? o?.[props.optionProperty]?.value ?? String(val);
            return (
              <option key={i} value={val}>
                {label}
              </option>
            );
          })}
        </select>
      )}

      {props.element === "textarea" && (
        <textarea
          id={props.id}
          className={`bg-white cursor-pointer ${
            props.inputcls || ""
          } !text-left outline-blue-300`}
          rows={props.rows || 3}
          onChange={changeHandler}
          {...valueProp}
          readOnly={!!props.readOnly}
        />
      )}

      {!inputState.isValid && inputState.isTouched && (
        <div className="message__wrapper py-1">
          <p className="message--error text-xs text-red-500 font-bold">
            {props.placeholder
              ? `${props.placeholder} - ${props.errorText}`
              : props.errorText}
          </p>
        </div>
      )}
    </div>
  );
};

export default Input;
