// components/shared/iconPicker/UniversalIconPicker.tsx
"use client";
import React, { useState, useMemo, useEffect } from "react";
import LazyIcon from "./LazyIcon";
import { Input } from "@mui/material";
import InfiniteScroll from "react-infinite-scroll-component";

const ITEMS_PER_PAGE = 50;

const UniversalIconPicker = ({ value, onSelect }) => {
  const [search, setSearch] = useState("");
  const [iconList, setIconList] = useState([]);
  const [visibleIcons, setVisibleIcons] = useState([]);
  const [hasMore, setHasMore] = useState(true);
  const [filteredIcons, setFilteredIcons] = useState([]);

  // Fetch icon list
  useEffect(() => {
    fetch("/icon-list.json")
      .then((res) => res.json())
      .then((data) => {
        setIconList(data);
        setFilteredIcons(data); // set initial filtered
        setVisibleIcons(data.slice(0, ITEMS_PER_PAGE));
      });
  }, []);

  // Filter icons when search changes
  useEffect(() => {
    const filtered = iconList.filter((icon) =>
      icon.name.toLowerCase().includes(search.toLowerCase())
    );
    setFilteredIcons(filtered);
    setVisibleIcons(filtered.slice(0, ITEMS_PER_PAGE));
    setHasMore(filtered.length > ITEMS_PER_PAGE);
  }, [search, iconList]);

  const loadMore = () => {
    const next = filteredIcons.slice(
      visibleIcons.length,
      visibleIcons.length + ITEMS_PER_PAGE
    );
    setVisibleIcons((prev) => [...prev, ...next]);
    if (visibleIcons.length + next.length >= filteredIcons.length) {
      setHasMore(false);
    }
  };

  return (
    <div className="w-full">
      <Input
        type="text"
        placeholder="Keresés ikon név szerint..."
        value={search}
        onChange={(e) => setSearch(e.target.value)}
        className="w-full mb-4 border p-2"
      />
      <div className="max-h-[300px] overflow-y-auto">
        <InfiniteScroll
          dataLength={visibleIcons.length}
          next={loadMore}
          hasMore={hasMore}
          loader={<p className="text-sm text-center py-2">Töltés...</p>}
          endMessage={
            <p className="text-center text-xs py-2 text-gray-400">
              Nincs több ikon.
            </p>
          }
          scrollableTarget={null} // you can skip this unless using a specific scroll container
        >
          <div className="grid grid-cols-6 md:grid-cols-8 lg:grid-cols-10 gap-4">
            {visibleIcons.map((icon) => (
              <button
                key={icon.name}
                title={icon.name}
                onClick={() => onSelect(icon.name)}
                className={`p-2 rounded border transition ${
                  value === icon.name
                    ? "border-blue-500 bg-blue-50"
                    : "border-transparent hover:border-gray-300"
                }`}>
                <LazyIcon lib={icon.lib} icon={icon.name} className="text-xl" />
              </button>
            ))}
          </div>
        </InfiniteScroll>
      </div>
    </div>
  );
};

export default UniversalIconPicker;
