// src/components/shared/data-table/DataTableWrapper.tsx
"use client";

import React, { useEffect, useMemo, useRef, useState } from "react";
import DataTable, { TableProps } from "react-data-table-component";
import Link from "next/link";

/** Only wrapper-specific props */
type ExtraProps<T> = {
  tableTitle?: React.ReactNode;
  tableOptionsPath?: string;
  onFilterClear?: () => void;
  onRowSelected?: (rows: T[]) => void;
  toggleCleared?: boolean;
  onSearch?: (q: string) => void; // API compatibility only
  renderExpandableRows?: (row: T) => React.ReactNode;
  /** NEW: allow caller to control current page */
  paginationCurrentPage?: number;
  /** NEW: allow caller to define row id field */
  keyField?: string;
};

/** All RDT props optional (except data/columns). */
type Props<T = any> = ExtraProps<T> &
  Omit<Partial<TableProps<T>>, "data" | "columns" | "subHeaderAlign"> & {
    data: T[];
    columns: any[];
    /** Accept both legacy "left/right" and RDT "start/end/center" */
    subHeaderAlign?: TableProps<T>["subHeaderAlign"] | "left" | "right";
  };

function normalizeAlign<T>(
  v: Props<T>["subHeaderAlign"]
): NonNullable<TableProps<T>["subHeaderAlign"]> {
  const map: Record<string, "start" | "end" | "center"> = {
    left: "start",
    right: "end",
    start: "start",
    end: "end",
    center: "center",
  };
  const k = typeof v === "string" ? v.toLowerCase() : "";
  return (map[k] || "end") as NonNullable<TableProps<T>["subHeaderAlign"]>;
}

export default function DataTableWrapper<T = any>(props: Props<T>) {
  const {
    data,
    columns,

    // optional RDT props with defaults
    customStyles = {},
    progressPending = false,
    progressComponent = null,
    paginationDefaultPage = 1,
    paginationServer = false,
    paginationTotalRows = 0,
    paginationPerPage = 25,
    onChangePage,
    onChangeRowsPerPage,
    onSort,
    sortServer = true,

    // selection
    selectableRows = false,
    onRowSelected,
    toggleCleared = false,

    // expandable
    expandableRows = false,
    expandableRowExpanded = () => false,
    onRowExpandToggled,
    renderExpandableRows,

    // subheader / filters
    subHeader = false,
    subHeaderComponent,
    subHeaderAlign,
    subHeaderWrap = true,
    onFilterClear,

    // wrapper-only
    tableTitle,
    tableOptionsPath,
    defaultSortFieldId,

    // NEW controlled page + key field
    paginationCurrentPage,
    keyField = "id",

    // compatibility (unused)
    onSearch,
    ...rest
  } = props;

  const normalizedAlign = normalizeAlign<T>(subHeaderAlign);

  /** Mirror the controlled page into a local state that only changes
   * when the prop actually changes. This prevents random resets. */
  const [effectivePage, setEffectivePage] = useState<number>(
    paginationCurrentPage ?? paginationDefaultPage ?? 1
  );

  const prevControlledPage = useRef<number | undefined>(paginationCurrentPage);
  useEffect(() => {
    if (
      typeof paginationCurrentPage === "number" &&
      paginationCurrentPage !== prevControlledPage.current
    ) {
      prevControlledPage.current = paginationCurrentPage;
      setEffectivePage(paginationCurrentPage);
    }
  }, [paginationCurrentPage]);

  /** Force a *targeted* remount only when page or per-page change intentionally.
   * This makes RDT apply paginationDefaultPage & paginationPerPage reliably. */
  const tableKey = useMemo(
    () => `rdt-${effectivePage}-${paginationPerPage}`,
    [effectivePage, paginationPerPage]
  );

  const handleRowSelected: TableProps<T>["onSelectedRowsChange"] = (state) => {
    onRowSelected?.(state.selectedRows as T[]);
  };

  return (
    <div
      className="w-full relative data-table-wrapper p-4"
      style={{ overflow: "auto hidden", width: "calc(100% - 2vw)" }}>
      <div className="flex items-center justify-between mb-3">
        {tableTitle ? (
          <h2 className="text-xl font-bold">{tableTitle}</h2>
        ) : (
          <span />
        )}
        {tableOptionsPath && (
          <Link
            href={`/admin/${tableOptionsPath}`}
            className="rounded bg-green-600 px-4 py-2 text-white hover:bg-green-700">
            Új hozzáadása
          </Link>
        )}
      </div>

      <DataTable
        key={tableKey}
        keyField={keyField}
        data={data}
        columns={columns}
        className="table-auto"
        /* selection */
        selectableRows={selectableRows}
        onSelectedRowsChange={handleRowSelected}
        clearSelectedRows={toggleCleared}
        /* look & feel */
        customStyles={customStyles}
        striped
        highlightOnHover
        /* progress */
        progressPending={progressPending}
        progressComponent={progressComponent}
        /* pagination */
        pagination
        paginationServer={!!paginationServer}
        paginationDefaultPage={effectivePage}
        paginationTotalRows={paginationTotalRows}
        paginationPerPage={paginationPerPage}
        onChangePage={onChangePage}
        onChangeRowsPerPage={onChangeRowsPerPage}
        paginationRowsPerPageOptions={[10, 25, 50, 100, 500, 1000, 10000]}
        /* sorting (RDT has no controlled sort props) */
        onSort={onSort as any}
        sortServer={sortServer}
        defaultSortFieldId={defaultSortFieldId}
        /* filters/subheader */
        subHeader={subHeader}
        subHeaderComponent={subHeaderComponent}
        subHeaderAlign={normalizedAlign}
        subHeaderWrap={subHeaderWrap}
        {...(onFilterClear ? ({ onClear: onFilterClear } as any) : {})}
        /* expandable rows */
        expandableRows={expandableRows}
        expandableRowExpanded={expandableRowExpanded as any}
        onRowExpandToggled={onRowExpandToggled as any}
        expandableRowsComponent={({ data }) => (
          <>{renderExpandableRows?.(data as T)}</>
        )}
        {...(rest as any)}
      />
    </div>
  );
}

// "use client";

// import React from "react";
// import DataTable, { TableProps } from "react-data-table-component";
// import Link from "next/link";

// /** Only wrapper-specific props */
// type ExtraProps<T> = {
//   tableTitle?: React.ReactNode;
//   tableOptionsPath?: string;
//   onFilterClear?: () => void;
//   onRowSelected?: (rows: T[]) => void;
//   toggleCleared?: boolean;
//   onSearch?: (q: string) => void; // API compatibility only
//   renderExpandableRows?: (row: T) => React.ReactNode;
// };

// /** All RDT props optional (except data/columns). IMPORTANT: omit subHeaderAlign to override it. */
// type Props<T = any> = ExtraProps<T> &
//   Omit<Partial<TableProps<T>>, "data" | "columns" | "subHeaderAlign"> & {
//     data: T[];
//     columns: any[];
//     /** Accept both legacy "left/right" and RDT "start/end/center" */
//     subHeaderAlign?: TableProps<T>["subHeaderAlign"] | "left" | "right";
//   };

// /** Normalize subHeaderAlign across RDT versions */
// function normalizeAlign<T>(
//   v: Props<T>["subHeaderAlign"]
// ): NonNullable<TableProps<T>["subHeaderAlign"]> {
//   const map: Record<string, "start" | "end" | "center"> = {
//     left: "start",
//     right: "end",
//     start: "start",
//     end: "end",
//     center: "center",
//   };
//   const k = typeof v === "string" ? v.toLowerCase() : "";
//   return (map[k] || "end") as NonNullable<TableProps<T>["subHeaderAlign"]>;
// }

// export default function DataTableWrapper<T = any>(props: Props<T>) {
//   const {
//     data,
//     columns,

//     // optional RDT props with defaults
//     customStyles = {},
//     progressPending = false,
//     progressComponent = null,
//     paginationDefaultPage = 1,
//     paginationServer = false,
//     paginationTotalRows = 0,
//     paginationPerPage = 25,
//     onChangePage,
//     onChangeRowsPerPage,
//     onSort,
//     sortServer = true,

//     // selection
//     selectableRows = false,
//     onRowSelected,
//     toggleCleared = false,

//     // expandable
//     expandableRows = false,
//     expandableRowExpanded = () => false,
//     onRowExpandToggled,
//     renderExpandableRows,

//     // subheader / filters
//     subHeader = false,
//     subHeaderComponent,
//     subHeaderAlign, // accept any of: left/right/start/end/center
//     subHeaderWrap = true,
//     onFilterClear,

//     // wrapper-only
//     tableTitle,
//     tableOptionsPath,
//     defaultSortFieldId,

//     // compatibility (unused)
//     onSearch,
//     ...rest
//   } = props;

//   const handleRowSelected: TableProps<T>["onSelectedRowsChange"] = (state) => {
//     onRowSelected?.(state.selectedRows as T[]);
//   };

//   const normalizedAlign = normalizeAlign<T>(subHeaderAlign);

//   return (
//     <div
//       className="w-full relative data-table-wrapper p-4"
//       style={{ overflow: "auto hidden" }}>
//       <div className="flex items-center justify-between mb-3">
//         {tableTitle ? (
//           <h2 className="text-xl font-bold">{tableTitle}</h2>
//         ) : (
//           <span />
//         )}
//         {tableOptionsPath && (
//           <Link
//             href={`/admin/${tableOptionsPath}`}
//             className="rounded bg-green-600 px-4 py-2 text-white hover:bg-green-700">
//             Új hozzáadása
//           </Link>
//         )}
//       </div>

//       <DataTable
//         keyField="_id"
//         data={data}
//         columns={columns}
//         className="table-auto"
//         /* selection */
//         selectableRows={selectableRows}
//         onSelectedRowsChange={handleRowSelected}
//         clearSelectedRows={toggleCleared}
//         /* look & feel */
//         customStyles={customStyles}
//         striped
//         highlightOnHover
//         /* progress */
//         progressPending={progressPending}
//         progressComponent={progressComponent}
//         /* pagination */
//         pagination
//         paginationServer={!!paginationServer}
//         paginationDefaultPage={paginationDefaultPage}
//         paginationTotalRows={paginationTotalRows}
//         paginationPerPage={paginationPerPage}
//         onChangePage={onChangePage}
//         onChangeRowsPerPage={onChangeRowsPerPage}
//         paginationRowsPerPageOptions={[10, 25, 50, 100, 500, 1000, 10000]}
//         /* sorting */
//         onSort={onSort as any}
//         sortServer={sortServer}
//         defaultSortFieldId={defaultSortFieldId}
//         /* filters/subheader */
//         subHeader={subHeader}
//         subHeaderComponent={subHeaderComponent}
//         subHeaderAlign={normalizedAlign}
//         subHeaderWrap={subHeaderWrap}
//         {...(onFilterClear ? ({ onClear: onFilterClear } as any) : {})}
//         /* expandable rows */
//         expandableRows={expandableRows}
//         expandableRowExpanded={expandableRowExpanded as any}
//         onRowExpandToggled={onRowExpandToggled as any}
//         expandableRowsComponent={({ data }) => (
//           <>{renderExpandableRows?.(data as T)}</>
//         )}
//         /* forward the rest */
//         {...(rest as any)}
//       />
//     </div>
//   );
// }
