// components/shared/Button.tsx
"use client";

import React from "react";
import Link from "next/link";

// small helper to join classes
const cn = (...xs: Array<string | false | null | undefined>) =>
  xs.filter(Boolean).join(" ");

type ButtonProps = {
  href?: string;
  onClick?: (e: React.MouseEvent) => void;
  type?: "button" | "submit" | "reset";
  variant?:
    | "primary"
    | "secondary"
    | "success"
    | "danger"
    | "warning"
    | "info"
    | "outline"
    | "ghost"
    | "link";
  size?: "sm" | "md" | "lg";
  fullWidth?: boolean;
  rounded?: "sm" | "md" | "lg" | "xl" | "full";
  className?: string;

  // icons
  leftIcon?: React.ReactNode;
  rightIcon?: React.ReactNode;
  icononly?: boolean;

  // state
  disabled?: boolean;
  loading?: boolean;

  // accessibility when icononly
  "aria-label"?: string;

  // children (preferred) + legacy props compatibility
  children?: React.ReactNode;
  btntext?: React.ReactNode; // legacy
  spinnerText?: React.ReactNode; // legacy
  spinning?: boolean; // legacy alias for loading
  icon?: React.ReactNode; // legacy single icon
  iconclass?: string; // legacy icon class
  target?: string; // for external links
  rel?: string;
};

const base =
  "inline-flex items-center justify-center gap-2 font-medium transition select-none focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-offset-2 disabled:opacity-50";

const roundedMap = {
  sm: "rounded",
  md: "rounded-md",
  lg: "rounded-lg",
  xl: "rounded-xl",
  full: "rounded-full",
};

const sizeMap = {
  sm: "text-sm px-3 py-1.5",
  md: "text-sm px-4 py-2",
  lg: "text-base px-5 py-2.5",
};

const iconOnlySizeMap = {
  sm: "p-2",
  md: "p-2.5",
  lg: "p-3",
};

const variantMap: Record<ButtonProps["variant"], string> = {
  primary:
    "bg-blue-600 text-white hover:bg-blue-700 focus-visible:ring-blue-600",
  secondary:
    "bg-gray-100 text-gray-900 hover:bg-gray-200 focus-visible:ring-gray-400",
  success:
    "bg-emerald-600 text-white hover:bg-emerald-700 focus-visible:ring-emerald-600",
  danger:
    "bg-rose-600 text-white hover:bg-rose-700 focus-visible:ring-rose-600",
  warning:
    "bg-amber-500 text-black hover:bg-amber-600 focus-visible:ring-amber-500",
  info: "bg-cyan-600 text-white hover:bg-cyan-700 focus-visible:ring-cyan-600",
  outline:
    "border border-gray-300 text-gray-900 hover:bg-gray-50 focus-visible:ring-gray-400",
  ghost:
    "bg-transparent text-gray-900 hover:bg-gray-100 focus-visible:ring-gray-300",
  link: "bg-transparent text-blue-600 hover:underline focus-visible:ring-blue-600",
};

const Spinner = () => (
  <svg className="animate-spin h-4 w-4" viewBox="0 0 24 24" aria-hidden="true">
    <circle
      className="opacity-25"
      cx="12"
      cy="12"
      r="10"
      stroke="currentColor"
      strokeWidth="4"
      fill="none"
    />
    <path
      className="opacity-75"
      fill="currentColor"
      d="M4 12a8 8 0 018-8v4a4 4 0 00-4 4H4z"
    />
  </svg>
);

const Button: React.FC<ButtonProps> = (props) => {
  const {
    href,
    onClick,
    type = "button",
    variant = "primary",
    size = "md",
    fullWidth,
    rounded = "md",
    className,

    leftIcon,
    rightIcon,
    icononly,

    disabled,
    loading,

    children,
    btntext,
    spinnerText,
    spinning,
    icon,
    iconclass,
    target,
    rel,
    ...rest
  } = props;

  // legacy aliases
  const isLoading = loading ?? spinning ?? false;
  const content = children ?? btntext ?? "";

  // auto place legacy `icon` on the right if no left/right provided
  const left = leftIcon ?? null;
  const right = rightIcon ?? (!leftIcon && icon ? icon : null);

  const sizing = icononly ? iconOnlySizeMap[size] : sizeMap[size] ?? sizeMap.md;

  const classes = cn(
    base,
    roundedMap[rounded],
    variantMap[variant],
    sizing,
    fullWidth && "w-full",
    isLoading && "cursor-wait",
    className
  );

  const iconWrap = (node: React.ReactNode) =>
    node ? (
      <span className={iconclass ? iconclass : "flex items-center"}>
        {node}
      </span>
    ) : null;

  const Label = () => (
    <>
      {isLoading && <Spinner />}
      <span className={cn(isLoading && "sr-only")}>
        {isLoading ? spinnerText ?? "Loading…" : content}
      </span>
    </>
  );

  const accessibleText =
    (typeof content === "string" && content) ||
    (props["aria-label"] as string) ||
    "Button";

  // LINK MODE
  if (href) {
    // emulate disabled for links
    const linkDisabled = disabled || isLoading;
    const handleLinkClick = (e: React.MouseEvent) => {
      if (linkDisabled) {
        e.preventDefault();
        e.stopPropagation();
        return;
      }
      onClick?.(e);
    };

    return (
      <Link
        href={href}
        onClick={handleLinkClick}
        className={`cursor-pointer ${classes}`}
        aria-disabled={linkDisabled || undefined}
        aria-busy={isLoading || undefined}
        role="button"
        target={target}
        rel={rel}
        {...rest}>
        {icononly ? (
          <>
            {isLoading ? <Spinner /> : iconWrap(left ?? right ?? icon)}
            <span className="sr-only">{accessibleText}</span>
          </>
        ) : (
          <>
            {left && iconWrap(left)}
            <Label />
            {right && iconWrap(right)}
          </>
        )}
      </Link>
    );
  }

  // BUTTON MODE
  return (
    <button
      type={type}
      onClick={onClick}
      className={`cursor-pointer ${classes}`}
      disabled={disabled || isLoading}
      aria-busy={isLoading || undefined}
      {...rest}>
      {icononly ? (
        <>
          {isLoading ? <Spinner /> : iconWrap(left ?? right ?? icon)}
          <span className="sr-only">{accessibleText}</span>
        </>
      ) : (
        <>
          {left && iconWrap(left)}
          <Label />
          {right && iconWrap(right)}
        </>
      )}
    </button>
  );
};

export default Button;

// import React from "react";
// import Link from "next/link";

// const Button = (props) => {
//   if (props.href) {
//     return (
//       <Link
//         className={`button button--${props.size || "default"} ${
//           props.className || ""
//         }${props.disabled ? "bg-gray-500" : ""}`}
//         href={props.href || ""}
//         spinning={props.spinning}
//         disabled={props.disabled ? 1 : 0}
//         text={props.btntext}
//         onClick={props.onClick}>
//         {props.spinning ? props.spinnerText : props.btntext}
//         {props.icon ? props.icon : ""}
//       </Link>
//     );
//   }

//   return (
//     <button {...props}>
//       {props.spinning ? props.spinnerText : props.btntext}
//       {props.icon ? (
//         <span className={props.iconclass ? props.iconclass : "flex fill-white"}>
//           {props.icon}
//         </span>
//       ) : (
//         ""
//       )}
//     </button>
//   );
// };

// export default Button;

// import React from "react";
// import Link from "next/link";

// const Button = (props) => {
//   if (props.href) {
//     return (
//       <Link
//         className={`button button--${props.size || "default"} ${
//           props.className || ""
//         }${props.disabled ? "bg-gray-500" : ""}`}
//         href={props.href || ""}
//         spinning={props.spinning}
//         disabled={props.disabled ? 1 : 0}
//         text={props.btntext}
//         onClick={props.onClick}>
//         {props.spinning ? props.spinnerText : props.btntext}
//         {props.icon ? props.icon : ""}
//       </Link>
//     );
//   }

//   return (
//     <button {...props}>
//       {props.spinning ? props.spinnerText : props.btntext}
//       {props.icon ? (
//         <span className={props.iconclass ? props.iconclass : "flex fill-white"}>
//           {props.icon}
//         </span>
//       ) : (
//         ""
//       )}
//     </button>
//   );
// };

// export default Button;
