"use client";
import { useRef, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";

export default function UploadVersionModal({
  documentId,
  onClose,
  onUploaded,
}: {
  documentId: string;
  onClose: () => void;
  onUploaded: () => void;
}) {
  const fileRef = useRef<HTMLInputElement | null>(null);
  const [changeNote, setChangeNote] = useState("");
  const [status, setStatus] = useState("published");
  const [effectiveFrom, setEffectiveFrom] = useState<string>("");
  const [expiresOn, setExpiresOn] = useState<string>("");
  const [busy, setBusy] = useState(false);

  const submit = async () => {
    const f = fileRef.current?.files?.[0];
    if (!f) return toast.error("Válassz fájlt");
    setBusy(true);
    try {
      const form = new FormData();
      form.append("file", f);
      form.append("change_note", changeNote || "");
      form.append("status", status);
      if (effectiveFrom) form.append("effective_from", effectiveFrom);
      if (expiresOn) form.append("expires_on", expiresOn);

      await ApiCall.qms.uploadVersion(documentId, form);
      toast.success("Verzió feltöltve");
      onUploaded();
      onClose();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Feltöltés sikertelen");
    } finally {
      setBusy(false);
    }
  };

  return (
    <div className="p-4 space-y-3">
      <div>
        <label className="text-xs text-gray-600">Fájl</label>
        <input
          ref={fileRef}
          type="file"
          className="w-full border rounded px-3 py-2"
        />
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-2">
        <div>
          <label className="text-xs text-gray-600">Státusz</label>
          <select
            className="w-full border rounded px-3 py-2"
            value={status}
            onChange={(e) => setStatus(e.target.value)}>
            <option value="published">published</option>
            <option value="draft">draft</option>
            <option value="archived">archived</option>
          </select>
        </div>
        <div>
          <label className="text-xs text-gray-600">Hatályos ettől</label>
          <input
            type="date"
            className="w-full border rounded px-3 py-2"
            value={effectiveFrom}
            onChange={(e) => setEffectiveFrom(e.target.value)}
          />
        </div>
        <div>
          <label className="text-xs text-gray-600">Lejár</label>
          <input
            type="date"
            className="w-full border rounded px-3 py-2"
            value={expiresOn}
            onChange={(e) => setExpiresOn(e.target.value)}
          />
        </div>
      </div>

      <div>
        <label className="text-xs text-gray-600">Változás megjegyzés</label>
        <input
          className="w-full border rounded px-3 py-2"
          value={changeNote}
          onChange={(e) => setChangeNote(e.target.value)}
        />
      </div>

      <div className="flex gap-2 justify-end pt-2 border-t">
        <button
          className="px-3 py-2 rounded border"
          onClick={onClose}
          disabled={busy}>
          Mégse
        </button>
        <button
          className="px-3 py-2 rounded bg-gray-800 text-white disabled:opacity-60"
          onClick={submit}
          disabled={busy}>
          {busy ? "Feltöltés…" : "Feltöltés"}
        </button>
      </div>
    </div>
  );
}
