"use client";

import { useEffect, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";

export default function PackageModal(props: {
  mode: "create" | "edit";
  pkg?: {
    id: string;
    name: string;
    description: string | null;
    is_active: boolean;
  };
  onClose: () => void;
  onSaved: (createdId?: string) => void;
}) {
  const [name, setName] = useState("");
  const [description, setDescription] = useState("");
  const [isActive, setIsActive] = useState(true);
  const [busy, setBusy] = useState(false);

  useEffect(() => {
    setName(props.pkg?.name ?? "");
    setDescription(props.pkg?.description ?? "");
    setIsActive(props.pkg?.is_active ?? true);
  }, [props.pkg]);

  const save = async () => {
    if (!name.trim()) return toast.error("Hiányzó név");
    setBusy(true);
    try {
      const payload = {
        id: props.mode === "edit" ? props.pkg?.id : undefined,
        name: name.trim(),
        description: description.trim() || null,
        is_active: isActive,
      };
      const r = await ApiCall.qms.upsertPackage(payload);
      toast.success("Mentve");
      props.onSaved(r.data?.id ?? props.pkg?.id);
      props.onClose();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Mentés sikertelen");
    } finally {
      setBusy(false);
    }
  };

  return (
    <div className="p-4 space-y-3">
      <div>
        <label className="text-xs text-gray-600">Név</label>
        <input
          className="w-full border rounded px-3 py-2"
          value={name}
          onChange={(e) => setName(e.target.value)}
        />
      </div>

      <div>
        <label className="text-xs text-gray-600">Leírás</label>
        <input
          className="w-full border rounded px-3 py-2"
          value={description}
          onChange={(e) => setDescription(e.target.value)}
        />
      </div>

      <label className="text-sm flex items-center gap-2">
        <input
          type="checkbox"
          checked={isActive}
          onChange={(e) => setIsActive(e.target.checked)}
        />
        Aktív
      </label>

      <div className="flex gap-2 justify-end pt-2 border-t">
        <button
          className="px-3 py-2 rounded border"
          onClick={props.onClose}
          disabled={busy}>
          Mégse
        </button>
        <button
          className="px-3 py-2 rounded bg-emerald-600 text-white disabled:opacity-60"
          onClick={save}
          disabled={busy}>
          {busy ? "Mentés…" : "Mentés"}
        </button>
      </div>
    </div>
  );
}
