"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import Modal from "@/components/shared/modal/Modal";
import useModalMap from "@/hooks/useModalMap";

export type PackageItem = {
  id?: string;

  item_type: "qms_document_version";
  document_id: string;
  version_id: string; // keep required for now (most reliable)
  required: boolean;
  sequence_no: number;

  // UI helpers if BE returns them
  document_title?: string | null;
  document_code?: string | null;
  version_no?: number | null;
};

type DocRow = { id: string; title: string; code: string | null };
type VersionRow = { id: string; version_no: number; file_name: string };

function sortBySeq(items: PackageItem[]) {
  return [...items].sort((a, b) => (a.sequence_no ?? 0) - (b.sequence_no ?? 0));
}

export default function PackageItemsEditor(props: {
  packageId: string;
  initialItems: PackageItem[];
  onSaved: () => void;
}) {
  const { isModalOpen, toggleModal } = useModalMap();

  const [items, setItems] = useState<PackageItem[]>([]);
  const [busy, setBusy] = useState(false);

  // add modal state
  const [docSearch, setDocSearch] = useState("");
  const [docOptions, setDocOptions] = useState<DocRow[]>([]);
  const [docBusy, setDocBusy] = useState(false);
  const [selectedDoc, setSelectedDoc] = useState<DocRow | null>(null);

  const [versions, setVersions] = useState<VersionRow[]>([]);
  const [verBusy, setVerBusy] = useState(false);
  const [selectedVersionId, setSelectedVersionId] = useState<string>("");

  const [required, setRequired] = useState(true);

  useEffect(() => {
    setItems(sortBySeq(props.initialItems ?? []));
  }, [props.initialItems]);

  const nextSeq = useMemo(() => {
    const max = items.reduce((m, x) => Math.max(m, x.sequence_no ?? 0), 0);
    return max + 1;
  }, [items]);

  const loadDocOptions = useCallback(async () => {
    const q = docSearch.trim();
    setDocBusy(true);
    try {
      // use listDocuments global search
      const p = new URLSearchParams();
      p.set("page", "1");
      p.set("pageSize", "25");
      if (q) p.set("q", q);
      p.set("sortField", "title");
      p.set("sortDir", "asc");

      const r = await ApiCall.qms.listDocuments(`?${p.toString()}`);
      const rows = (r.data?.items ?? []) as any[];
      setDocOptions(
        rows.map((x) => ({
          id: String(x.id),
          title: String(x.title ?? "—"),
          code: x.code ?? null,
        }))
      );
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Dokumentum keresés sikertelen");
    } finally {
      setDocBusy(false);
    }
  }, [docSearch]);

  useEffect(() => {
    const t = setTimeout(() => loadDocOptions(), 300);
    return () => clearTimeout(t);
  }, [loadDocOptions]);

  const loadVersions = useCallback(async (docId: string) => {
    setVerBusy(true);
    try {
      const r = await ApiCall.qms.listVersions(docId);
      const vs = (r.data?.versions ?? []) as any[];
      const mapped = vs.map((v) => ({
        id: String(v.id),
        version_no: Number(v.version_no),
        file_name: String(v.file_name ?? ""),
      }));
      setVersions(mapped);
      setSelectedVersionId(mapped[0]?.id ?? "");
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Verziók betöltése sikertelen");
    } finally {
      setVerBusy(false);
    }
  }, []);

  const openAdd = () => {
    setSelectedDoc(null);
    setVersions([]);
    setSelectedVersionId("");
    setRequired(true);
    toggleModal("addItemModal", true);
  };

  const addItem = () => {
    if (!selectedDoc) return toast.error("Válassz dokumentumot");
    if (!selectedVersionId) return toast.error("Válassz verziót");

    const ver = versions.find((v) => v.id === selectedVersionId);

    setItems((prev) =>
      sortBySeq([
        ...prev,
        {
          item_type: "qms_document_version",
          document_id: selectedDoc.id,
          version_id: selectedVersionId,
          required,
          sequence_no: nextSeq,
          document_title: selectedDoc.title,
          document_code: selectedDoc.code,
          version_no: ver?.version_no ?? null,
        },
      ])
    );

    toggleModal("addItemModal", false);
  };

  const move = (idx: number, dir: "up" | "down") => {
    setItems((prev) => {
      const arr = sortBySeq(prev);
      const j = dir === "up" ? idx - 1 : idx + 1;
      if (j < 0 || j >= arr.length) return arr;
      const swapped = [...arr];
      [swapped[idx], swapped[j]] = [swapped[j], swapped[idx]];
      // rewrite sequence_no to be 1..N
      return swapped.map((x, i) => ({ ...x, sequence_no: i + 1 }));
    });
  };

  const remove = (idx: number) => {
    setItems((prev) =>
      sortBySeq(prev.filter((_, i) => i !== idx)).map((x, i) => ({
        ...x,
        sequence_no: i + 1,
      }))
    );
  };

  const toggleRequired = (idx: number) => {
    setItems((prev) =>
      prev.map((x, i) => (i === idx ? { ...x, required: !x.required } : x))
    );
  };

  const save = async () => {
    setBusy(true);
    try {
      // BE expects: PUT /packages/:id/items { items: [...] }
      // keep minimal fields
      const payload = sortBySeq(items).map((x, i) => ({
        item_type: "qms_document_version",
        document_id: x.document_id,
        version_id: x.version_id,
        required: !!x.required,
        sequence_no: i + 1,
      }));

      await ApiCall.qms.replacePackageItems(props.packageId, payload);
      toast.success("Package items mentve");
      props.onSaved();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Mentés sikertelen");
    } finally {
      setBusy(false);
    }
  };

  return (
    <section className="border rounded p-3 space-y-3">
      <div className="flex items-center gap-2">
        <h2 className="font-semibold">Package items</h2>
        <div className="ml-auto flex gap-2">
          <button className="px-2 py-1 rounded border" onClick={openAdd}>
            + Item
          </button>
          <button
            className="px-2 py-1 rounded bg-emerald-600 text-white disabled:opacity-60"
            onClick={save}
            disabled={busy}>
            {busy ? "Mentés…" : "Mentés"}
          </button>
        </div>
      </div>

      <div className="overflow-auto border rounded">
        <table className="min-w-full text-sm">
          <thead className="text-left border-b bg-gray-50">
            <tr>
              <th className="py-2 px-3">#</th>
              <th className="py-2 px-3">Dokumentum</th>
              <th className="py-2 px-3">Verzió</th>
              <th className="py-2 px-3">Required</th>
              <th className="py-2 px-3 text-right">Művelet</th>
            </tr>
          </thead>
          <tbody>
            {sortBySeq(items).map((x, idx) => (
              <tr
                key={`${x.document_id}:${x.version_id}:${idx}`}
                className="border-b hover:bg-gray-50">
                <td className="py-2 px-3">{x.sequence_no}</td>
                <td className="py-2 px-3">
                  <div className="font-medium">
                    {(x.document_code ? `${x.document_code} • ` : "") +
                      (x.document_title ?? x.document_id)}
                  </div>
                  <div className="text-xs text-gray-500">{x.item_type}</div>
                </td>
                <td className="py-2 px-3">
                  {x.version_no != null ? `v${x.version_no}` : x.version_id}
                </td>
                <td className="py-2 px-3">
                  <button
                    className="px-2 py-1 rounded border"
                    onClick={() => toggleRequired(idx)}>
                    {x.required ? "yes" : "no"}
                  </button>
                </td>
                <td className="py-2 px-3">
                  <div className="flex justify-end gap-2">
                    <button
                      className="px-2 py-1 rounded border"
                      onClick={() => move(idx, "up")}>
                      ▲
                    </button>
                    <button
                      className="px-2 py-1 rounded border"
                      onClick={() => move(idx, "down")}>
                      ▼
                    </button>
                    <button
                      className="px-2 py-1 rounded border"
                      onClick={() => remove(idx)}>
                      Töröl
                    </button>
                  </div>
                </td>
              </tr>
            ))}

            {!items.length && (
              <tr>
                <td colSpan={5} className="py-8 text-center text-gray-500">
                  Nincs item. Adj hozzá dokumentum verziókat.
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>

      <Modal
        show={isModalOpen("addItemModal")}
        onClose={() => toggleModal("addItemModal", false)}
        title="Package item hozzáadása"
        height="auto"
        width="860px">
        <div className="p-4 space-y-3">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-2">
            <div>
              <label className="text-xs text-gray-600">
                Dokumentum keresés
              </label>
              <input
                className="w-full border rounded px-3 py-2"
                value={docSearch}
                onChange={(e) => setDocSearch(e.target.value)}
                placeholder="Cím / kód…"
              />
              <div className="mt-2 max-h-[220px] overflow-auto border rounded">
                {docBusy ? (
                  <div className="p-3 text-xs text-gray-500">Keresés…</div>
                ) : docOptions.length ? (
                  docOptions.map((d) => (
                    <button
                      key={d.id}
                      className={[
                        "w-full text-left px-3 py-2 border-b last:border-b-0 hover:bg-gray-50",
                        selectedDoc?.id === d.id ? "bg-amber-50" : "",
                      ].join(" ")}
                      onClick={async () => {
                        setSelectedDoc(d);
                        await loadVersions(d.id);
                      }}>
                      <div className="font-medium">
                        {(d.code ? `${d.code} • ` : "") + d.title}
                      </div>
                      <div className="text-xs text-gray-500">{d.id}</div>
                    </button>
                  ))
                ) : (
                  <div className="p-3 text-xs text-gray-500">
                    Nincs találat.
                  </div>
                )}
              </div>
            </div>

            <div>
              <label className="text-xs text-gray-600">Verzió</label>
              <select
                className="w-full border rounded px-3 py-2"
                value={selectedVersionId}
                onChange={(e) => setSelectedVersionId(e.target.value)}
                disabled={!selectedDoc || verBusy}>
                {!selectedDoc ? (
                  <option value="">Válassz dokumentumot</option>
                ) : null}
                {verBusy ? <option value="">Betöltés…</option> : null}
                {versions.map((v) => (
                  <option key={v.id} value={v.id}>
                    v{v.version_no} • {v.file_name}
                  </option>
                ))}
              </select>

              <label className="mt-3 text-sm flex items-center gap-2">
                <input
                  type="checkbox"
                  checked={required}
                  onChange={(e) => setRequired(e.target.checked)}
                />
                Required
              </label>

              <div className="mt-4 text-xs text-gray-500">
                Tipp: most fixen “verzió” alapú itemet rakunk be (legstabilabb),
                nem “default/latest”-et.
              </div>
            </div>
          </div>

          <div className="flex gap-2 justify-end pt-2 border-t">
            <button
              className="px-3 py-2 rounded border"
              onClick={() => toggleModal("addItemModal", false)}>
              Mégse
            </button>
            <button
              className="px-3 py-2 rounded bg-emerald-600 text-white"
              onClick={addItem}>
              Hozzáad
            </button>
          </div>
        </div>
      </Modal>
    </section>
  );
}
