"use client";
import { useEffect, useRef, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";

export default function DocumentModal({
  mode,
  categoryId,
  doc,
  onClose,
  onSaved,
}: {
  mode: "create" | "edit";
  categoryId: string;
  doc?: any;
  onClose: () => void;
  onSaved: (createdId?: string) => void;
}) {
  const [code, setCode] = useState("");
  const [title, setTitle] = useState("");
  const [description, setDescription] = useState("");
  const [status, setStatus] = useState("active");
  const [busy, setBusy] = useState(false);

  // ✅ only for create: upload the first version
  const fileRef = useRef<HTMLInputElement | null>(null);
  const [versionStatus, setVersionStatus] = useState("published");
  const [changeNote, setChangeNote] = useState("");
  const [effectiveFrom, setEffectiveFrom] = useState("");
  const [expiresOn, setExpiresOn] = useState("");

  useEffect(() => {
    setCode(doc?.code ?? "");
    setTitle(doc?.title ?? "");
    setDescription(doc?.description ?? "");
    setStatus(doc?.status ?? "active");
  }, [doc]);

  const save = async () => {
    if (!title.trim()) return toast.error("Hiányzó cím");

    // ✅ if creating, strongly encourage uploading file
    if (mode === "create") {
      const f = fileRef.current?.files?.[0];
      if (!f) {
        if (
          !confirm(
            "Nem választottál fájlt.\n\nLétrehozzuk a dokumentumot fájl nélkül? (később feltölthető verzióként)"
          )
        )
          return;
      }
    }

    setBusy(true);
    try {
      if (mode === "create") {
        // 1) create document shell
        const r = await ApiCall.qms.createDocument({
          categoryId,
          code: code.trim() || null,
          title: title.trim(),
          description: description.trim() || null,
          status,
        });

        const docId = r.data?.id as string;
        if (!docId) throw new Error("Missing created document id");

        // 2) optional: upload first version
        const f = fileRef.current?.files?.[0];
        if (f) {
          const form = new FormData();
          form.append("file", f);
          form.append("status", versionStatus);
          form.append("change_note", changeNote || "");
          if (effectiveFrom) form.append("effective_from", effectiveFrom);
          if (expiresOn) form.append("expires_on", expiresOn);

          await ApiCall.qms.uploadVersion(docId, form);
        }

        toast.success(
          f ? "Dokumentum + első verzió létrehozva" : "Dokumentum létrehozva"
        );
        onSaved(docId);
      } else {
        await ApiCall.qms.updateDocument(doc.id, {
          code: code.trim() || null,
          title: title.trim(),
          description: description.trim() || null,
          status,
        });
        toast.success("Mentve");
        onSaved();
      }

      onClose();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Mentés sikertelen");
    } finally {
      setBusy(false);
    }
  };

  return (
    <div className="p-4 space-y-3">
      <div className="grid grid-cols-1 md:grid-cols-3 gap-2">
        <div>
          <label className="text-xs text-gray-600">Kód</label>
          <input
            className="w-full border rounded px-3 py-2"
            value={code}
            onChange={(e) => setCode(e.target.value)}
          />
        </div>
        <div className="md:col-span-2">
          <label className="text-xs text-gray-600">Cím</label>
          <input
            className="w-full border rounded px-3 py-2"
            value={title}
            onChange={(e) => setTitle(e.target.value)}
          />
        </div>
      </div>

      <div>
        <label className="text-xs text-gray-600">Leírás</label>
        <input
          className="w-full border rounded px-3 py-2"
          value={description}
          onChange={(e) => setDescription(e.target.value)}
        />
      </div>

      <div>
        <label className="text-xs text-gray-600">Dokumentum státusz</label>
        <select
          className="w-full border rounded px-3 py-2"
          value={status}
          onChange={(e) => setStatus(e.target.value)}>
          <option value="active">active</option>
          <option value="draft">draft</option>
          <option value="obsolete">obsolete</option>
        </select>
      </div>

      {mode === "create" ? (
        <div className="border rounded p-3 space-y-2 bg-gray-50">
          <div className="font-medium text-sm">
            Első verzió feltöltés (ajánlott)
          </div>

          <div>
            <label className="text-xs text-gray-600">Fájl</label>
            <input
              ref={fileRef}
              type="file"
              className="w-full border rounded px-3 py-2 bg-white"
            />
          </div>

          <div className="grid grid-cols-1 md:grid-cols-3 gap-2">
            <div>
              <label className="text-xs text-gray-600">Verzió státusz</label>
              <select
                className="w-full border rounded px-3 py-2"
                value={versionStatus}
                onChange={(e) => setVersionStatus(e.target.value)}>
                <option value="published">published</option>
                <option value="draft">draft</option>
                <option value="archived">archived</option>
              </select>
            </div>
            <div>
              <label className="text-xs text-gray-600">Hatályos ettől</label>
              <input
                type="date"
                className="w-full border rounded px-3 py-2"
                value={effectiveFrom}
                onChange={(e) => setEffectiveFrom(e.target.value)}
              />
            </div>
            <div>
              <label className="text-xs text-gray-600">Lejár</label>
              <input
                type="date"
                className="w-full border rounded px-3 py-2"
                value={expiresOn}
                onChange={(e) => setExpiresOn(e.target.value)}
              />
            </div>
          </div>

          <div>
            <label className="text-xs text-gray-600">Változás megjegyzés</label>
            <input
              className="w-full border rounded px-3 py-2"
              value={changeNote}
              onChange={(e) => setChangeNote(e.target.value)}
            />
          </div>
        </div>
      ) : null}

      <div className="flex gap-2 justify-end pt-2 border-t">
        <button
          className="px-3 py-2 rounded border"
          onClick={onClose}
          disabled={busy}>
          Mégse
        </button>
        <button
          className="px-3 py-2 rounded bg-emerald-600 text-white disabled:opacity-60"
          onClick={save}
          disabled={busy}>
          {busy ? "Mentés…" : "Mentés"}
        </button>
      </div>
    </div>
  );
}
