"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";

type SubjectType = "user" | "person" | "role";

type ApproverDraft = {
  subjectType: SubjectType;
  subjectId: string;
  displayName: string;
};

type StepDraft = {
  sequenceNo: number; // 1..N
  approvers: ApproverDraft[];
};

type RoleRow = { id: string; name: string };

type PeopleOption = { id: string; name: string };
type UserOption = { id: string; name?: string | null; email?: string | null };

function normalizeUserLabel(u: UserOption) {
  return u.name || u.email || u.id;
}

async function searchUsersBestEffort(search: string): Promise<UserOption[]> {
  const q = search.trim();
  if (!q) return [];

  // admin-users/list query shape is unknown, so we try a couple common patterns.
  // If your BE expects something else, update these querystrings.
  const tries = [
    `?q=${encodeURIComponent(q)}`,
    `?search=${encodeURIComponent(q)}`,
    `?query=${encodeURIComponent(q)}`,
    `?page=1&pageSize=10&q=${encodeURIComponent(q)}`,
  ];

  for (const t of tries) {
    try {
      const r = await ApiCall.users.getList(t);
      const data = r.data;
      const items = (data?.items ??
        data?.rows ??
        data?.data ??
        data?.list ??
        data?.results ??
        []) as any[];

      // normalize id/name/email
      const mapped = items
        .map((x) => ({
          id: String(x.id ?? x.user_id ?? x.uuid ?? ""),
          name: x.name ?? x.full_name ?? null,
          email: x.email ?? null,
        }))
        .filter((x) => x.id);

      if (mapped.length) return mapped;
    } catch {
      // ignore and try next
    }
  }
  return [];
}

export default function CreateApprovalRequestModal(props: {
  versionId: string;
  onClose: () => void;
  onCreated: (id: string) => void;
}) {
  const [note, setNote] = useState("");

  const [roles, setRoles] = useState<RoleRow[]>([]);
  const [loadingRoles, setLoadingRoles] = useState(false);

  // Steps: sequencing enforced
  const [steps, setSteps] = useState<StepDraft[]>([
    { sequenceNo: 1, approvers: [] },
  ]);

  // picker
  const [pickType, setPickType] = useState<SubjectType>("role");
  const [search, setSearch] = useState("");
  const [options, setOptions] = useState<
    { id: string; label: string; subjectType: SubjectType }[]
  >([]);
  const [busySearch, setBusySearch] = useState(false);

  // which step are we adding into?
  const [activeStepNo, setActiveStepNo] = useState(1);

  const loadRoles = useCallback(async () => {
    setLoadingRoles(true);
    try {
      const r = await ApiCall.qms.listRoles();
      const items = (r.data?.items ?? r.data ?? []) as any[];
      setRoles(
        items.map((x) => ({
          id: String(x.id),
          name: String(x.name ?? x.role_name ?? "Role"),
        }))
      );
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Roles betöltése sikertelen");
    } finally {
      setLoadingRoles(false);
    }
  }, []);

  useEffect(() => {
    loadRoles();
  }, [loadRoles]);

  // debounced search
  useEffect(() => {
    const q = search.trim();
    const handle = setTimeout(async () => {
      setBusySearch(true);
      try {
        if (pickType === "role") {
          const filtered = roles
            .filter((r) => r.name.toLowerCase().includes(q.toLowerCase()))
            .slice(0, 20)
            .map((r) => ({
              id: r.id,
              label: r.name,
              subjectType: "role" as const,
            }));
          setOptions(filtered);
          return;
        }

        if (pickType === "person") {
          const r = await ApiCall.peoples.searchPeopleOptions(q);
          const items = (r.data?.items ?? r.data ?? []) as any[];
          const mapped = items
            .map((x) => ({
              id: String(x.id),
              label: String(x.name ?? x.label ?? "Person"),
              subjectType: "person" as const,
            }))
            .slice(0, 20);
          setOptions(mapped);
          return;
        }

        // user
        const users = await searchUsersBestEffort(q);
        setOptions(
          users.slice(0, 20).map((u) => ({
            id: u.id,
            label: normalizeUserLabel(u),
            subjectType: "user" as const,
          }))
        );
      } catch (e) {
        setOptions([]);
      } finally {
        setBusySearch(false);
      }
    }, 350);

    return () => clearTimeout(handle);
  }, [search, pickType, roles]);

  const step = useMemo(
    () => steps.find((s) => s.sequenceNo === activeStepNo) ?? null,
    [steps, activeStepNo]
  );

  const addStep = () => {
    const nextNo = (steps.at(-1)?.sequenceNo ?? 0) + 1;
    setSteps((s) => [...s, { sequenceNo: nextNo, approvers: [] }]);
    setActiveStepNo(nextNo);
  };

  const removeStep = (seq: number) => {
    if (steps.length <= 1) return;
    setSteps((prev) => prev.filter((s) => s.sequenceNo !== seq));
    // re-number sequentially
    setSteps((prev) =>
      prev
        .filter((s) => s.sequenceNo !== seq)
        .map((s, idx) => ({ ...s, sequenceNo: idx + 1 }))
    );
    setActiveStepNo(1);
  };

  const addApproverToStep = (seq: number, a: ApproverDraft) => {
    setSteps((prev) =>
      prev.map((s) => {
        if (s.sequenceNo !== seq) return s;
        const exists = s.approvers.some(
          (x) => x.subjectType === a.subjectType && x.subjectId === a.subjectId
        );
        if (exists) return s;
        return { ...s, approvers: [...s.approvers, a] };
      })
    );
  };

  const removeApproverFromStep = (seq: number, idx: number) => {
    setSteps((prev) =>
      prev.map((s) =>
        s.sequenceNo !== seq
          ? s
          : { ...s, approvers: s.approvers.filter((_, i) => i !== idx) }
      )
    );
  };

  const canSubmit = useMemo(() => {
    const anyApprover = steps.some((s) => s.approvers.length > 0);
    return anyApprover;
  }, [steps]);

  const submit = async () => {
    if (!canSubmit) return toast.error("Adj hozzá legalább 1 approvert");

    // payload expected by BE (based on our controllers earlier):
    // { note, steps: [{sequenceNo, approvers:[{subjectType,subjectId}]}] }
    const payload = {
      note: note.trim() || null,
      steps: steps
        .filter((s) => s.approvers.length > 0)
        .map((s) => ({
          sequenceNo: s.sequenceNo,
          approvers: s.approvers.map((a) => ({
            subjectType: a.subjectType,
            subjectId: a.subjectId,
          })),
        })),
    };

    try {
      const r = await ApiCall.qms.createApprovalRequest(
        props.versionId,
        payload
      );
      const id = String(r.data?.id ?? "");
      if (!id) {
        toast.success("Created");
        props.onClose();
        return;
      }
      props.onCreated(id);
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ?? "Approval request létrehozása sikertelen"
      );
    }
  };

  return (
    <div className="p-4 space-y-3">
      <div className="grid grid-cols-1 md:grid-cols-3 gap-2 items-end">
        <div className="md:col-span-2">
          <label className="text-xs text-gray-600">
            Megjegyzés (opcionális)
          </label>
          <input
            className="w-full border rounded px-3 py-2"
            value={note}
            onChange={(e) => setNote(e.target.value)}
            placeholder="Pl. mi változott, mire figyeljenek"
          />
        </div>

        <div>
          <label className="text-xs text-gray-600">Aktív lépés</label>
          <select
            className="w-full border rounded px-2 py-2"
            value={activeStepNo}
            onChange={(e) => setActiveStepNo(Number(e.target.value))}>
            {steps.map((s) => (
              <option key={s.sequenceNo} value={s.sequenceNo}>
                Step #{s.sequenceNo}
              </option>
            ))}
          </select>
        </div>
      </div>

      {/* steps overview */}
      <div className="border rounded p-3 space-y-2">
        <div className="flex items-center gap-2">
          <div className="font-semibold">Steps (sequencing enforced)</div>
          <button
            className="ml-auto px-2 py-1 rounded border"
            onClick={addStep}>
            + Step
          </button>
        </div>

        <div className="space-y-2">
          {steps.map((s) => (
            <div key={s.sequenceNo} className="border rounded p-2">
              <div className="flex items-center gap-2">
                <div className="font-medium">Step #{s.sequenceNo}</div>
                <div className="ml-auto flex gap-2">
                  <button
                    className="px-2 py-1 rounded border"
                    onClick={() => setActiveStepNo(s.sequenceNo)}>
                    Szerkeszt
                  </button>
                  <button
                    className="px-2 py-1 rounded border disabled:opacity-50"
                    disabled={steps.length <= 1}
                    onClick={() => removeStep(s.sequenceNo)}
                    title={
                      steps.length <= 1 ? "Legalább 1 step kell" : "Töröl"
                    }>
                    Töröl
                  </button>
                </div>
              </div>

              {s.approvers.length ? (
                <div className="mt-2 flex flex-wrap gap-2">
                  {s.approvers.map((a, idx) => (
                    <span
                      key={`${a.subjectType}:${a.subjectId}`}
                      className="text-xs px-2 py-1 rounded border bg-gray-50 flex items-center gap-2">
                      <b>{a.displayName}</b>
                      <span className="text-gray-500">({a.subjectType})</span>
                      <button
                        className="text-red-600"
                        onClick={() =>
                          removeApproverFromStep(s.sequenceNo, idx)
                        }
                        title="Eltávolít">
                        ✕
                      </button>
                    </span>
                  ))}
                </div>
              ) : (
                <div className="mt-2 text-xs text-gray-500">
                  Nincs approver ebben a stepben.
                </div>
              )}
            </div>
          ))}
        </div>
      </div>

      {/* picker */}
      <div className="border rounded p-3 space-y-2">
        <div className="flex items-center gap-2 flex-wrap">
          <div className="font-semibold">
            Approver hozzáadása • Step #{activeStepNo}
          </div>

          <select
            className="ml-auto border rounded px-2 py-1"
            value={pickType}
            onChange={(e) => setPickType(e.target.value as SubjectType)}
            disabled={loadingRoles}>
            <option value="role">Role</option>
            <option value="person">Person</option>
            <option value="user">User</option>
          </select>

          <input
            className="border rounded px-3 py-1"
            value={search}
            onChange={(e) => setSearch(e.target.value)}
            placeholder={
              pickType === "role"
                ? "Role keresés…"
                : pickType === "person"
                ? "Person keresés…"
                : "User keresés…"
            }
          />
        </div>

        {busySearch ? (
          <div className="text-xs text-gray-500">Keresés…</div>
        ) : options.length ? (
          <div className="max-h-[220px] overflow-auto border rounded">
            {options.map((o) => (
              <button
                key={`${o.subjectType}:${o.id}`}
                className="w-full text-left px-3 py-2 hover:bg-gray-50 border-b last:border-b-0"
                onClick={() => {
                  if (!step) return;
                  addApproverToStep(step.sequenceNo, {
                    subjectType: o.subjectType,
                    subjectId: o.id,
                    displayName: o.label,
                  });
                }}>
                <div className="font-medium">{o.label}</div>
                <div className="text-xs text-gray-500">{o.subjectType}</div>
              </button>
            ))}
          </div>
        ) : (
          <div className="text-xs text-gray-500">Nincs találat.</div>
        )}
      </div>

      <div className="flex gap-2 justify-end pt-2 border-t">
        <button className="px-3 py-2 rounded border" onClick={props.onClose}>
          Mégse
        </button>
        <button
          className="px-3 py-2 rounded bg-emerald-600 text-white disabled:opacity-60"
          onClick={submit}
          disabled={!canSubmit}>
          Létrehoz
        </button>
      </div>
    </div>
  );
}
