"use client";
import React, {
  useState,
  useRef,
  useEffect,
  useContext,
  useCallback,
} from "react";
import Image from "next/image";
// import MediaPreview from "../MediaObject/MediaPreview";
import Loader from "@/components/shared/loader/global-loading";
import icons from "@/assets/icons/icons";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";
import UploadLogPanel from "@/components/features/upload-log-panel/UploadLogPanel";
import useDynamicManager from "@/store/managers/useDynamicManager";
import UploadProgressSteps from "@/components/features/upload-log-panel/UploadProgressSteps";
import UploadProgressBar from "@/components/features/upload-log-panel/UploadProgressBar";
import Modal from "@/components/shared/modal/Modal";
import ReviewProductGroups from "@/components/product-groups/upload/ReviewProductGroups";
import ReviewCategoryConflicts from "@/components/product-groups/upload/ReviewProductCategories";
import ReviewProducts from "@/components/product-groups/upload/ReviewProducts";
import { useModalManager } from "@/hooks/useModalManager";

const UploadProducts = () => {
  const [selectedFiles, setSelectedFiles] = useState([]);
  const [isDragging, setIsDragging] = useState(false);
  const [dragAreaStyle, setDragAreaStyle] = useState({});
  const [filePreviews, setFilePreviews] = useState([]);
  const [status, setStatus] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const [useStaging, setUseStaging] = useState(true);
  const [activeTab, setActiveTab] = useState("productGroups");
  const { toggleModal, isModalOpen } = useModalManager();
  const { state, clearLogs } = useDynamicManager("uploadLog");
  const [conflictBatchId, setConflictBatchId] = useState(null);

  console.log("state", state);
  useEffect(() => {
    // Reset all state on initial mount
    setSelectedFiles([]);
    setFilePreviews([]);
    setUseStaging(true);
    clearLogs();
    setStatus("");
  }, []);

  useEffect(() => {
    return () => {
      filePreviews.forEach((fp) => URL.revokeObjectURL(fp.preview));
    };
  }, [filePreviews]);
  useEffect(() => {
    setDragAreaStyle({
      border: isDragging ? "2px dashed rgb(156, 163, 175)" : "none",
      backgroundColor: isDragging ? "rgb(147, 197, 253)" : "lightgray",
    });
  }, [isDragging]);

  useEffect(() => {
    const handleDrop = (e) => {
      e.preventDefault();
      clearLogs();
      const files = Array.from(e.dataTransfer.files);
      setSelectedFiles(files);
      setFilePreviews(
        files.map((file) => {
          const item = { preview: URL.createObjectURL(file), data: file };
          return item;
        })
      );
      setIsDragging(false);
    };

    const handleDragOver = (e) => {
      e.preventDefault();
      setIsDragging(true);
    };
    const handleDragLeave = (e) => {
      e.preventDefault();
      setIsDragging(false);
    };
    document.addEventListener("drop", handleDrop);
    document.addEventListener("dragover", handleDragOver);
    document.addEventListener("dragleave", handleDragLeave);

    return () => {
      document.removeEventListener("drop", handleDrop);
      document.removeEventListener("dragover", handleDragOver);
      document.removeEventListener("dragleave", handleDragLeave);
    };
  }, []);

  const handleSelect = (e) => {
    clearLogs();
    const files = Array.from(e.target.files);
    setSelectedFiles(files);
    setFilePreviews(
      files.map((file) => {
        const item = { preview: URL.createObjectURL(file), data: file };
        return item;
      })
    );
  };
  const handleRemove = (index) => {
    const newFiles = [...selectedFiles];
    newFiles.splice(index, 1);
    setSelectedFiles(newFiles);
    setFilePreviews(
      newFiles.map((file) => {
        const item = { preview: URL.createObjectURL(file), data: file };
        return item;
      })
    );
  };
  const handleSubmit = useCallback(
    async (e) => {
      e.preventDefault();
      setIsLoading(true);

      try {
        const snapshot = await ApiCall.snapshots.postCreate();
        console.log("snapshot", snapshot);
        if (snapshot?.data?.success) {
          setIsLoading(false);

          toast.success("Az adatbázisok mentése sikerült");
          const formData = new FormData();

          formData.append("setUseStaging", setUseStaging);

          for (let i = 0; i < selectedFiles.length; i++) {
            formData.append("file", selectedFiles[i]);
          }

          const response = await ApiCall.staging.uploadStagingProducts(
            formData
          );

          if (response.data.error) {
            toast.error(response.data.error, { removeDelay: 15000 });
            return;
          }
          if (response.status === 200) {
            const groups = response.data?.groups || {};

            // ✅ Look into stagedGroups for batchId if conflicts exist
            if (groups.conflictCount >= 0 && groups.stagedGroups?.length > 0) {
              const batchId = groups.stagedGroups?.[0]?.importBatchId;
              if (batchId) {
                setConflictBatchId(batchId);
                //   setShowConflictModal(true);
                toggleModal("showConflictModal");
              }
            }
          }

          toast.success("✅ Sikeres feltöltés");
          setSelectedFiles([]);
          setFilePreviews([]);
          setStatus("Sikeres feltöltés");
        } else {
          toast.error(
            "Az adatbázisok mentése NEM sikerült! Kérjük kezdje újra a folyamatot!"
          );
        }
      } catch (error) {
        if (error.response) {
          const serverMsg =
            error.response.data?.message || "Ismeretlen szerverhiba";
          const missingSheets = error.response.data?.missingSheets;
          const missingHeaders = error.response.data?.missingHeaders;

          const sheetInfo = missingSheets?.length
            ? `Hiányzó munkalap(ok): ${missingSheets.join(", ")}`
            : "";

          const headerInfo = missingHeaders?.length
            ? `Hiányzó fejléc(ek): ${missingHeaders
                .map((h) => `${h.sheet}: ${h.headers.join(", ")}`)
                .join(" | ")}`
            : "";

          const fullMessage = [serverMsg, sheetInfo, headerInfo]
            .filter(Boolean)
            .join(" • ");

          toast.error(`❌ ${fullMessage}`, {
            id: "socket-status",
            duration: 15000,
          });
        } else if (error.request) {
          toast.error("❌ Nem sikerült kapcsolatot létesíteni a szerverrel.", {
            id: "socket-status",
            duration: 15000,
          });
        } else {
          toast.error(`Hiba történt: ${error.message}`, {
            id: "socket-status",
            duration: 15000,
          });
        }
      } finally {
        setIsLoading(false);
      }
    },
    [selectedFiles, useStaging]
  );

  return (
    <div className="w-full">
      {isLoading && <Loader position={"absolute"} />}
      <div className="flex flex-row flex-wrap justify-evenly items-center gap-2 mb-4">
        <label className="flex items-center gap-2 text-sm">
          <input
            type="checkbox"
            checked={useStaging}
            onChange={(e) => setUseStaging(e.target.checked)}
            className="w-[1.5rem] h-[1.5rem]"
          />
          Staging feltöltés aktiválva
        </label>
      </div>
      <div
        className={`w-auto bg-gray-100 text-blue-600 h-40 flex flex-row justify-center items-center`}
        style={dragAreaStyle}
        onDrop={(e) => {
          e.preventDefault();
          const files = Array.from(e.dataTransfer.files);
          setSelectedFiles(files);
          setFilePreviews(
            files.map((file) => {
              const item = { preview: URL.createObjectURL(file), data: file };
              return item;
            })
          );
          setIsDragging(false);
        }}
        onDragOver={(e) => {
          e.preventDefault();
          setIsDragging(true);
        }}
        onDragLeave={(e) => {
          e.preventDefault();
          setIsDragging(false);
        }}>
        <div className="flex flex-col justify-center items-center">
          <p className="font-bold text-xl">
            Húzza ide a kiválasztott fájlokat a feltöltéshez vagy
          </p>
          <input id="file-input" type="file" onChange={handleSelect} />
        </div>
      </div>
      {state && (
        <>
          <UploadProgressBar />
          <UploadProgressSteps />
          <UploadLogPanel />
        </>
      )}

      {selectedFiles.length > 0 && (
        <div className="flex flex-row flex-wrap justify-even items-start my-4 w-full">
          {filePreviews?.map((file, index) => {
            switch (file?.data?.type) {
              case "application/pdf":
                return (
                  <div
                    className={`flex flex-col mr-2 mb-2 items-center border-1 w-1/6 mx-2`}
                    key={index}>
                    pdf
                    {/* <MediaPreview
                      data={file?.data}
                      className={"w-full"}
                      id={file?.data?.name}
                      preview={file?.preview}
                    /> */}
                    <button
                      onClick={() => handleRemove(index)}
                      className="text-red-600 my-4">
                      Eltávolítás
                    </button>
                  </div>
                );
              case "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet":
                return (
                  <div
                    className={`flex flex-col mr-2 mb-2 items-center border-1 w-1/6 mx-2`}
                    key={index}>
                    <span className="flex size-60">{icons.excel}</span>
                    <p>{file?.data?.name}</p>
                    <button
                      onClick={() => handleRemove(index)}
                      className="text-red-600 my-4">
                      Eltávolítás
                    </button>
                  </div>
                );
              default:
                return (
                  <div key={index} className="w-1/5 mx-2 ">
                    <Image
                      src={file?.preview}
                      width={250}
                      height={250}
                      className="w-full h-full object-contain flex flex-col justify-center items-start bg-gray-100"
                      style={{ maxHeight: "250px", height: "250px" }}
                      alt={file.data.name ?? ""}
                    />
                    <p>{file?.data?.name}</p>
                    <button
                      onClick={() => handleRemove(index)}
                      className="text-red-600">
                      Eltávolítás
                    </button>
                  </div>
                );
            }
          })}
        </div>
      )}
      {status && <h4 className="text-green-500/75 font-bold my-4">{status}</h4>}
      <button
        className="rounded bg-blue-500/75 hover:bg-blue-500 text-white px-4 py-2 my-4 sticky bottom-0 w-auto"
        disabled={selectedFiles.length === 0}
        onClick={(e) => handleSubmit(e)}>
        Feltöltés
      </button>
      <Modal
        title={"Termék feltöltés konfliktus kezelése"}
        show={isModalOpen("showConflictModal")}
        onClose={() => toggleModal("showConflictModal")}
        height={"100%"}>
        <div className="flex gap-4 text-sm mb-2">
          <span>➕ Új érték</span>
          <span>➖ Törölt érték</span>
          <span>🔁 Módosított érték</span>
          <span>✅ Egyező</span>
        </div>
        <div className="flex flex-col">
          <div className="flex gap-4 mb-4">
            <button
              onClick={() => setActiveTab("productGroups")}
              className={`px-4 py-2 ${
                activeTab === "productGroups"
                  ? "bg-blue-500 text-white"
                  : "bg-gray-200"
              }`}>
              Product Group Conflicts
            </button>
            <button
              onClick={() => setActiveTab("products")}
              className={`px-4 py-2 ${
                activeTab === "products"
                  ? "bg-blue-500 text-white"
                  : "bg-gray-200"
              }`}>
              Product Conflicts
            </button>
            {/* <button
                    onClick={() => setActiveTab("categories")}
                    className={`px-4 py-2 ${
                      activeTab === "categories"
                        ? "bg-blue-500 text-white"
                        : "bg-gray-200"
                    }`}>
                    Category Conflicts
                  </button> */}
          </div>
          {/* {activeTab === "categories" && (
                  <ReviewCategoryConflicts batchId={conflictBatchId} />
                )} */}
          {activeTab === "productGroups" && (
            <ReviewProductGroups batchId={conflictBatchId} />
          )}
          {activeTab === "products" && (
            <ReviewProducts batchId={conflictBatchId} />
          )}
        </div>
      </Modal>
    </div>
  );
};

export default UploadProducts;
