"use client";
import React, { useEffect, useState } from "react";
import ApiCall from "@/api/api-calls";
import Image from "next/image";

export default function ReviewProductGroups({ batchId }) {
  const [items, setItems] = useState([]);
  const [statusFilter, setStatusFilter] = useState("conflict");

  useEffect(() => {
    const fetchData = async () => {
      const { data } = await ApiCall.staging.getProductGroupStaging({
        batchId,
        conflictStatus: statusFilter,
      });
      setItems(data.items || []);
    };
    fetchData();
  }, [batchId, statusFilter]);

  const getDiffClass = (live, staging) => {
    const normalize = (val) =>
      typeof val === "string" ? val.trim().toLowerCase() : val;

    const cleanLive = normalize(live);
    const cleanStaging = normalize(staging);

    if (!cleanStaging && cleanLive)
      return "bg-red-100 text-red-800 px-2 py-1 rounded";
    if (cleanLive !== cleanStaging)
      return "bg-yellow-100 text-yellow-800 font-medium px-2 py-1 rounded";
    return "bg-green-100 text-green-800 font-medium px-2 py-1 rounded";
  };

  const highlightDiff = (live = "", staging = "") => {
    const liveStr = String(live || "");
    const stagingStr = String(staging || "");

    if (!liveStr || !stagingStr || liveStr === stagingStr) return stagingStr;

    const liveWords = liveStr.split(/\s+/);
    const stagingWords = stagingStr.split(/\s+/);

    return stagingWords.map((word, idx) => {
      const liveWord = liveWords[idx];
      const isDifferent = word !== liveWord;
      return (
        <span
          key={idx}
          className={isDifferent ? "bg-red-200 px-1 rounded-sm" : ""}>
          {word + " "}
        </span>
      );
    });
  };

  const getReadableValue = (val, key) => {
    if (!val || val?.length === 0) return "(nincs)";
    if (typeof val === "string") return val;

    // if (Array.isArray(val)) {
    //   return val
    //     .map((entry) => {
    //       if (!entry) return "?";
    //       if (typeof entry === "string") return entry;
    //       switch (key) {
    //         case "productColors":
    //           return entry?.colorTitle?.value ?? entry?._id ?? "?";
    //         case "productSizes":
    //           return entry?.sizeTitle?.value ?? entry?._id ?? "?";
    //         case "productGroupMark":
    //           return entry?.markTitle?.value ?? entry?._id ?? "?";
    //         case "productGroupCategory":
    //         case "productGroupCategoryHierarchy":
    //           return entry?.productCategoryTitle?.value ?? entry?._id ?? "?";
    //         default:
    //           return entry?._id ?? "[?]";
    //       }
    //     })
    //     .join(", ");
    // }
    if (key === "productPieceWebshopPrice") {
      if (typeof val === "object" && val?.value != null) {
        return `${val.value.toFixed(2)} Ft`;
      }
    }
    if (key === "productPriceType") {
      return val?.value || "(nincs)";
    }
    if (Array.isArray(val)) {
      return val
        .map((entry) => {
          if (!entry) return "(nincs)";
          if (typeof entry === "string") return entry;

          // Try common fields
          return (
            entry?.markTitle?.value ||
            entry?.productCategoryTitle?.value ||
            entry?.colorTitle?.value ||
            entry?.sizeTitle?.value ||
            entry?.value ||
            entry?._id?.toString() ||
            "(nincs)"
          );
        })
        .join(", ");
    }
    if (typeof val === "object") {
      return (
        val?.productCategoryTitle?.value ||
        val?.markTitle?.value ||
        val?.colorTitle?.value ||
        val?.sizeTitle?.value ||
        val?.value ||
        val?._id?.toString() ||
        JSON.stringify(val)
      );
    }

    // if (typeof val === "object" && val?.value !== undefined) {
    //   return val.value || "(nincs)";
    // }

    return JSON.stringify(val);
  };

  const getStagingValue = (item, key) => {
    if (key === "productGroupCategory") {
      return item.productGroupCategoryHierarchy;
    }
    return item[key];
  };

  const getLiveValue = (item, key) => {
    if (key === "productGroupCategory") {
      return item.conflictMeta?.liveMatchId?.productGroupCategoryHierarchy;
    }
    return item.conflictMeta?.liveMatchId?.[key];
  };

  const getImageSrc = (img) => {
    if (!img) return "/images/no-image.png";
    const name = typeof img === "string" ? img : img?.originalname || img?._id;
    return ApiCall.mediaStore.getResized("/2/2/0", name);
  };

  const translateKey = (key) => {
    const productGroupKeys = [
      { original: "productGroupTitle", name: "Termékcsoport neve" },
      { original: "productGroupTitleLink", name: "Termék csoport link név" },
      { original: "productSizes", name: "Termék méretek" },
      { original: "productColors", name: "Termék színek" },
      { original: "productPieceWebshopPrice", name: "Alap ár" },
      { original: "productPriceType", name: "Ár típusa" },
      { original: "productGroupCategory", name: "Termék csoport kategória" },
      { original: "productGroupMark", name: "Termék csoport márkanév" },
      {
        original: "productFeaturedImageMeta",
        name: "Termék csoport kiemelt kép",
      },
    ];
    const translatedKey = productGroupKeys.find((k) => k.original === key);
    return translatedKey ? translatedKey.name : key;
  };

  const handleApprove = async (id) => {
    try {
      await ApiCall.staging.approveProductGroup(id);
      const { data } = await ApiCall.staging.getProductGroupStaging({
        batchId,
        conflictStatus: statusFilter,
      });
      setItems(data.items || []);
    } catch (err) {
      console.error("Elfogad failed", err);
    }
  };

  const handleDiscard = async (id) => {
    try {
      await ApiCall.staging.discardProductGroup(id);
      const { data } = await ApiCall.staging.getProductGroupStaging({
        batchId,
        conflictStatus: statusFilter,
      });
      setItems(data.items || []);
    } catch (err) {
      console.error("Eldob failed", err);
    }
  };

  return (
    <div className="p-4">
      <h2 className="text-xl font-bold mb-4">📦 Review Product Groups</h2>
      <div className="flex gap-4 mb-4">
        <input
          type="text"
          placeholder="Import batch ID"
          value={batchId}
          onChange={(e) => setBatchId(e.target.value)}
          className="border px-2 py-1"
        />
        <select
          value={statusFilter}
          onChange={(e) => setStatusFilter(e.target.value)}
          className="border px-2 py-1">
          <option value="">All</option>
          <option value="conflict">Conflict</option>
          <option value="new">New</option>
          <option value="update">Update</option>
        </select>
      </div>
      <table className="w-full table-auto border text-sm">
        <thead>
          <tr className="bg-gray-100">
            <th className="p-2 border">Title</th>
            <th className="p-2 border">Conflicts</th>
            <th className="p-2 border">Live vs Staging</th>
            <th className="p-2 border">Actions</th>
          </tr>
        </thead>
        <tbody>
          {items.map((item) => {
            const conflicts = item.conflictFields || [];

            return (
              <tr key={item._id} className="align-top">
                <td className="p-2 border">
                  {item.productGroupTitle?.value || "(nincs)"}
                </td>
                <td className="p-2 border space-y-1">
                  {conflicts.map((key) => {
                    const diff = item.conflictMeta?.differences?.[key];
                    let icon = "❔";
                    if (
                      diff?.staging &&
                      !diff?.live &&
                      !diff?.added &&
                      !diff?.removed
                    ) {
                      icon = "➕";
                    } else if (
                      diff?.live &&
                      !diff?.staging &&
                      !diff?.added &&
                      !diff?.removed
                    ) {
                      icon = "➖";
                    } else if (diff?.added || diff?.removed) {
                      icon = "🔁";
                    } else if (
                      typeof diff?.live === "string" &&
                      typeof diff?.staging === "string" &&
                      diff.live.trim().toLowerCase() !==
                        diff.staging.trim().toLowerCase()
                    ) {
                      icon = "🔁";
                    }

                    return (
                      <div key={key} className="flex items-center gap-2">
                        <span>{icon}</span>
                        <span className="text-gray-700 font-medium">
                          {translateKey(key)}
                        </span>
                      </div>
                    );
                  })}
                </td>
                <td className="p-2 border space-y-3">
                  {conflicts.map((key) => {
                    if (key === "productFeaturedImageMeta") {
                      const liveImg =
                        item.conflictMeta?.liveMatchId
                          ?.productFeaturedImage?.[0];
                      const stagingImg = item.productFeaturedImage?.[0];
                      const stagingMeta = item.productFeaturedImageMeta || {};

                      return (
                        <details open key={key} className="space-y-1">
                          <summary className="font-semibold text-gray-600 mb-1">
                            {translateKey(key)}
                          </summary>
                          <div className="flex gap-4">
                            <div className="text-center text-xs flex flex-col justify-start items-center">
                              <div className="text-gray-500">Live</div>
                              {liveImg ? (
                                <>
                                  <Image
                                    src={getImageSrc(liveImg)}
                                    width={80}
                                    height={80}
                                    alt="live"
                                    className="w-16 h-16 object-contain border rounded"
                                  />
                                  <p className="text-gray-600 text-xs mt-1">
                                    Fájlnév: <span>{liveImg.originalname}</span>
                                  </p>
                                </>
                              ) : (
                                <Image
                                  src="/images/no-image-gray.png"
                                  width={80}
                                  height={80}
                                  alt="no-live"
                                  className="w-16 h-16 object-contain border rounded"
                                />
                              )}
                            </div>
                            <div className="text-center text-xs flex flex-col justify-start items-center">
                              <div className="text-blue-500">Staging</div>
                              {stagingImg ? (
                                <>
                                  <Image
                                    src={getImageSrc(stagingImg)}
                                    width={80}
                                    height={80}
                                    alt="staging"
                                    className="w-16 h-16 object-contain border rounded"
                                  />
                                  <p className="text-gray-600 text-xs mt-1">
                                    Fájlnév:{" "}
                                    <span>{stagingImg.originalname}</span>
                                  </p>
                                </>
                              ) : (
                                <Image
                                  src="/images/no-image-gray.png"
                                  width={80}
                                  height={80}
                                  alt="no-staging"
                                  className="w-16 h-16 object-contain border rounded"
                                />
                              )}
                              <div className="text-gray-500 flex flex-col mt-2">
                                <p>
                                  Kép kód:{" "}
                                  <span>{stagingMeta.usedCode || "—"}</span>
                                </p>
                                <p>
                                  Módszer:{" "}
                                  <span>
                                    {stagingMeta.selectionMethod || "—"}
                                  </span>
                                </p>
                              </div>
                            </div>
                          </div>
                        </details>
                      );
                    }

                    const liveRaw = getLiveValue(item, key);
                    const stagingRaw = getStagingValue(item, key);
                    const liveDisplay = getReadableValue(liveRaw, key);
                    const stagingDisplay = getReadableValue(stagingRaw, key);
                    const shouldHighlight =
                      typeof liveDisplay === "string" &&
                      typeof stagingDisplay === "string";

                    return (
                      <details open key={key}>
                        <summary className="font-semibold text-gray-600 mb-1">
                          {translateKey(key)}
                        </summary>
                        <div className="mb-1">
                          <span className="text-gray-500">Live:</span>{" "}
                          <span className="bg-gray-100 px-2 py-1 rounded inline-block max-w-[300px] break-words">
                            {liveDisplay}
                          </span>
                        </div>
                        <div>
                          <span className="text-blue-500">Staging:</span>{" "}
                          <span
                            className={`${getDiffClass(
                              liveDisplay,
                              stagingDisplay
                            )} inline-block max-w-[300px] break-words`}>
                            {shouldHighlight
                              ? highlightDiff(liveDisplay, stagingDisplay)
                              : stagingDisplay}
                          </span>
                        </div>
                      </details>
                    );
                  })}
                </td>
                <td className="p-2 border">
                  <button
                    onClick={() => handleApprove(item._id)}
                    className="bg-green-500 text-white px-2 py-1 rounded mr-2">
                    Elfogad
                  </button>
                  <button
                    onClick={() => handleDiscard(item._id)}
                    className="bg-red-500 text-white px-2 py-1 rounded">
                    Eldob
                  </button>
                </td>
              </tr>
            );
          })}
        </tbody>
      </table>
    </div>
  );
}
