import React, { useCallback, useEffect, useState, useMemo } from "react";
import SuneditorComponent from "@/components/shared/texteditor/SuneditorComponent";
import { useForm } from "@/hooks/custom-form-hook";
import {
  VALIDATOR_REQUIRE,
  VALIDATOR_MINLENGTH,
  VALIDATOR_MAXLENGTH,
  VALIDATOR_NUMBER,
} from "@/utils/validators";
import Input from "@/components/shared/input/Input";
import Select from "react-select";
import useProductItemManager from "@/store/managers/useProductItemManager";
import icons from "@/assets/icons/icons";
import ImageWithFallback from "@/components/shared/image/ImageWithFallback";
import ApiCall from "@/api/api-calls";
import MediaGalleryModal from "@/components/media-store/GalleryModal/MediaGalleryModal";
import PageBoxList from "@/components/editor/page-box/PageBoxList";
import { useModalManager } from "@/hooks/useModalManager";
import Modal from "@/components/shared/modal/Modal";
import { usePageEditorContextManager } from "@/context/PageEditorContext";
import { initialData } from "@/data/data";
import toast from "react-hot-toast";
import { normalizeCheckboxes } from "@/utils/normalizeCheckboxes";
import useDynamicManager from "@/store/managers/useDynamicManager";
import useContainerActions from "@/hooks/useContainerActions";
import ClearFix from "material-ui/internal/ClearFix";
import Container from "@/components/editor/container/Container";
import { useEditorGallery } from "@/hooks/useEditorGallery";
import { useExchangeRate } from "@/context/ExchangeRateContext";
const Product = ({
  product,
  productGroupOptions,
  priceTypeList,
  productColorsList,
  productSizesList,
  productMarksList,
  productTagList,
  onProductGroupUpdated,
  parentId,
  handleRemoveLocalProduct,
}) => {
  const [formState, handleInput] = useForm([], true);
  const [textContent, setTextContent] = useState(null);
  const [statePopulated, setStatePopulated] = useState(false);
  const [mediaStoreOpen, setMediaStoreOpen] = useState(false);
  const [showContainerModal, setShowContainerModal] = useState(false);
  const { toggleModal, isModalOpen } = useModalManager();
  const { boxColumnsList } = usePageEditorContextManager();
  const isNew = product?._id?.startsWith("new-");
  const exchangeRate = useExchangeRate();
  const [checkboxes, setCheckboxes] = useState([
    { title: "Publikus", name: "isActive", checked: true },
    { title: "Raktáron", name: "inStock", checked: true },
    { title: "Új termék", name: "isLatest", checked: false },
    { title: "Kiemelt termék", name: "isEmphasized", checked: false },
  ]);

  const { handleAddContainer } = useContainerActions(
    initialData,
    toggleModal,
    "productItem"
  );
  const {
    selectImageModal,
    editImageModal,
    editedImage,

    // Setters
    setSelectImageModal,
    setEditImageModal,
    setEditedImage,

    // Handlers
    handleEditSingleImage,
    handleRemoveSingleImage,
    handleCloneGalleryImage,
    handleUpdateGalleryImageOrder,
    handleGallerySubmit,
  } = useEditorGallery({
    toggleModal,
    modalName: "gallery",
    type: "productItem",
  });

  const {
    state,
    setFeaturedImage,
    resetFeaturedImageState,
    setSize,
    setColor,
    setTag,
    loadProductInitialData,
    setContainerTitle,
    addContainer,
    setContainerList,
    setSelectedGalleryImages,
    resetItemState,
    setProductParentGroup,
  } = useDynamicManager("productItem");

  const handleSetSelectedGalleryImages = useCallback((images) => {
    const ret = [];
    images?.map((image) => ret.push({ image: image, options: {} }));
    setSelectedGalleryImages({ selectedImages: ret });
  }, []);

  useEffect(() => {
    if (!product) return;

    const ret = {
      data: product,
      productProperties: {
        productTagList,
        productSizesList,
        productColorsList,
        productGroupOptions,
      },
    };

    loadProductInitialData(ret); // handles tags, size, color, etc.

    // setProductContainerList({ data: product.productContent }); // handles containers
    setStatePopulated(true);
  }, [
    product,
    productTagList,
    productSizesList,
    productColorsList,
    productGroupOptions,
  ]);

  useEffect(() => {
    if (product) {
      setTextContent(product?.productDescription);
      const featured = Array.isArray(product?.productFeaturedImage)
        ? product?.productFeaturedImage[0]
        : product?.productFeaturedImage;
      if (featured) {
        setFeaturedImage({ image: featured });
      }
    }
  }, [product]);

  const handleSetProductFeaturedImage = (selectedImage) => {
    setFeaturedImage({ image: selectedImage });
    setMediaStoreOpen(false);
  };

  const handleRemoveFeaturedImage = () => {
    resetFeaturedImageState();
  };

  const handleRemoveProduct = useCallback(() => {
    if (isNew) {
      handleRemoveLocalProduct?.(product._id); // remove from UI only
      toast.success("Új termék törölve.");
      onProductGroupUpdated?.();
      return;
    }

    const sendDelete = async () => {
      const deleteResponse = await ApiCall.products.deleteProduct([
        product._id,
      ]);
      if (deleteResponse.status === 200) {
        toast.success(
          deleteResponse.data.message || "Termék sikeresen eltávolítva!"
        );
        onProductGroupUpdated?.();
      } else {
        toast.error(
          deleteResponse.data.error || "Termék eltávolítása sikertelen."
        );
      }
    };

    sendDelete();
  }, [product, isNew]);
  console.log("state.productParentGroup", state.productParentGroup);
  const handleUpdateProduct = useCallback(() => {
    const cleanedInputs = normalizeCheckboxes(formState.inputs, [
      "isActive",
      "inStock",
      "isLatest",
      "isEmphasized",
    ]);
    console.log("cleanedInputs", cleanedInputs);
    const ret = {
      ...state,
      ...cleanedInputs,
      productDescription: textContent,
      productParentGroup: state?.productParentGroup?.id
        ? state?.productParentGroup?.id
        : state?.productParentGroup || parentId,
      productPieceWebshopPrice: cleanedInputs.productPieceWebshopPrice,
      customPrice: cleanedInputs.overridePrice.isChecked || false,
      productDiscount: cleanedInputs.productDiscount,
      customDiscount: cleanedInputs.overrideDiscount.isChecked || false,
    };
    console.log("ret", ret);
    const sendData = async () => {
      const response = isNew
        ? await ApiCall.products.createProduct(parentId, ret)
        : await ApiCall.products.postEdit(product._id, ret);

      if (response.status === 200) {
        toast.success(response.data.message);
        resetItemState();
        onProductGroupUpdated?.();
      } else {
        toast.error(response.data.message);
      }
    };

    sendData();
  }, [product, formState?.inputs, textContent, state]);

  return (
    <div className="w-full grid grid-cols-1 md:grid-cols-4 gap-4 my-4">
      {product && statePopulated && state && (
        <>
          {/* left */}
          <div className="col-span-3" key={1}>
            <div className="form-group flex flex-col my-2 w-full">
              <label className="font-bold text-xs text-gray-400">
                Termék áthelyezése másik termékcsoportba
              </label>
              <Select
                options={productGroupOptions}
                className="w-full"
                menuPosition="fixed"
                placeholder="Válasszon új termékcsoportot"
                name="productParentGroup"
                id="productParentGroup"
                defaultValue={state?.productParentGroup}
                onChange={(val) => {
                  console.log("val", val);
                  setProductParentGroup({ productParentGroup: val.id });
                  handleInput("productParentGroup", val.value, true);
                }}
              />
            </div>
            {/* Termék azonosító */}
            <div className="form-group flex flex-col my-2 w-full">
              <Input
                element="input"
                type="text"
                name="productItemNumber"
                id="productItemNumber"
                placeholder="Termék azonosító"
                onInput={handleInput}
                initialValid={true}
                inputcls="outline-blue-500"
                value={product?.productItemNumber?.value}
              />
            </div>
            {/* Termék EAN */}
            <div className="form-group flex flex-col my-2 w-full">
              <Input
                element="input"
                type="text"
                name="productEAN"
                id="productEAN"
                placeholder="Termék EAN"
                onInput={handleInput}
                initialValid={true}
                inputcls="outline-blue-500"
                value={product?.productEAN?.value}
              />
            </div>
            <div className="w-full grid grid-cols-1 md:grid-cols-2 gap-2">
              {/* Override Price */}
              <div className="form-group flex items-center my-2">
                <Input
                  element="checkbox"
                  type="checkbox"
                  name={"overridePrice"}
                  id={"overridePrice"}
                  placeholder={
                    "Ár kézi beállítása (ne szinkronizálja csoportból)"
                  }
                  onInput={handleInput}
                  initialValid={true}
                  className="justify-center items-center"
                  inputcls="outline-blue-500 w-8 h-8"
                  isChecked={state["priceOverridden"]}
                />
              </div>
              {/* Termék ár */}
              <div className="form-group flex flex-col my-2 w-full">
                <Input
                  element="input"
                  type="text"
                  name="productPieceWebshopPrice"
                  id="productPieceWebshopPrice"
                  placeholder="Termék Ár"
                  onInput={handleInput}
                  initialValid={true}
                  inputcls="outline-blue-500"
                  value={product?.productPieceWebshopPrice?.value}
                />
              </div>
              {/* Termék ár típus*/}
              <div className="form-group flex flex-col my-2 w-full">
                <Input
                  element="input"
                  type="text"
                  name="productPriceType"
                  id="productPriceType"
                  placeholder="Termék Ár típus"
                  onInput={handleInput}
                  initialValid={true}
                  inputcls="outline-blue-500"
                  value={product?.productPriceType?.value}
                  readOnly
                />
              </div>
              {/* Termék ár átváltva*/}
              <div className="form-group flex flex-col my-2 w-full">
                <Input
                  element="input"
                  type="text"
                  placeholder="Termék Ár átváltva"
                  initialValid={true}
                  onInput={handleInput}
                  inputcls="outline-blue-500"
                  value={
                    product?.productPriceType?.value === "EUR"
                      ? product?.productPieceWebshopPrice?.value *
                          exchangeRate +
                        " HUF"
                      : product?.productPieceWebshopPrice?.value /
                          exchangeRate +
                        " EUR"
                  }
                  readOnly
                />
              </div>
              {/* Jelenlegi árfolyam */}
              <div className="form-group flex flex-col my-2 w-full">
                <Input
                  element="input"
                  type="text"
                  placeholder="Jelenlegi árfolyam"
                  initialValid={true}
                  onInput={handleInput}
                  inputcls="outline-blue-500"
                  value={exchangeRate}
                  readOnly
                />
              </div>
            </div>

            {/* Termék méret */}
            <div className="w-full grid grid-cols-1 md:grid-cols-2 gap-2">
              <div className="form-group flex flex-col my-2 w-full justify-center">
                <label
                  htmlFor="productSizesList"
                  className="font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                  Válasszon termék méreteket
                </label>
                <Select
                  options={productSizesList}
                  className="w-full"
                  menuPosition={"fixed"}
                  placeholder={" Válasszon termék méretet"}
                  backspaceRemovesValue={false}
                  name={"productSizesList"}
                  id={"productSizesList"}
                  defaultValue={state?.size} // default selected
                  onChange={(val, e) => {
                    setSize({ size: val });
                  }}
                  required
                />
              </div>
              {/* Termék szín */}
              <div className="form-group flex flex-col my-2 w-full justify-center">
                <label
                  htmlFor="productColorsList"
                  className="font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                  Válasszon termék színeket
                </label>
                <Select
                  options={productColorsList}
                  className="w-full"
                  menuPosition={"fixed"}
                  placeholder={"Válasszon termék színt"}
                  backspaceRemovesValue={false}
                  name={"productColorsList"}
                  id={"productColorsList"}
                  defaultValue={state.color} // default selected
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    setColor({ color: val });
                  }}
                  required
                />
              </div>
            </div>
            <div className="w-full grid grid-cols-1 md:grid-cols-2 gap-2">
              {/* Termék súly */}
              <div className="form-group flex flex-col my-2 max-lg:w-full w-full">
                <Input
                  element="input"
                  type="text"
                  name="productWeight"
                  id="productWeight"
                  onInput={handleInput}
                  validators={[VALIDATOR_NUMBER()]}
                  errorText="Kérjük, tizedesvessző helyett pontot használjon (pl. 3.5)"
                  initialValid={true}
                  placeholder="Termék súly (kg/db)"
                  label="Termék súly (kg/db)"
                  inputcls="outline-blue-500"
                  value={product?.productWeight?.value}
                />
              </div>

              {/* Termék címke */}
              <div className="form-group flex flex-col my-2 max-lg:w-full w-full">
                <label
                  htmlFor="productTagList"
                  className="font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                  Válasszon termék címkét
                </label>
                <Select
                  options={productTagList}
                  className="w-full"
                  menuPosition={"fixed"}
                  placeholder={"Válasszon termék címkét"}
                  backspaceRemovesValue={false}
                  name={"productTagList"}
                  isMulti
                  defaultValue={state.tags} // default selected
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    setTag({ tags: val });
                  }}
                  required
                />
              </div>
            </div>
            {/* Termék leírás */}
            <div className="form-group flex flex-col my-2 w-full !z-0">
              <label
                htmlFor={"productDescription"}
                className="w-1/4  text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                <p className="w-full font-bold text-xs">Termék leírás: </p>
              </label>
              <SuneditorComponent
                lang={"en"}
                name="productDescription"
                id="productDescription"
                className="w-full h-auto !z-0"
                defaultValue={textContent}
                placeholder={"Termék leírás"}
                onChange={(value) => setTextContent(value)}
                options={{
                  buttonList: [
                    ["font", "fontSize", "formatBlock"],
                    [
                      "bold",
                      "underline",
                      "italic",
                      "strike",
                      "subscript",
                      "superscript",
                    ],
                    ["align", "list", "table"],
                    ["fontColor", "hiliteColor"],
                    ["outdent", "indent"],
                    ["undo", "redo"],
                    ["removeFormat"],
                    ["link", "image"],
                    ["preview", "print"],
                    ["fullScreen", "showBlocks", "codeView"],
                  ],
                  tableWidth: "100%",
                  tableStyles: {
                    border: "1px solid black",
                  },
                }}
              />
            </div>
            <button
              onClick={() => toggleModal("containerModal")}
              className="w-full flex flex-row justify-center items-center bg-blue-500/75 hover:bg-blue-500 text-white px-4 py-2">
              Konténer hozzáadása
            </button>
            {/* box content */}
            {state?.containerData?.map((c, idx) => (
              <Container
                key={idx}
                data={c}
                alldata={state?.containerData}
                parent={idx}
                handleTitleChange={state.setContainerTitle}
                type={"productItem"}
              />
            ))}
          </div>
          {/* right */}
          <div className="col-span-1">
            <div className="w-full flex flex-col">
              {/* Featured Image */}
              <div className="w-full h-80 border-2 border-dashed relative mb-4">
                {state.featuredImage?.length > 0 && (
                  <ImageWithFallback
                    src={ApiCall.mediaStore.getResized(
                      "/4/4/1",
                      state.featuredImage[0].originalname
                    )}
                    alt="Featured"
                    fill
                    className="object-cover"
                  />
                )}
                <span
                  onClick={() => setMediaStoreOpen(true)}
                  className="absolute top-2 left-2 w-[3rem] h-[3rem] hover:border-1 bg-white rounded-full flex justify-center items-center cursor-pointer">
                  {icons.change}
                </span>
                <span
                  onClick={handleRemoveFeaturedImage}
                  className="absolute top-2 right-2 w-[2rem] h-[2rem] !fill-red !stroke-red rounded-full flex justify-center items-center cursor-pointer">
                  {icons.clear}
                </span>
              </div>
              {/* checkboxes */}
              <div className="w-full flex flex-row flex-wrap justify-center items-center">
                {checkboxes?.map((checkbox, index) => (
                  <div
                    key={checkbox.name}
                    className="form-group w-1/2 flex flex-col my-4 justify-center items-center">
                    <Input
                      element="checkbox"
                      type="checkbox"
                      name={checkbox.name}
                      id={checkbox.name}
                      placeholder={checkbox.title}
                      onInput={handleInput}
                      initialValid={true}
                      className="justify-center items-center"
                      inputcls="outline-blue-500 w-8 h-8"
                      isChecked={state[checkbox.name]}
                    />
                  </div>
                ))}
              </div>
              {/* Override Discount */}
              <div className="form-group flex items-center my-2">
                <Input
                  element="checkbox"
                  type="checkbox"
                  name={"overrideDiscount"}
                  id={"overrideDiscount"}
                  placeholder={
                    "Kedvezmény kézi beállítása (ne szinkronizálja csoportból)"
                  }
                  onInput={handleInput}
                  initialValid={true}
                  className="justify-center items-center"
                  inputcls="outline-blue-500 w-8 h-8"
                  isChecked={state["discountOverridden"]}
                />
              </div>
              {/* <div className="form-group flex items-center my-2">
                <input
                  type="checkbox"
                  id="overrideDiscount"
                  checked={formState.inputs.overrideDiscount?.value ?? false}
                  onChange={(e) =>
                    handleInput({
                      target: {
                        name: "overrideDiscount",
                        value: e.target.checked,
                      },
                    })
                  }
                />
                <label htmlFor="overrideDiscount" className="ml-2 text-sm">
                  Kedvezmény kézi beállítása (ne szinkronizálja csoportból)
                </label>
              </div> */}
              {/* Termék kedvezmény % */}
              <div className="form-group flex flex-col my-2 max-lg:w-full w-full ">
                <Input
                  element="input"
                  type="number"
                  name="productDiscount"
                  id="productDiscount"
                  validators={[
                    VALIDATOR_NUMBER(),
                    VALIDATOR_MINLENGTH(1),
                    VALIDATOR_MAXLENGTH(254),
                  ]}
                  onInput={handleInput}
                  errorText={"Csak számot adhat meg."}
                  initialValid={true}
                  placeholder="Termék kedvezmény %"
                  inputcls="outline-blue-500"
                  value={product?.productDiscount?.value}
                />
              </div>
              {/* Termék raktár mennyiség */}
              <div className="form-group flex flex-col my-2 max-lg:w-full w-full">
                <Input
                  element="input"
                  type="number"
                  name="stock"
                  id="stock"
                  validators={[
                    VALIDATOR_NUMBER(),
                    VALIDATOR_MINLENGTH(1),
                    VALIDATOR_MAXLENGTH(254),
                  ]}
                  onInput={handleInput}
                  errorText={"Csak számot adhat meg."}
                  initialValid={true}
                  placeholder="Termék db raktáron"
                  inputcls="outline-blue-500"
                  value={product?.stock?.value || 0}
                />
              </div>
              {/* Áfa érték */}
              <div className="form-group flex flex-col my-2 max-lg:w-full w-full">
                <Input
                  element="input"
                  type="text"
                  name="productVat"
                  id="productVat"
                  placeholder="Áfa érték %"
                  onInput={handleInput}
                  initialValid={true}
                  inputcls="outline-blue-500"
                  value={product?.productVat?.value ?? 27}
                  readOnly={
                    formState.inputs &&
                    formState?.inputs?.productPieceNettoBuyPrice?.value > 0
                      ? false
                      : true
                  }
                />
              </div>
              <button
                className="w-full px-4 py-2 bg-red-500/75 hover:bg-red-500 text-white"
                onClick={() => toggleModal("deleteProductConsent")}>
                {" "}
                Termék törlése
              </button>
            </div>
          </div>
          <div className="w-full col-span-4">
            {formState.isValid && (
              <button
                onClick={() => handleUpdateProduct()}
                className="w-full px-4 py-2 flex flex-row justify-center items-center bg-green-500/75 hover:bg-green-500 text-white">
                {isNew ? "Mentés új termékként" : "Frissítés"}
              </button>
            )}
          </div>
        </>
      )}
      {/* featured image */}
      <MediaGalleryModal
        isOpen={selectImageModal}
        onClose={() => setSelectImageModal(false)}
        action="productGalleryImages"
        onConfirmSelection={(selectedImage) =>
          handleSetSelectedGalleryImages(selectedImage)
        }
      />
      <MediaGalleryModal
        isOpen={mediaStoreOpen}
        onClose={() => setMediaStoreOpen(false)}
        action="productFeatured"
        onConfirmSelection={(selectedImage) =>
          handleSetProductFeaturedImage(selectedImage)
        }
      />
      {/* add new container */}
      <Modal
        title={"Válasszon az alábbi elemek közül"}
        show={isModalOpen("containerModal")}
        onClose={() => toggleModal("containerModal")}
        height={"100%"}>
        <div className="grid grid-cols-6 gap-4 my-6">
          {boxColumnsList?.columnList?.map((c, i) => (
            <div
              key={i}
              className="border border-dashed border-gray-400 hover:border-gray-700 p-2 cursor-pointer"
              onClick={() => {
                handleAddContainer(c);
              }}>
              <div className="grid w-full grid-cols-1 sm:grid-cols-2 md:grid-cols-12 gap-1">
                {c?.columnSize?.map((col, idx) => {
                  const colX = parseInt(col.x, 10);
                  const colY = parseInt(col.y, 10);
                  const colSpan = Math.round((colX / colY) * 12); // Tailwind's 12-col grid

                  return (
                    <div
                      key={idx}
                      className={`bg-gray-500 text-white text-xs grid justify-center items-center h-20 gap-1 col-span-${colSpan}`}>
                      {colX}/{colY}
                    </div>
                  );
                })}
              </div>
            </div>
          ))}
        </div>
      </Modal>
      {/* delete consent */}
      <Modal
        title={`Biztosan szeretné eltávolítani ezt a konténert? <span style="width: 100%">A művelet nem vonható vissza!</span>`}
        show={isModalOpen("deleteProductConsent")}
        onClose={() => toggleModal("deleteProductConsent")}
        width="max-content">
        <div className="flex flex-row flex-wrap justify-evenly h-max">
          <button
            className="px-4 py-2 border border-gray-200 text-white hover:bg-red-500/50 bg-red-500"
            onClick={() => {
              handleRemoveProduct(); // 👈 smart handler handles both new + saved
              toggleModal("deleteProductConsent");
            }}>
            Törlés
          </button>
          <button
            className="px-4 py-2 border border-gray-200 text-black hover:bg-gray-100/50 bg-gray-100"
            onClick={() => toggleModal("deleteProductConsent")}>
            Mégsem
          </button>
        </div>
      </Modal>
    </div>
  );
};

export default Product;
