import React, { useEffect, useState } from "react";
import { toast } from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import ProductGroupTabs from "./ProductGroupTabs";
import ProductGroup from "./ProductGroup";
import ProductList from "./product/ProductList";
import Loader from "@/components/shared/loader/global-loading";
import VersionListPage from "./ProductGroupVersions";
import useDynamicManager from "@/store/managers/useDynamicManager";
import { ExchangeRateProvider } from "@/context/ExchangeRateContext";

const ExpandedProductGroups = ({ id, image, token, onProductGroupUpdated }) => {
  const [product, setProduct] = useState(null);
  const [activeTab, setActiveTab] = useState(1);
  const [loading, setLoading] = useState(false);
  const { resetFeaturedImageState } = useDynamicManager("productGroup");
  // get initial product data
  useEffect(() => {
    setLoading(true);
    if (!id) return;
    resetFeaturedImageState();
    const fetchProduct = async () => {
      try {
        const productElement = await ApiCall.productGroups.getEdit(id);

        if (productElement.status === 200) {
          setProduct(productElement);
          toast.success(
            "Termékcsoport sikeresen betöltve: " +
              productElement?.data?.data?.productGroupTitle.value
          );
        }
      } catch (error) {
        console.error("Error fetching product data:", error);
        toast.error("Hiba a termékcsoport betöltésekor: " + error.message);
      } finally {
        setLoading(false);
      }
    };
    fetchProduct();
  }, [id, token]);

  if (loading) return <Loader />;

  return (
    <ExchangeRateProvider>
      <div className="w-full my-4">
        <ProductGroupTabs
          activeTab={activeTab}
          onTabChange={setActiveTab}
          productCnt={product?.data?.data?.productList?.length}
        />
        {activeTab === 1 && product && (
          <ProductGroup
            product={product}
            image={image}
            onProductGroupUpdated={onProductGroupUpdated}
          />
        )}
        {activeTab === 2 && product && (
          <ProductList
            parentId={product?.data?.data?._id}
            productGroup={product?.data?.data}
            productIds={product?.data?.data?.productList}
            onProductGroupUpdated={onProductGroupUpdated}
          />
        )}
        {activeTab === 3 && product && (
          <VersionListPage
            id={product?.data?.data?._id}
            onProductGroupUpdated={onProductGroupUpdated}
          />
        )}
      </div>
    </ExchangeRateProvider>
  );
};

export default ExpandedProductGroups;
