// src/components/phone-registry/PhoneRegistryForm.tsx
"use client";

import { useEffect, useMemo, useState, useCallback } from "react";
import AsyncSelect from "react-select/async";
import ApiCall from "@/api/api-calls";

export type PhoneUpsertPayload = {
  sort_order?: number | null;
  phone_number: string;
  group?: string | null; // Csoport (segment)
  handler?: string | null; // Person
  status?: string | null;
  pause_start?: string | Date | null;
  pause_end?: string | Date | null;
  display_name?: string | null;
  issued_at?: string | Date | null;

  // 👇 legacy free text still kept; BE resolves location_text into either short_code or department id
  location_text?: string | null;

  device_brand?: string | null;
  device_model?: string | null;
  device_serial?: string | null;
  imei?: string[] | null;
  sim_pins?: string[] | null;
  sim_puks?: string[] | null;
  sim_card_serial?: string | null;

  // these are strings (codes/names); BE normalizes to FKs
  data_plan?: string | null;
  current_cost_center?: string | null;
  cost_bearer?: string | null;
  future_cost_center?: string | null;

  notes?: string | null;
};

type Opt = { value: string; label: string };
type GroupedOpt = { label: string; options: Opt[] };

export default function PhoneRegistryForm({
  initial,
  onSubmit,
  saving,
}: {
  initial?: Partial<PhoneUpsertPayload>;
  onSubmit: (payload: PhoneUpsertPayload) => void | Promise<void>;
  saving?: boolean;
}) {
  // --- controlled selects for location / cost centers / internet / people / status / segments ---
  const [locationSel, setLocationSel] = useState<Opt | null>(null);
  const [currentCcSel, setCurrentCcSel] = useState<Opt | null>(null);
  const [costBearerSel, setCostBearerSel] = useState<Opt | null>(null);
  const [futureCcSel, setFutureCcSel] = useState<Opt | null>(null);
  const [internetSel, setInternetSel] = useState<Opt | null>(null);

  const [handlerSel, setHandlerSel] = useState<Opt | null>(null); // person
  const [segmentSel, setSegmentSel] = useState<Opt | null>(null); // Csoport
  const [statusSel, setStatusSel] = useState<Opt | null>(null);

  // --- basic fields ---
  const [sortOrder, setSortOrder] = useState<number | "">(
    initial?.sort_order ?? ""
  );
  const [phone, setPhone] = useState(initial?.phone_number ?? "");

  // raw status as fallback (if select not used)
  const [status, setStatus] = useState(initial?.status ?? "");

  const [pauseStart, setPauseStart] = useState<string | "">(
    initial?.pause_start ? toDateInput(initial?.pause_start) : ""
  );
  const [pauseEnd, setPauseEnd] = useState<string | "">(
    initial?.pause_end ? toDateInput(initial?.pause_end) : ""
  );
  const [displayName, setDisplayName] = useState(initial?.display_name ?? "");
  const [issuedAt, setIssuedAt] = useState<string | "">(
    initial?.issued_at ? toDateInput(initial?.issued_at) : ""
  );

  const [locationText, setLocationText] = useState(
    initial?.location_text ?? ""
  ); // legacy text (kept)

  const [deviceBrand, setDeviceBrand] = useState(initial?.device_brand ?? "");
  const [deviceModel, setDeviceModel] = useState(initial?.device_model ?? "");
  const [deviceSerial, setDeviceSerial] = useState(
    initial?.device_serial ?? ""
  );
  const [imeiText, setImeiText] = useState((initial?.imei ?? []).join(", "));
  const [simPinsText, setSimPinsText] = useState(
    (initial?.sim_pins ?? []).join(", ")
  );
  const [simPuksText, setSimPuksText] = useState(
    (initial?.sim_puks ?? []).join(", ")
  );
  const [simCardSerial, setSimCardSerial] = useState(
    initial?.sim_card_serial ?? ""
  );

  const [dataPlan, setDataPlan] = useState(initial?.data_plan ?? ""); // legacy string
  const [currentCC, setCurrentCC] = useState(
    initial?.current_cost_center ?? ""
  );
  const [costBearer, setCostBearer] = useState(initial?.cost_bearer ?? "");
  const [futureCC, setFutureCC] = useState(initial?.future_cost_center ?? "");
  const [notes, setNotes] = useState(initial?.notes ?? "");

  // --- pre-hydrate selects from initial values ---
  useEffect(() => {
    // Location: prefer whatever text we have
    const loc = (initial?.location_text ?? "").trim();
    setLocationSel(loc ? { value: loc, label: loc } : null);

    setCurrentCcSel(currentCC ? { value: currentCC, label: currentCC } : null);
    setCostBearerSel(
      costBearer ? { value: costBearer, label: costBearer } : null
    );
    setFutureCcSel(futureCC ? { value: futureCC, label: futureCC } : null);
    setInternetSel(dataPlan ? { value: dataPlan, label: dataPlan } : null);
  }, [initial, currentCC, costBearer, futureCC, dataPlan]);

  // hydrate from initial row once for handler, segment & status
  useEffect(() => {
    const h = (initial?.handler ?? "").trim();
    setHandlerSel(h ? { value: h, label: h } : null);

    const g = (initial?.group ?? "").trim();
    setSegmentSel(g ? { value: g, label: g } : null);

    const s = (initial?.status ?? "").trim();
    setStatusSel(s ? { value: s, label: s } : null);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  // --- loaders for AsyncSelect ---

  const loadShortCodes = useCallback(async (inputValue: string) => {
    const r = await ApiCall.phoneRegistry.shortCodeOptions(
      inputValue || "",
      50
    );
    return (r.data?.items ?? []).map((it: any) => ({
      value: it.code,
      label: it.description ? `${it.code} — ${it.description}` : it.code,
    })) as Opt[];
  }, []);

  const loadDepartments = useCallback(async (inputValue: string) => {
    const r = await ApiCall.phoneRegistry.departmentOptions(
      inputValue || "",
      50
    );
    return (r.data?.items ?? []).map((it: any) => ({
      value: it.code || it.name,
      label: it.code ? `${it.code} — ${it.name}` : it.name,
    })) as Opt[];
  }, []);

  const loadInternetPackages = useCallback(async (inputValue: string) => {
    const r = await ApiCall.phoneRegistry.internetPackageOptions(
      inputValue || "",
      50
    );
    return (r.data?.items ?? []).map((it: any) => ({
      value: it.name,
      label: it.name,
    })) as Opt[];
  }, []);

  // Grouped loader for Location (short-codes + departments)
  const loadLocationOptions = useCallback(
    async (inputValue: string) => {
      const [sc, dep] = await Promise.all([
        loadShortCodes(inputValue),
        loadDepartments(inputValue),
      ]);
      const groups: GroupedOpt[] = [
        { label: "Rövid kódok", options: sc },
        { label: "Osztályok", options: dep },
      ];
      return groups as any;
    },
    [loadShortCodes, loadDepartments]
  );

  // People (handler)
  const loadPeople = useCallback(async (inputValue: string) => {
    const r = await ApiCall.phoneRegistry.peopleOptions(inputValue || "", 50);
    return (r.data?.items ?? []).map((it: any) => ({
      value: it.value ?? it.name ?? it.display_name,
      label: it.label ?? it.display_name ?? it.name,
    })) as Opt[];
  }, []);

  // Segments (Csoport)
  const loadSegments = useCallback(async (inputValue: string) => {
    const r = await ApiCall.phoneRegistry.companySegmentOptions(
      inputValue || "",
      50
    );
    return (r.data?.items ?? []).map((it: any) => ({
      value: it.value ?? it.name,
      label: it.label ?? it.name,
    })) as Opt[];
  }, []);

  // Status options
  const loadStatuses = useCallback(async (inputValue: string) => {
    const r = await ApiCall.phoneRegistry.statusOptions(inputValue || "", 50);
    return (r.data?.items ?? []).map((it: any) => ({
      value: it.value ?? it.name,
      label: it.label ?? it.name,
    })) as Opt[];
  }, []);

  const canSubmit = useMemo(() => /\d{3,}/.test(phone), [phone]);

  const submit = async (e: React.FormEvent) => {
    e.preventDefault();
    const payload: PhoneUpsertPayload = {
      sort_order: sortOrder === "" ? null : Number(sortOrder),
      phone_number: phone,

      pause_start: pauseStart ? new Date(pauseStart) : null,
      pause_end: pauseEnd ? new Date(pauseEnd) : null,
      display_name: displayName || null,
      issued_at: issuedAt ? new Date(issuedAt) : null,

      handler: handlerSel?.value || null,
      group: segmentSel?.value || null,
      status: statusSel?.value || status || null,

      // keep raw text but prefer selected value if present
      location_text: locationSel?.value || locationText || null,

      device_brand: deviceBrand || null,
      device_model: deviceModel || null,
      device_serial: deviceSerial || null,
      imei: splitList(imeiText),
      sim_pins: splitList(simPinsText),
      sim_puks: splitList(simPuksText),
      sim_card_serial: simCardSerial || null,

      // prefer selects; fall back to legacy text
      data_plan: internetSel?.value || dataPlan || null,
      current_cost_center: currentCcSel?.value || currentCC || null,
      cost_bearer: costBearerSel?.value || costBearer || null,
      future_cost_center: futureCcSel?.value || futureCC || null,

      notes: notes || null,
    };
    await onSubmit(payload);
  };

  return (
    <form
      onSubmit={submit}
      className="grid grid-cols-1 md:grid-cols-2 gap-3"
      autoComplete="off">
      {/* --- Basic identification --- */}
      {/* <div className="flex flex-col">
        <label className="text-xs text-gray-500">Sorszám</label>
        <input
          type="number"
          className="border rounded px-3 py-2"
          value={sortOrder}
          onChange={(e) =>
            setSortOrder(e.target.value === "" ? "" : Number(e.target.value))
          }
        />
      </div> */}
      {/* phone number */}
      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Telefonszám</label>
        <input
          className="border rounded px-3 py-2"
          value={phone}
          onChange={(e) => setPhone(e.target.value)}
          required
        />
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Telefon neve</label>
        <input
          className="border rounded px-3 py-2"
          value={displayName}
          onChange={(e) => setDisplayName(e.target.value)}
          placeholder="pl. Ügyelet, Rendelő 1"
        />
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Kiadás dátuma</label>
        <input
          type="date"
          className="border rounded px-3 py-2"
          value={issuedAt}
          onChange={(e) => setIssuedAt(e.target.value)}
        />
      </div>

      {/* --- Pause dates --- */}
      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Szüneteltetés kezdete</label>
        <input
          type="date"
          className="border rounded px-3 py-2"
          value={pauseStart}
          onChange={(e) => setPauseStart(e.target.value)}
        />
      </div>
      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Szüneteltetés lejárata</label>
        <input
          type="date"
          className="border rounded px-3 py-2"
          value={pauseEnd}
          onChange={(e) => setPauseEnd(e.target.value)}
        />
      </div>

      {/* --- Handler / Segment / Status --- */}
      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Személy</label>
        <AsyncSelect
          cacheOptions
          defaultOptions
          loadOptions={loadPeople}
          value={handlerSel}
          onChange={(v) => setHandlerSel(v as Opt)}
          placeholder="Válassz személyt…"
          isClearable
        />
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Csoport</label>
        <AsyncSelect
          cacheOptions
          defaultOptions
          loadOptions={loadSegments}
          value={segmentSel}
          onChange={(v) => setSegmentSel(v as Opt)}
          placeholder="Válassz csoportot…"
          isClearable
        />
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Státusz</label>
        <AsyncSelect
          cacheOptions
          defaultOptions
          loadOptions={loadStatuses}
          value={statusSel}
          onChange={(v) => setStatusSel(v as Opt)}
          placeholder="Válassz státuszt…"
          isClearable
        />
        {/* Optional raw text fallback for custom status */}
        {/* <input
          className="border rounded px-3 py-2 mt-2"
          value={status}
          onChange={(e) => setStatus(e.target.value)}
          placeholder="Szabad szöveg (új státusz esetén)"
        /> */}
      </div>

      {/* --- Location --- */}
      <div className="flex flex-col md:col-span-2">
        <label className="text-xs text-gray-500">Hol</label>
        <AsyncSelect
          cacheOptions
          defaultOptions
          loadOptions={loadLocationOptions}
          value={locationSel}
          onChange={(v) => setLocationSel(v as Opt)}
          placeholder="Válassz helyszínt (rövid kód vagy osztály)…"
          isClearable
        />
        {/* Optional raw text fallback / display */}
        {/* <input
          className="border rounded px-3 py-2 mt-2"
          value={locationText ?? ""}
          onChange={(e) => setLocationText(e.target.value)}
          placeholder="Szabad szöveg (pl. Budapest, központ)"
        /> */}
      </div>

      {/* --- Device info --- */}
      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Készülék márka</label>
        <input
          className="border rounded px-3 py-2"
          value={deviceBrand}
          onChange={(e) => setDeviceBrand(e.target.value)}
        />
      </div>
      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Készülék modell</label>
        <input
          className="border rounded px-3 py-2"
          value={deviceModel}
          onChange={(e) => setDeviceModel(e.target.value)}
        />
      </div>
      <div className="flex flex-col md:col-span-2">
        <label className="text-xs text-gray-500">
          Készülék gyári szám (serial)
        </label>
        <input
          className="border rounded px-3 py-2"
          value={deviceSerial}
          onChange={(e) => setDeviceSerial(e.target.value)}
        />
      </div>

      {/* --- SIM / IMEI --- */}
      <div className="flex flex-col md:col-span-2">
        <label className="text-xs text-gray-500">IMEI(k)</label>
        <input
          className="border rounded px-3 py-2"
          value={imeiText}
          onChange={(e) => setImeiText(e.target.value)}
          placeholder="Több érték: vesszővel elválasztva"
        />
      </div>
      <div className="flex flex-col">
        <label className="text-xs text-gray-500">SIM PIN1/2</label>
        <input
          className="border rounded px-3 py-2"
          value={simPinsText}
          onChange={(e) => setSimPinsText(e.target.value)}
          placeholder="Több érték: vesszővel elválasztva"
        />
      </div>
      <div className="flex flex-col">
        <label className="text-xs text-gray-500">PUK1/PUK2</label>
        <input
          className="border rounded px-3 py-2"
          value={simPuksText}
          onChange={(e) => setSimPuksText(e.target.value)}
          placeholder="Több érték: vesszővel elválasztva"
        />
      </div>
      <div className="flex flex-col md:col-span-2">
        <label className="text-xs text-gray-500">SIM kártyaszám</label>
        <input
          className="border rounded px-3 py-2"
          value={simCardSerial}
          onChange={(e) => setSimCardSerial(e.target.value)}
        />
      </div>

      {/* --- Internet + cost centers --- */}
      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Internet csomag</label>
        <AsyncSelect
          cacheOptions
          defaultOptions
          loadOptions={loadInternetPackages}
          value={internetSel}
          onChange={(v) => setInternetSel(v as Opt)}
          placeholder="Válassz internet csomagot…"
          isClearable
        />
        {/* <input
          className="border rounded px-3 py-2 mt-2"
          value={dataPlan}
          onChange={(e) => setDataPlan(e.target.value)}
          placeholder="Szabad szöveg / új csomag"
        /> */}
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Jelenlegi költséghely</label>
        <AsyncSelect
          cacheOptions
          defaultOptions
          loadOptions={loadShortCodes}
          value={currentCcSel}
          onChange={(v) => setCurrentCcSel(v as Opt)}
          placeholder="Válassz rövid kódot…"
          isClearable
        />
        {/* <input
          className="border rounded px-3 py-2 mt-2"
          value={currentCC}
          onChange={(e) => setCurrentCC(e.target.value)}
          placeholder="Szabad szöveg / új kód"
        /> */}
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Költségviselő</label>
        <AsyncSelect
          cacheOptions
          defaultOptions
          loadOptions={loadShortCodes}
          value={costBearerSel}
          onChange={(v) => setCostBearerSel(v as Opt)}
          placeholder="Válassz rövid kódot…"
          isClearable
        />
        {/* <input
          className="border rounded px-3 py-2 mt-2"
          value={costBearer}
          onChange={(e) => setCostBearer(e.target.value)}
          placeholder="Szabad szöveg / új kód"
        /> */}
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Jövőbeni költséghely</label>
        <AsyncSelect
          cacheOptions
          defaultOptions
          loadOptions={loadShortCodes}
          value={futureCcSel}
          onChange={(v) => setFutureCcSel(v as Opt)}
          placeholder="Válassz rövid kódot…"
          isClearable
        />
        {/* <input
          className="border rounded px-3 py-2 mt-2"
          value={futureCC}
          onChange={(e) => setFutureCC(e.target.value)}
          placeholder="Szabad szöveg / új kód"
        /> */}
      </div>

      {/* --- Notes --- */}
      <div className="flex flex-col md:col-span-2">
        <label className="text-xs text-gray-500">Megjegyzés</label>
        <textarea
          className="border rounded px-3 py-2 min-h-[80px]"
          value={notes}
          onChange={(e) => setNotes(e.target.value)}
        />
      </div>

      {/* --- Save --- */}
      <div className="md:col-span-2 flex justify-end gap-2">
        <button
          type="submit"
          disabled={!canSubmit || !!saving}
          className="px-4 py-2 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white disabled:opacity-50">
          {saving ? "Mentés…" : "Mentés"}
        </button>
      </div>
    </form>
  );
}

// helpers (unchanged)
function splitList(s: string) {
  const arr = String(s ?? "")
    .split(/[,\s;]+/)
    .map((v) => v.trim())
    .filter(Boolean);
  return arr.length ? arr : null;
}

function toDateInput(d: string | Date | null | undefined) {
  if (!d) return "";
  const dt = typeof d === "string" ? new Date(d) : d;
  const y = dt.getFullYear();
  const m = String(dt.getMonth() + 1).padStart(2, "0");
  const da = String(dt.getDate()).padStart(2, "0");
  return `${y}-${m}-${da}`;
}
