"use client";
import { useState, useEffect, useCallback } from "react";
import { useRouter } from "next/navigation";
import useInfiniteFetch from "@/hooks/useInfiniteFetch";
import ApiCall from "@/api/api-calls";
import { wrapPaginationFetcher } from "@/lib/data-utils";
import useDebounce from "@/hooks/useDebounce";
import EditMediaModal from "@/components/media-store/EditMediaModal/EditMediaModal";
import Filters from "@/components/media-store/Filters/Filters";
import ImageGrid from "@/components/media-store/ImageGrid/ImageGrid";
import UploadPanel from "@/components/media-store/UploadPanel/UploadPanel";
import TabSwitcher from "@/components/shared/tabswitcher/TabSwitcher";
import { toast } from "react-hot-toast";
import { Tab } from "@headlessui/react";

const MediaGallery = ({
  mode = "picker",
  action = "productFeatured",
  onSelect,
  onConfirmSelection,
  onClose,
}) => {
  const router = useRouter();
  const [activeTab, setActiveTab] = useState(1);
  const [searchQuery, setSearchQuery] = useState("");
  const [itemsPerPage, setItemsPerPage] = useState(36);
  const [sortBy, setSortBy] = useState("createdAt");
  const [sortOrder, setSortOrder] = useState("desc");
  const [dateFrom, setDateFrom] = useState("");
  const [dateTo, setDateTo] = useState("");
  const [editingImage, setEditingImage] = useState(null);
  const [showEditModal, setShowEditModal] = useState(false);
  const debouncedSearch = useDebounce(searchQuery, 300);
  const [shouldReload, setShouldReload] = useState(false);
  const [selectedItems, setSelectedItems] = useState([]);

  const isSingleSelect =
    action === "productFeatured" ||
    action === "productGroupFeatured" ||
    action === "productCategoryFeatured" ||
    action === "productBrandFeatured" ||
    action === "separatorImage" ||
    action === "pageEditorImage" ||
    action === "container-image" ||
    action === "editorImage";

  const fetchImages = useCallback(
    wrapPaginationFetcher(async ({ page, limit }) => {
      const res = await ApiCall.mediaStore.getAll({
        page,
        limit,
        search: debouncedSearch,
        sortBy,
        sortOrder,
        dateFrom,
        dateTo,
      });

      return {
        items: res.data?.images || [],
        total: res.data?.total || 0,
      };
    }),
    [debouncedSearch, sortBy, sortOrder, dateFrom, dateTo]
  );

  const {
    items: images,
    loadNext,
    loading,
    hasMore,
    reset,
  } = useInfiniteFetch({
    fetchItems: ({ page }) => fetchImages({ page, limit: itemsPerPage }),
    onSessionExpired: () => {
      alert("Session expired. Redirecting to login...");
      router.push("/");
    },
  });

  useEffect(() => {
    setShouldReload(true);
    reset();
  }, [debouncedSearch, sortBy, sortOrder, dateFrom, dateTo, itemsPerPage]);

  useEffect(() => {
    if (shouldReload && images.length === 0) {
      loadNext();
      setShouldReload(false);
    }
  }, [shouldReload, images]);

  const handleResetFilters = () => {
    setSearchQuery("");
    setItemsPerPage(20);
    setSortBy("created_at");
    setSortOrder("desc");
    setDateFrom("");
    setDateTo("");
    setShouldReload(true);
    reset();
  };

  const handleEdit = (image) => {
    setEditingImage(image);
    setShowEditModal(true);
  };

  const handleSaveEdit = async (formData) => {
    try {
      await ApiCall.mediaStore.editImage(editingImage.id, formData);
      toast.success("Media updated");
      setShouldReload(true);
      reset();
    } catch (err) {
      console.error(err);
      toast.error("Failed to update media");
    }
  };

  const handleDelete = async (image) => {
    const confirm = window.confirm(`Delete ${image.original_name}?`);

    if (!confirm) return;
    try {
      await ApiCall.mediaStore.deleteImage([image.id]);
      toast.success("Media deleted");
      setSelectedItems([]);
      setShouldReload(true);
      reset();
    } catch (err) {
      console.error(err);
      toast.error("Failed to delete media");
    }
  };

  const handleDeleteSelected = async () => {
    if (!selectedItems.length) return toast.error("No files selected");

    const confirm = window.confirm(
      `Are you sure you want to delete ${selectedItems.length} file(s)?`
    );
    if (!confirm) return;

    try {
      const ids = selectedItems.map((item) => item.id);
      await ApiCall.mediaStore.deleteImage(ids);
      toast.success("Selected media deleted");
      setSelectedItems([]);
      setShouldReload(true);
      reset();
    } catch (err) {
      console.error(err);
      toast.error("Failed to delete selected media");
    }
  };

  const toggleSelect = (item) => {
    if (isSingleSelect) {
      setSelectedItems([item]);
    } else {
      const already = selectedItems.find((i) => i.id === item.id);
      const updated = already
        ? selectedItems.filter((i) => i.id !== item.id)
        : [...selectedItems, item];
      setSelectedItems(updated);
    }
  };

  const handleConfirmSelection = () => {
    console.log("selectedItems", selectedItems);
    if (onConfirmSelection) {
      onConfirmSelection(isSingleSelect ? selectedItems[0] : selectedItems);
    }
    if (onClose) onClose(); // Optional: auto-close modal
  };

  const handleDownloadSelected = async () => {
    if (!selectedItems.length) return toast.error("No files selected");
    try {
      const res = await fetch(
        `${process.env.SERVER_URL}/api/admin/media-store/download-zip`,
        {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify({
            ids: selectedItems.map((item) => item.id),
          }),
        }
      );

      if (!res.ok) throw new Error("Failed to generate ZIP");

      const blob = await res.blob();
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = "media-files.zip";
      document.body.appendChild(a);
      a.click();
      a.remove();
    } catch (err) {
      console.error(err);
      toast.error("ZIP download failed");
    }
  };
  const tabData = [
    { id: 1, tabTitle: "Médiatár" },
    { id: 2, tabTitle: "Feltöltés" },
  ];
  console.log("first", activeTab);
  return (
    <div className="px-4 py-6">
      <h2 className="text-2xl font-semibold mb-4">
        {mode === "picker" ? "Fájlok kiválasztása" : "Médiatár"}
      </h2>

      <TabSwitcher
        data={tabData}
        activeTab={activeTab}
        onTabClick={setActiveTab}
      />

      {activeTab === 1 && (
        <>
          <Filters
            searchQuery={searchQuery}
            setSearchQuery={setSearchQuery}
            sortBy={sortBy}
            setSortBy={setSortBy}
            sortOrder={sortOrder}
            setSortOrder={setSortOrder}
            itemsPerPage={itemsPerPage}
            setItemsPerPage={setItemsPerPage}
            dateFrom={dateFrom}
            setDateFrom={setDateFrom}
            dateTo={dateTo}
            setDateTo={setDateTo}
            handleResetFilters={handleResetFilters}
          />
          {mode === "picker" && (
            <div className="my-6 flex justify-end gap-3">
              <button
                className="px-4 py-2 bg-green-600 text-white rounded hover:bg-green-700"
                onClick={handleConfirmSelection}
                disabled={selectedItems.length === 0}>
                {isSingleSelect ? "Elem kiválasztása" : "Elemek kiválasztása"} (
                {selectedItems.length})
              </button>
              {selectedItems.length > 0 && (
                <>
                  <button
                    className="px-4 py-2 disabled:bg-gray-500 bg-red-600 text-white rounded hover:bg-red-700"
                    onClick={handleDeleteSelected}
                    disabled={selectedItems.length === 0}>
                    Törlés ({selectedItems.length})
                  </button>
                  <button
                    className="px-4 py-2 disabled:bg-gray-500 bg-blue-500 text-white rounded hover:bg-blue-700"
                    onClick={handleDownloadSelected}
                    disabled={selectedItems.length === 0}>
                    Kijelöltek letöltése
                  </button>
                </>
              )}
            </div>
          )}

          <ImageGrid
            images={images}
            selectedItems={mode === "picker" ? selectedItems : []}
            onSelectItem={mode === "picker" ? toggleSelect : undefined}
            onEdit={handleEdit}
            onDelete={handleDelete}
          />

          {loading && (
            <p className="mt-6 text-gray-500">
              További elemek betöltése folyamatban...
            </p>
          )}
          {hasMore && !loading && (
            <div className="text-center mt-6">
              <button
                onClick={loadNext}
                className="px-4 py-2 bg-indigo-600 text-white rounded hover:bg-indigo-700">
                Továbbiak betöltése
              </button>
            </div>
          )}
        </>
      )}

      {activeTab === 2 && (
        <UploadPanel
          onUploadSuccess={() => {
            setActiveTab("Médiatár");
            setShouldReload(true);
            reset();
          }}
        />
      )}

      <EditMediaModal
        image={editingImage}
        isOpen={showEditModal}
        onClose={() => setShowEditModal(false)}
        onSave={handleSaveEdit}
      />
    </div>
  );
};

export default MediaGallery;
