import Image from "next/image";
import ApiCall from "@/api/api-calls";
import { Download, Pencil, Trash2 } from "lucide-react";

const ImageGrid = ({
  images,
  selectedItems = [],
  onSelectItem,
  onEdit,
  onDelete,
}) => {
  const isSelected = (image) =>
    selectedItems?.some((item) => item.id === image.id);

  return (
    <div className="grid grid-cols-4 sm:grid-cols-4 md:grid-cols-4 lg:grid-cols-4 gap-4 ">
      {images.map((image, idx) => {
        return (
          <div
            key={idx}
            className={`relative group border rounded overflow-hidden cursor-pointer border border-gray-200 hover:border-green-600 hover:shadow-xl ${
              isSelected(image) ? "ring-2 ring-green-600" : "hover:shadow-md"
            }`}
            onClick={() => onSelectItem?.(image)}>
            {image.props.mimetype === "application/pdf" ? (
              <>{image.original_name}</>
            ) : (
              <div className="flex flex-col">
                <img
                  src={ApiCall.mediaStore.getResized(
                    "/300/200/0",
                    image.original_name,
                    { mode: "pixel" }
                  )}
                  width={100}
                  height={100}
                  alt={image?.original_name}
                  className="rounded w-full h-full aspect-square max-h-[200px]"
                  objectFit="cover"
                />
                {/* <p>{image.original_name}</p> */}
              </div>
            )}

            {/* Selection indicator */}
            {isSelected(image) && (
              <div className="absolute top-1 left-1 bg-green-600 text-white text-xs px-2 py-1 rounded">
                Kijelölve
              </div>
            )}

            {/* Filename */}
            <div className="mt-1 text-sm truncate text-center">
              {image.original_name}
            </div>

            {/* Hover Edit/Delete buttons */}
            <div
              className="absolute top-1 right-1 flex gap-1 opacity-0 group-hover:opacity-100 transition"
              onClick={(e) => e.stopPropagation()} // Prevent click propagation to selection
            >
              {/* ✅ Download link */}
              {image.props?.dataPath && (
                <a
                  href={`${
                    process.env.SERVER_URL
                  }/download-file/${image.props.dataPath.replace(
                    "/uploads/",
                    ""
                  )}`} // removes "/uploads/"
                  download
                  onClick={(e) => e.stopPropagation()}
                  className=" bg-blue-600 text-white text-xs px-2 py-1 rounded hover:bg-blue-700">
                  <Download className="w-4 h-4 fill-white" />
                </a>
              )}
              {onEdit && (
                <button
                  className="bg-yellow-400 text-black text-xs px-2 py-1 rounded"
                  onClick={() => onEdit(image)}>
                  <Pencil className="w-4 h-4 fill-white" />
                </button>
              )}
              {onDelete && (
                <button
                  className="bg-red-600 text-white text-xs px-2 py-1 rounded"
                  onClick={() => onDelete(image)}>
                  <Trash2 className="w-4 h-4 fill-white" />
                </button>
              )}
            </div>
          </div>
        );
      })}
    </div>
  );
};

export default ImageGrid;
