// src/components/layout/TopNavBar.tsx
"use client";

import React, { useEffect, useRef, useState } from "react";
import { useRouter } from "next/navigation";
import { LogOut, Bell, FileText, MessageSquareDot } from "lucide-react";

import Button from "@/components/shared/button/Button";
import { useAuth } from "@/context/AuthContext";
import { getTranslation } from "@/i18n";
import SocketStatusBanner from "@/components/features/socket-status/SocketStatusBanner";
import { useMyApprovals } from "@/hooks/useMyApprovalsBadge";
import { useContext } from "react";
import { ChatUnreadContext } from "@/context/SocketProvider";
import ChatUnreadBadge from "@/components/layout/ChatUnreadBadge";

const TopNavBar = () => {
  const { user, isAuthenticated, loading, logout } = useAuth();
  const { t } = getTranslation();
  const router = useRouter();
  const { unread, setUnread } = useContext(ChatUnreadContext);
  const { pendingCount, items, loadingList, loadList, refreshSummary } =
    useMyApprovals();

  const [dropdownOpen, setDropdownOpen] = useState(false);
  const bellRef = useRef<HTMLButtonElement | null>(null);
  const dropdownRef = useRef<HTMLDivElement | null>(null);

  const hasPending = pendingCount > 0;

  const handleLogout = async () => {
    try {
      logout();
    } catch (error) {
      console.error("Error during logout:", error);
    }
  };

  const toggleDropdown = async () => {
    const next = !dropdownOpen;
    setDropdownOpen(next);
    if (next) {
      await Promise.all([loadList(), refreshSummary()]);
    }
  };

  const handleGoToDoc = (id: string) => {
    setDropdownOpen(false);
    router.push(`/admin/hr/person-documents/${id}`);
  };

  const handleGoToMyApprovals = () => {
    setDropdownOpen(false);
    router.push("/admin/hr/my-approvals");
  };

  // Close dropdown on outside click
  useEffect(() => {
    if (!dropdownOpen) return;
    const handleClick = (e: MouseEvent) => {
      const target = e.target as Node;
      if (
        dropdownRef.current &&
        !dropdownRef.current.contains(target) &&
        bellRef.current &&
        !bellRef.current.contains(target)
      ) {
        setDropdownOpen(false);
      }
    };
    window.addEventListener("mousedown", handleClick);
    return () => window.removeEventListener("mousedown", handleClick);
  }, [dropdownOpen]);

  return (
    <header className="h-12 bg-white shadow-md flex items-center justify-between px-6 border-b border-gray-200">
      {/* left side: logo / title if you want */}

      <div className="flex items-center ml-auto space-x-4 relative">
        <SocketStatusBanner />

        {isAuthenticated && (
          <div className="relative">
            {/* <ChatUnreadBadge /> */}
            <button
              ref={bellRef}
              type="button"
              onClick={toggleDropdown}
              className="relative inline-flex items-center justify-center p-1 rounded-full hover:bg-gray-100 focus:outline-none focus:ring-2 focus:ring-emerald-500"
              title="Jóváhagyásra váró dokumentumok">
              <FileText className="w-5 h-5 text-gray-600" />
              {hasPending && (
                <span className="absolute -top-1 -right-1 inline-flex items-center justify-center px-1.5 py-0.5 rounded-full bg-red-500 text-[10px] font-semibold text-white">
                  {pendingCount > 9 ? "9+" : pendingCount}
                </span>
              )}
            </button>

            {/* Dropdown panel */}
            {dropdownOpen && (
              <div
                ref={dropdownRef}
                className="absolute right-0 mt-2 w-96 bg-white border border-gray-200 rounded-md shadow-lg z-40">
                <div className="px-3 py-2 border-b flex items-center justify-between">
                  <span className="text-xs font-semibold text-gray-700">
                    Jóváhagyásra váró dokumentumok
                  </span>
                  {hasPending && (
                    <span className="text-[11px] text-gray-500">
                      Összes: {pendingCount}
                    </span>
                  )}
                </div>

                <div className="max-h-80 overflow-y-auto">
                  {loadingList ? (
                    <div className="px-3 py-4 text-xs text-gray-500">
                      Betöltés…
                    </div>
                  ) : items.length === 0 ? (
                    <div className="px-3 py-4 text-xs text-gray-500">
                      Jelenleg nincs jóváhagyásra váró dokumentumod.
                    </div>
                  ) : (
                    <ul className="divide-y divide-gray-100">
                      {items.slice(0, 5).map((item) => (
                        <li
                          key={item.approver_id + item.document_id}
                          className="px-3 py-2 text-xs hover:bg-gray-50">
                          <div className="flex justify-between gap-2">
                            <div className="flex flex-col">
                              <span className="font-semibold text-gray-800 truncate">
                                {item.title}
                              </span>
                              {item.person_name && (
                                <span className="text-[11px] text-gray-500">
                                  Dolgozó: {item.person_name}
                                </span>
                              )}
                              {item.approver_role && (
                                <span className="text-[11px] text-gray-400">
                                  Szerep: {item.approver_role}
                                </span>
                              )}
                            </div>
                            <button
                              type="button"
                              onClick={() => handleGoToDoc(item.document_id)}
                              className="self-start text-[11px] text-emerald-700 hover:underline">
                              Megnyitás
                            </button>
                          </div>
                        </li>
                      ))}
                    </ul>
                  )}
                </div>

                <div className="px-3 py-2 border-t flex justify-between items-center">
                  <button
                    type="button"
                    onClick={handleGoToMyApprovals}
                    className="text-[11px] text-emerald-700 hover:underline">
                    Összes megnyitása
                  </button>
                  {items.length > 5 && (
                    <span className="text-[11px] text-gray-500">
                      + {items.length - 5} további
                    </span>
                  )}
                </div>
              </div>
            )}
          </div>
        )}

        <p className="text-xs">
          {t("logged_in")}:
          <span className="text-sm text-gray-600 font-semibold ml-1">
            {isAuthenticated && !loading && user?.username}
          </span>
        </p>

        <Button
          type="button"
          variant="danger"
          icononly
          rounded="full"
          size="sm"
          aria-label="Logout"
          onClick={handleLogout}
          leftIcon={
            <LogOut
              className="w-5 h-5 stroke-white fill-red-600"
              aria-hidden="true"
            />
          }
        />
      </div>
    </header>
  );
};

export default TopNavBar;
