// src/components/layout/Sidebar.tsx
"use client";

import { useEffect, useState } from "react";
import Link from "next/link";
import { usePathname } from "next/navigation";
import PerfectScrollbar from "react-perfect-scrollbar";
import AnimateHeight from "react-animate-height";
import { useDispatch, useSelector } from "react-redux";
import { IRootState } from "@/store/store";
import { toggleSidebar } from "@/store/slices/themeConfigSlice";
import clsx from "clsx";
import { useAuth } from "@/context/AuthContext";
import { getTranslation } from "@/i18n";

import IconCaretsDown from "@/components/icon/icon-carets-down";
import IconCaretDown from "@/components/icon/icon-caret-down";
import IconMenuDashboard from "@/components/icon/menu/icon-menu-dashboard";

import RenderMenuItem from "./RenderMenuItem";
import { useAdminMenu } from "@/hooks/useAdminMenu";

const Sidebar = () => {
  const dispatch = useDispatch();
  const pathname = usePathname();
  const { t } = getTranslation();
  const { isAuthenticated, loading } = useAuth();

  const themeConfig = useSelector((state: IRootState) => state.themeConfig);
  const semidark = themeConfig.semidark;
  const isSidebarOpen = themeConfig.sidebar;

  const [currentMenu, setCurrentMenu] = useState<string>("");

  // ✅ shared admin menu from context
  const { menuData } = useAdminMenu();

  const toggleMenu = (value: string) => {
    setCurrentMenu((prev) => (prev === value ? "" : value));
  };

  // highlight current route + auto-collapse on small screens
  useEffect(() => {
    const active = document.querySelectorAll(".sidebar ul a.active");
    active.forEach((el) => el.classList.remove("active"));
    const selector = document.querySelector(
      `.sidebar ul a[href="${window.location.pathname}"]`
    );
    selector?.classList.add("active");

    if (window.innerWidth < 1024 && themeConfig.sidebar) {
      dispatch(toggleSidebar());
    }
  }, [pathname, dispatch, themeConfig.sidebar]);

  if (loading || !isAuthenticated) return null;

  return (
    <div className={semidark ? "dark" : ""}>
      <nav
        aria-label="Sidebar"
        className={clsx(
          "sidebar fixed bottom-0 top-0 z-50 h-full min-h-screen shadow-[5px_0_25px_0_rgba(94,92,154,0.1)] transition-all duration-300",
          "overflow-hidden", // ensure no spill when collapsed
          semidark ? "text-white-dark" : "",
          isSidebarOpen ? "w-[260px]" : "w-[72px]"
        )}>
        <div className="h-full bg-white dark:bg-black">
          {/* 🔹 Header (logo + collapse button) */}
          <div className="flex items-center justify-between px-4 py-3">
            <Link
              href="/admin/dashboard"
              className="main-logo flex shrink-0 items-center">
              {/* collapse width, not just opacity */}
              <span className="material-icons text-blue-500">dashboard</span>
              <span
                className={clsx(
                  "align-middle text-2xl font-semibold ltr:ml-1.5 rtl:mr-1.5 dark:text-white-light transition-all duration-200 whitespace-nowrap overflow-hidden",
                  isSidebarOpen
                    ? "max-w-[160px] opacity-100"
                    : "max-w-0 opacity-0 pointer-events-none"
                )}>
                PortableCMS
              </span>
            </Link>

            <button
              type="button"
              className="collapse-icon flex h-8 w-8 items-center cursor-pointer rounded-full transition duration-300 hover:bg-gray-500/10 rtl:rotate-180 dark:text-white-light dark:hover:bg-dark-light/10"
              onClick={() => dispatch(toggleSidebar())}
              aria-label="Toggle sidebar"
              title={isSidebarOpen ? t("collapse") : t("expand")}>
              <IconCaretsDown
                className={clsx(
                  "m-auto",
                  isSidebarOpen ? "rotate-90" : "-rotate-90"
                )}
              />
            </button>
          </div>

          {/* 🔹 Scrollable menu */}
          <PerfectScrollbar className="relative h-[calc(100vh-80px)] overflow-y-auto">
            <ul className="relative space-y-0.5 p-4 py-0 font-semibold ">
              {/* 🔹 Static dashboard group (same as old code) */}
              <li className="menu nav-item">
                <button
                  type="button"
                  title={t("dashboard")}
                  className={clsx(
                    "nav-link group w-full flex items-center justify-between no-before-icon cursor-pointer hover:bg-blue-100 mb-[0.25rem] p-[0.625rem]",
                    "overflow-hidden", // row-level clipping
                    currentMenu === "dashboard" && isSidebarOpen
                      ? "bg-gray-100"
                      : ""
                  )}
                  onClick={() => toggleMenu("dashboard")}
                  aria-expanded={
                    isSidebarOpen ? currentMenu === "dashboard" : undefined
                  }>
                  <div className="flex items-center min-w-0">
                    <IconMenuDashboard className="shrink-0 group-hover:!text-primary" />
                    <span
                      className={clsx(
                        "transition-all duration-200 origin-left text-[1rem] whitespace-nowrap overflow-hidden",
                        isSidebarOpen
                          ? "ml-3 max-w-[170px] opacity-100"
                          : "ml-0 max-w-0 opacity-0 pointer-events-none"
                      )}>
                      {t("dashboard")}
                    </span>
                  </div>

                  {isSidebarOpen && (
                    <div
                      className={
                        currentMenu !== "dashboard"
                          ? "-rotate-90 rtl:rotate-90"
                          : ""
                      }
                      aria-hidden>
                      <IconCaretDown />
                    </div>
                  )}
                </button>

                <AnimateHeight
                  duration={300}
                  height={
                    isSidebarOpen && currentMenu === "dashboard" ? "auto" : 0
                  }>
                  <ul className="sub-menu text-gray-500">
                    {/* add dashboard sub-links here if needed */}
                    {/* <li>
                      <Link href="/admin/dashboard" className="nav-link">
                        Fő mutatók
                      </Link>
                    </li> */}
                  </ul>
                </AnimateHeight>
              </li>

              {/* 🔹 Dynamic tree from AdminMenuProvider */}
              <li className="nav-item">
                <ul>
                  {menuData.map((item) => (
                    <RenderMenuItem
                      key={item.id}
                      item={item}
                      level={0}
                      isSidebarOpen={isSidebarOpen}
                      currentPath={pathname}
                    />
                  ))}
                </ul>
              </li>
            </ul>
          </PerfectScrollbar>
        </div>
      </nav>
    </div>
  );
};

export default Sidebar;

// "use client";

// import { useEffect, useState } from "react";
// import Link from "next/link";
// import { usePathname } from "next/navigation";
// import PerfectScrollbar from "react-perfect-scrollbar";
// import AnimateHeight from "react-animate-height";
// import { useDispatch, useSelector } from "react-redux";
// import { IRootState } from "@/store/store";
// import { toggleSidebar } from "@/store/slices/themeConfigSlice";
// import clsx from "clsx";
// import { toast } from "react-hot-toast";
// import ApiCall from "@/api/api-calls";
// import { useAuth } from "@/context/AuthContext";
// import { getTranslation } from "@/i18n";

// import IconCaretsDown from "@/components/icon/icon-carets-down";
// import IconCaretDown from "@/components/icon/icon-caret-down";
// import IconMenuDashboard from "@/components/icon/menu/icon-menu-dashboard";

// import RenderMenuItem from "./RenderMenuItem";

// type AdminMenuItem = {
//   id: string | number;
//   adminmenutitle: string;
//   adminmenulink?: string | null;
//   icon?: string | null;
//   children?: AdminMenuItem[];
// };

// const Sidebar = () => {
//   const dispatch = useDispatch();
//   const pathname = usePathname();
//   const { t } = getTranslation();
//   const { isAuthenticated, loading } = useAuth();

//   const themeConfig = useSelector((state: IRootState) => state.themeConfig);
//   const semidark = themeConfig.semidark;
//   const isSidebarOpen = themeConfig.sidebar;

//   const [currentMenu, setCurrentMenu] = useState<string>("");
//   const [menuData, setMenuData] = useState<AdminMenuItem[]>([]);

//   const toggleMenu = (value: string) => {
//     setCurrentMenu((prev) => (prev === value ? "" : value));
//   };

//   useEffect(() => {
//     const fetchNavigation = async () => {
//       try {
//         const response = await ApiCall.adminMenu.getAdminMenuTree();
//         if (response?.status === 200) {
//           setMenuData(response.data?.body || []);
//         } else {
//           toast.error("A menü betöltése nem sikerült");
//         }
//       } catch (err) {
//         console.error("Sidebar fetch error:", err);
//         toast.error("Hiba történt a menü betöltésekor");
//       }
//     };

//     if (isAuthenticated) fetchNavigation();
//   }, [isAuthenticated]);

//   // highlight current route + auto-collapse on small screens
//   useEffect(() => {
//     const active = document.querySelectorAll(".sidebar ul a.active");
//     active.forEach((el) => el.classList.remove("active"));
//     const selector = document.querySelector(
//       `.sidebar ul a[href="${window.location.pathname}"]`
//     );
//     selector?.classList.add("active");

//     if (window.innerWidth < 1024 && themeConfig.sidebar) {
//       dispatch(toggleSidebar());
//     }
//   }, [pathname, dispatch, themeConfig.sidebar]);

//   if (loading || !isAuthenticated) return null;

//   return (
//     <div className={semidark ? "dark" : ""}>
//       <nav
//         aria-label="Sidebar"
//         className={clsx(
//           "sidebar fixed bottom-0 top-0 z-50 h-full min-h-screen shadow-[5px_0_25px_0_rgba(94,92,154,0.1)] transition-all duration-300",
//           "overflow-hidden", // ensure no spill when collapsed
//           semidark ? "text-white-dark" : "",
//           isSidebarOpen ? "w-[260px]" : "w-[72px]"
//         )}>
//         <div className="h-full bg-white dark:bg-black">
//           {/* Header */}
//           <div className="flex items-center justify-between px-4 py-3">
//             <Link
//               href="/admin/dashboard"
//               className="main-logo flex shrink-0 items-center">
//               {/* collapse width, not just opacity */}
//               <span className="material-icons text-blue-500">dashboard</span>
//               <span
//                 className={clsx(
//                   "align-middle text-2xl font-semibold ltr:ml-1.5 rtl:mr-1.5 dark:text-white-light transition-all duration-200 whitespace-nowrap overflow-hidden",
//                   isSidebarOpen
//                     ? "max-w-[160px] opacity-100"
//                     : "max-w-0 opacity-0 pointer-events-none"
//                 )}>
//                 PortableCMS
//               </span>
//             </Link>

//             <button
//               type="button"
//               className="collapse-icon flex h-8 w-8 items-center cursor-pointer rounded-full transition duration-300 hover:bg-gray-500/10 rtl:rotate-180 dark:text-white-light dark:hover:bg-dark-light/10"
//               onClick={() => dispatch(toggleSidebar())}
//               aria-label="Toggle sidebar"
//               title={isSidebarOpen ? t("collapse") : t("expand")}>
//               <IconCaretsDown
//                 className={clsx(
//                   "m-auto",
//                   isSidebarOpen ? "rotate-90" : "-rotate-90"
//                 )}
//               />
//             </button>
//           </div>

//           {/* Scrollable menu */}
//           <PerfectScrollbar className="relative h-[calc(100vh-80px)] overflow-y-auto">
//             <ul className="relative space-y-0.5 p-4 py-0 font-semibold ">
//               {/* Example static group */}
//               <li className="menu nav-item">
//                 <button
//                   type="button"
//                   title={t("dashboard")}
//                   className={clsx(
//                     "nav-link group w-full flex items-center justify-between no-before-icon cursor-pointer hover:bg-blue-100 mb-[0.25rem] p-[0.625rem]",
//                     "overflow-hidden", // row-level clipping
//                     currentMenu === "dashboard" && isSidebarOpen
//                       ? "bg-gray-100"
//                       : ""
//                   )}
//                   onClick={() => toggleMenu("dashboard")}
//                   aria-expanded={
//                     isSidebarOpen ? currentMenu === "dashboard" : undefined
//                   }>
//                   <div className="flex items-center min-w-0">
//                     <IconMenuDashboard className="shrink-0 group-hover:!text-primary" />
//                     <span
//                       className={clsx(
//                         "transition-all duration-200 origin-left text-[1rem] whitespace-nowrap overflow-hidden",
//                         isSidebarOpen
//                           ? "ml-3 max-w-[170px] opacity-100"
//                           : "ml-0 max-w-0 opacity-0 pointer-events-none"
//                       )}>
//                       {t("dashboard")}
//                     </span>
//                   </div>

//                   {isSidebarOpen && (
//                     <div
//                       className={
//                         currentMenu !== "dashboard"
//                           ? "-rotate-90 rtl:rotate-90"
//                           : ""
//                       }
//                       aria-hidden>
//                       <IconCaretDown />
//                     </div>
//                   )}
//                 </button>

//                 <AnimateHeight
//                   duration={300}
//                   height={
//                     isSidebarOpen && currentMenu === "dashboard" ? "auto" : 0
//                   }>
//                   <ul className="sub-menu text-gray-500">
//                     {/* your dashboard links here */}
//                   </ul>
//                 </AnimateHeight>
//               </li>

//               {/* Dynamic tree */}
//               <li className="nav-item">
//                 <ul>
//                   {menuData.map((item) => (
//                     <RenderMenuItem
//                       key={item.id}
//                       item={item}
//                       level={0}
//                       isSidebarOpen={isSidebarOpen}
//                       currentPath={pathname}
//                     />
//                   ))}
//                 </ul>
//               </li>
//             </ul>
//           </PerfectScrollbar>
//         </div>
//       </nav>
//     </div>
//   );
// };

// export default Sidebar;
