import React, { useEffect, useState, useMemo } from "react";
import Input from "@/components/shared/input/Input";
import Select from "react-select";
import { useForm } from "@/hooks/custom-form-hook";
import { VALIDATOR_REQUIRE, VALIDATOR_MINLENGTH } from "@/utils/validators";
import SuneditorComponent from "@/components/shared/texteditor/SuneditorComponent";
import LabeledInput from "@/components/shared/labeledinput/LabeledInput";
import SelectInput from "@/components/shared/selectinput/SelectInput";
import ApiCall from "@/api/api-calls";
import { toast } from "react-hot-toast";
import { convertGetDateString } from "@/utils/converDateString";
import { currencyDisplayHandler } from "@/utils/currencyDisplay";
import Separator from "@/components/shared/separator/Separator";

// Simple collapsible section/card
const Section = ({ title, description, children, defaultOpen = true }) => {
  const [open, setOpen] = useState(defaultOpen);

  return (
    <div className="border rounded-lg bg-white shadow-sm">
      <button
        type="button"
        className="w-full flex items-center justify-between px-3 py-2 border-b text-left"
        onClick={() => setOpen((o) => !o)}>
        <div>
          <div className="font-semibold">{title}</div>
          {description && (
            <div className="text-xs text-gray-500">{description}</div>
          )}
        </div>
        <span className="text-gray-500 text-sm">{open ? "▲" : "▼"}</span>
      </button>
      {open && <div className="p-3 grid gap-3 grid-cols-4">{children}</div>}
    </div>
  );
};

const InventoryElement = ({
  asset,
  references,
  onRowUpdated,
  isCreateMode = false,
}) => {
  const [formState, inputHandler, setFormData] = useForm({}, false);
  const [assetDescription, setAssetDescription] = useState(
    asset.description || ""
  );
  const [extendedFunctions, setExtendedFunctions] = useState(
    asset.extended_functions || ""
  );

  const boolFields = [
    { id: 0, value: true, label: "Dokumentum csatolva." },
    { id: 1, value: false, label: "Nincs dokumentum." },
  ];

  const [selected, setSelected] = useState({
    asset_type_id: asset.asset_type_id,
    buyer_id: asset.buyer_id,
    department_id: asset.department_id,
    placed_at: asset.placed_at,
    site_id: asset.site_id,
    used_at: asset.used_at,
    price_currency: asset.price_currency_id ?? null,
    netto_price_currency: asset.netto_price_currency_id ?? null,
    item_status_id: asset.item_status_id ?? null,
  });

  const eqId = (a, b) => (a?.toString?.() ?? a) === (b?.toString?.() ?? b);

  // Filtered lists
  const filteredDepartments = useMemo(() => {
    const deps = references?.departments ?? [];
    if (!selected.site_id) return deps;

    return deps.filter((d) => eqId(d.site_id, selected.site_id));
  }, [references, selected.site_id]);

  const filteredLocations = useMemo(() => {
    const locs = references?.locations ?? [];
    if (selected.department_id) {
      return locs.filter((l) => eqId(l.department_id, selected.department_id));
    }
    if (selected.site_id) {
      return locs.filter((l) => eqId(l.site_id, selected.site_id));
    }
    return locs;
  }, [references, selected.department_id, selected.site_id]);

  useEffect(() => {
    // if site changed and department no longer valid
    if (
      selected.department_id &&
      !filteredDepartments.some((d) => eqId(d.id, selected.department_id))
    ) {
      setSelected((prev) => ({ ...prev, department_id: null }));
    }
  }, [filteredDepartments, selected.department_id]);

  useEffect(() => {
    // clear placed/used if not in filteredLocations anymore
    const ids = new Set(filteredLocations.map((l) => l.id));
    setSelected((prev) => ({
      ...prev,
      placed_at:
        prev.placed_at && ids.has(prev.placed_at) ? prev.placed_at : null,
      used_at: prev.used_at && ids.has(prev.used_at) ? prev.used_at : null,
    }));
  }, [filteredLocations]);

  const selectOptions = (items) =>
    Array.isArray(items) && items.length && (items[0].id || items[0].value)
      ? items.map((x) => ({ value: x.value ?? x.id, label: x.label ?? x.name }))
      : items || [];

  const handleSelectChange = (field, option) => {
    setSelected((prev) => ({ ...prev, [field]: option?.value || null }));
  };

  // Init form from asset
  useEffect(() => {
    if (asset) {
      const initialData = {
        name: { value: asset.name || "", isValid: true },
        inventory_number: {
          value: asset.inventory_number || "",
          isValid: true,
        },
        invoice_number: { value: asset.invoice_number || "", isValid: true },
        revision_frequency: {
          value: asset.revision_frequency || "",
          isValid: true,
        },
        serial_number: { value: asset.serial_number || "", isValid: true },
        external_id: { value: asset.external_id || "", isValid: true },
        description: { value: asset.description || "", isValid: true },
        extended_functions: {
          value: asset.extended_functions || "",
          isValid: true,
        },
        moved_at_date: { value: asset.moved_at_date || null, isValid: true },
        purchase_date: { value: asset.purchase_date || null, isValid: true },
        revise_date: { value: asset.revise_date || null, isValid: true },
        price: { value: parseFloat(asset.price) || 0, isValid: true },
        netto_price: {
          value: parseFloat(asset.netto_price) || 0,
          isValid: true,
        },
        cost_center: { value: asset.cost_center || "", isValid: true },
        seller_id: { value: asset.seller_id || "", isValid: true },
        assigned_user_id: {
          value: asset.assigned_user_id || "",
          isValid: true,
        },
        device_brand: { value: asset.device_brand || "", isValid: true },
        device_model: { value: asset.device_model || "", isValid: true },
        isLeased: { value: !!asset.is_leased, isValid: true },
        service_phone: { value: asset.service_phone || "", isValid: true },
        contact_person: { value: asset.contact_person || "", isValid: true },
        contact_phone: { value: asset.contact_phone || "", isValid: true },
        install_steps: { value: asset.install_steps || "", isValid: true },
        install_requirements: {
          value: asset.install_requirements || "",
          isValid: true,
        },
        device_isLeased: {
          value: asset.device_is_leased || null,
          isValid: true,
        }, // read-only in UI
      };

      const currentData = formState.inputs;
      const isSame =
        JSON.stringify(initialData) === JSON.stringify(currentData);

      if (!isSame) {
        setFormData(initialData, true);
      }

      setAssetDescription(asset.description || "");
      setExtendedFunctions(asset.extended_functions || "");
    }
  }, [asset]);

  // Default currency to HUF
  useEffect(() => {
    if (!selected.price_currency && references?.currencies?.length) {
      const defaultCurrency = references.currencies.find(
        (c) => c.label === "HUF"
      );
      setSelected((prev) => ({
        ...prev,
        price_currency: defaultCurrency?.value,
        netto_price_currency: defaultCurrency?.value,
      }));
    }
  }, [references]);

  // If department has a site, keep in sync
  useEffect(() => {
    if (
      !selected.site_id &&
      selected.department_id &&
      references?.departments?.length
    ) {
      const dep = references.departments.find(
        (d) => d.id === selected.department_id
      );
      if (dep?.site_id) {
        setSelected((prev) => ({ ...prev, site_id: dep.site_id }));
      }
    }
  }, [selected.site_id, selected.department_id, references?.departments]);

  const handleSave = async () => {
    const rawFormData = Object.fromEntries(
      Object.entries(formState.inputs).map(([k, v]) => [k, v.value])
    );

    const payload = {
      ...rawFormData,
      price: rawFormData.price ? parseFloat(rawFormData.price) : null,
      netto_price: rawFormData.netto_price
        ? parseFloat(rawFormData.netto_price)
        : null,
      asset_type_id: selected.asset_type_id,
      buyer_id: selected.buyer_id,
      department_id: selected.department_id,
      placed_at: selected.placed_at,
      site_id: selected.site_id,
      used_at: selected.used_at,
      extended_functions: extendedFunctions,
      description: assetDescription,
      price_currency_id: selected.price_currency,
      netto_price_currency_id: selected.netto_price_currency,
      isLeased: !!rawFormData.isLeased,
      service_phone: rawFormData.service_phone || "",
      contact_person: rawFormData.contact_person || "",
      contact_phone: rawFormData.contact_phone || "",
      install_steps: rawFormData.install_steps || "",
      install_requirements: rawFormData.install_requirements || "",
      item_status_id: selected.item_status_id || null,
    };

    console.log("formState before save:", formState.inputs);
    console.log("rawFormData:", rawFormData);

    try {
      if (isCreateMode) {
        await ApiCall.inventory.createAsset(payload);
        toast.success("Eszköz létrehozva");
      } else {
        await ApiCall.inventory.updateAsset(asset.id, payload);
        toast.success("Mentés sikeres");
      }
      onRowUpdated?.();
    } catch (err) {
      toast.error(isCreateMode ? "Létrehozás sikertelen" : "Mentés sikertelen");
    }
  };

  const currentStatus =
    (references?.assetItemStatuses || []).find((s) =>
      eqId(s.id, selected.item_status_id)
    ) || null;

  const statusColorChipStyle = currentStatus?.color_rgb
    ? {
        backgroundColor: `rgb(${currentStatus.color_rgb})`,
      }
    : {};
  return (
    <div className="flex flex-col space-y-4">
      {formState?.inputs && (
        <>
          {/* Sticky summary header */}
          <div className="sticky top-0 z-50 bg-gray-50 border-b py-2 px-1 mb-2">
            <div className="flex flex-wrap items-center justify-between gap-2">
              <div>
                <div className="font-semibold text-lg">
                  {formState.inputs?.name?.value || "Új eszköz"}
                </div>
                <div className="text-xs text-gray-500">
                  Leltári szám:{" "}
                  {formState.inputs?.inventory_number?.value || "még nincs"}
                </div>
              </div>

              {currentStatus && (
                <div className="flex items-center gap-2">
                  <span
                    className="inline-block w-4 h-4 rounded-full border"
                    style={statusColorChipStyle}
                  />
                  <span className="text-sm">{currentStatus.name_hu}</span>
                </div>
              )}

              <div className="text-xs text-gray-500 text-right">
                {references?.sites?.find((s) => s.id === selected.site_id)
                  ?.name || "Nincs telephely"}
                {" · "}
                {references?.departments?.find((d) =>
                  eqId(d.id, selected.department_id)
                )?.name || "Nincs szervezeti egység"}
              </div>
            </div>
          </div>

          {/* 1. Alapadatok */}
          <Section
            title="Alapadatok"
            description="Az eszköz azonosításhoz és státuszához szükséges fő adatok."
            defaultOpen={true}>
            {/* Eszköznév */}
            <Input
              element="input"
              type="text"
              id="name"
              name="name"
              className="col-span-4"
              placeholder="Eszköz neve"
              validators={[VALIDATOR_REQUIRE(), VALIDATOR_MINLENGTH(2)]}
              onChange={inputHandler}
              errorText={"Nem lehet üres"}
              initialValid={true}
              initialValue={formState.inputs?.name?.value || ""}
            />

            {/* Eszköz státusz */}
            <LabeledInput
              label="Eszköz státusz:"
              description="Válassz egy státuszt az eszközhöz."
              className="w-full col-span-2 flex flex-col"
              contentCls="w-full">
              <div className="flex items-center gap-2">
                <div className="flex-1">
                  <SelectInput
                    name="item_status_id"
                    options={selectOptions(
                      references.assetItemStatuses || []
                    ).map((o) => ({
                      ...o,
                      label:
                        (references.assetItemStatuses || []).find(
                          (s) => s.id === (o.value ?? o.id) || s.id === o.value
                        )?.name_hu ?? o.label,
                    }))}
                    value={selectOptions(
                      references.assetItemStatuses || []
                    ).find((x) => x.value === selected.item_status_id)}
                    onChange={(val) =>
                      setSelected((prev) => ({
                        ...prev,
                        item_status_id: val?.value || null,
                      }))
                    }
                    placeholder="Válassz státuszt"
                    isClearable
                  />
                </div>
                {currentStatus && (
                  <div className="flex items-center gap-1 text-xs text-gray-700">
                    <span
                      className="inline-block w-4 h-4 rounded-full border"
                      style={statusColorChipStyle}
                      title={currentStatus.name_hu}
                    />
                    <span>{currentStatus.name_hu}</span>
                  </div>
                )}
              </div>
            </LabeledInput>

            {/* Eszköz csoport */}
            <LabeledInput
              label="Eszköz csoport:"
              description="Válassz ki, hogy milyen eszközcsoportba tartozik az eszköz!"
              className="w-full col-span-2 flex flex-col"
              contentCls="w-full">
              <SelectInput
                name="assetTypes"
                options={selectOptions(references.assetTypes)}
                value={selectOptions(references.assetTypes).find(
                  (x) => x.value === selected.asset_type_id
                )}
                onChange={(val) => handleSelectChange("asset_type_id", val)}
                placeholder="Eszköz csoport"
                isClearable
              />
            </LabeledInput>

            {/* Leltári szám */}
            <LabeledInput
              label="Leltári szám:"
              description="Adjon meg az eszközhöz leltári számot"
              className="w-full col-span-2 flex flex-col"
              contentCls="w-full">
              <Input
                element="input"
                type="text"
                id="inventory_number"
                name="inventory_number"
                inputcls="w-full col-span-2"
                validators={[VALIDATOR_MINLENGTH(2)]}
                onChange={inputHandler}
                errorText={"Min. 2 karakter"}
                initialValid={true}
                initialValue={formState.inputs?.inventory_number?.value || ""}
              />
            </LabeledInput>

            {/* Számlaszám */}
            <LabeledInput
              label="Számlaszám:"
              description="Adjon meg az eszközhöz számlaszámot"
              className="w-full col-span-2 flex flex-col"
              contentCls="w-full">
              <Input
                element="input"
                type="text"
                id="invoice_number"
                name="invoice_number"
                inputcls="w-full col-span-2"
                onChange={inputHandler}
                initialValid={true}
                initialValue={formState.inputs.invoice_number?.value || ""}
              />
            </LabeledInput>
          </Section>

          {/* 2. Elhelyezés */}
          <Section
            title="Elhelyezés"
            description="Hol található és hol használják az eszközt."
            defaultOpen={false}>
            {/* Telephely */}
            <LabeledInput
              label="Telephely:"
              description="Válassz ki, hogy az eszköz melyik telephelyen kerüljön leltárba!"
              className="col-span-4 md:col-span-2"
              contentCls={"w-full"}>
              <SelectInput
                name="site"
                options={selectOptions(references.sites)}
                value={selectOptions(references.sites).find(
                  (x) => x.value === selected.site_id
                )}
                onChange={(val) =>
                  setSelected((prev) => ({
                    ...prev,
                    site_id: val?.value || null,
                    department_id: null,
                    placed_at: null,
                    used_at: null,
                  }))
                }
                placeholder="Telephely"
                isClearable
              />
            </LabeledInput>

            {/* Szervezeti egység */}
            <LabeledInput
              label="Szervezeti egység:"
              description="Válassz ki, hogy az eszköz melyik szervezeti egységnél kerüljön leltárba!"
              className="col-span-4 md:col-span-2"
              contentCls={"w-full"}>
              <SelectInput
                name="department"
                options={selectOptions(filteredDepartments)}
                value={selectOptions(filteredDepartments).find(
                  (x) => x.value === selected.department_id
                )}
                onChange={(val) => {
                  const depId = val?.value || null;
                  const dep = (references?.departments ?? []).find((d) =>
                    eqId(d.id, depId)
                  );
                  setSelected((prev) => ({
                    ...prev,
                    department_id: depId,
                    site_id: dep?.site_id ?? prev.site_id ?? null,
                    placed_at: null,
                    used_at: null,
                  }));
                }}
                placeholder="Szervezeti egység"
                isDisabled={
                  !selected.site_id && filteredDepartments.length === 0
                }
                isClearable
              />
            </LabeledInput>

            {/* Feltalálási hely */}
            <LabeledInput
              label="Feltalálási hely:"
              description="Válassz ki, hogy mely szervezeti egységben található az eszköz!"
              className="w-full col-span-2 flex flex-col"
              contentCls="w-full">
              <SelectInput
                name="placed_at"
                options={selectOptions(filteredLocations)}
                value={selectOptions(filteredLocations).find(
                  (x) => x.value === selected.placed_at
                )}
                onChange={(val) => handleSelectChange("placed_at", val)}
                placeholder="Feltalálási hely"
                isDisabled={!selected.site_id && filteredLocations.length === 0}
                isClearable
              />
            </LabeledInput>

            {/* Felhasználási hely */}
            <LabeledInput
              label="Felhasználási hely:"
              description="Válassz ki, hogy hol kerül felhasználásra az eszköz!"
              className="w-full col-span-2 flex flex-col"
              contentCls="w-full">
              <SelectInput
                name="used_at"
                options={selectOptions(filteredLocations)}
                value={selectOptions(filteredLocations).find(
                  (x) => x.value === selected.used_at
                )}
                onChange={(val) => handleSelectChange("used_at", val)}
                placeholder="Felhasználási hely"
                isDisabled={!selected.site_id && filteredLocations.length === 0}
                isClearable
              />
            </LabeledInput>

            {/* Átmozgatás dátuma */}
            <LabeledInput
              label="Átmozgatás dátuma:"
              className="w-full col-span-2 flex flex-col"
              contentCls="w-full">
              <Input
                element="input"
                type="date"
                id="moved_at_date"
                name="moved_at_date"
                className="w-full col-span-2"
                onChange={inputHandler}
                initialValid={true}
                initialValue={convertGetDateString(
                  formState.inputs.moved_at_date?.value
                )}
              />
            </LabeledInput>
          </Section>

          {/* 3. Pénzügyi adatok */}
          <Section
            title="Pénzügyi adatok"
            description="Beszerzés, partner, költséghely, nettó/bruttó érték."
            defaultOpen={false}>
            {/* Beszerző cég */}
            <LabeledInput
              label="Beszerző cég:"
              description="Válassz ki, hogy melyik cég végezte az eszköz beszerzést!"
              className="w-full col-span-2"
              contentCls="w-full">
              <SelectInput
                name="buyer"
                options={selectOptions(references.ownCompanies)}
                value={selectOptions(references.ownCompanies).find(
                  (x) => x.value === selected.buyer_id
                )}
                onChange={(val) => handleSelectChange("buyer_id", val)}
                placeholder="Beszerző cég"
                isClearable
              />
            </LabeledInput>

            {/* Költséghely */}
            <LabeledInput
              label="Költséghely:"
              description="Válassz költséghelyet."
              className="w-full col-span-2"
              contentCls="w-full">
              <SelectInput
                name="cost_center"
                options={selectOptions(references.ownCompanies)}
                value={selectOptions(references.ownCompanies).find(
                  (x) => x.value === formState.inputs.cost_center?.value
                )}
                onChange={(val) =>
                  inputHandler("cost_center", val?.value, true)
                }
                placeholder="Költséghely"
                isClearable
              />
            </LabeledInput>

            {/* Beszerzés ideje */}
            <LabeledInput
              label="Beszerzés ideje:"
              description="Eszköz beszerzésének napja."
              className="w-full col-span-2"
              contentCls="w-full">
              <Input
                element="input"
                type="date"
                id="purchase_date"
                name="purchase_date"
                className="w-full col-span-2"
                onChange={inputHandler}
                initialValid={true}
                initialValue={
                  formState.inputs?.purchase_date?.value
                    ? convertGetDateString(
                        formState.inputs?.purchase_date?.value
                      )
                    : ""
                }
              />
            </LabeledInput>

            {/* Eladó partner */}
            <LabeledInput
              label="Eladó partner:"
              description="Válassz eladó partnert."
              className="w-full col-span-2"
              contentCls="w-full">
              <SelectInput
                name="seller_id"
                options={selectOptions(references.partnerCompanies)}
                value={selectOptions(references.partnerCompanies).find(
                  (x) => x.value === formState.inputs.seller_id?.value
                )}
                onChange={(val) => inputHandler("seller_id", val?.value, true)}
                placeholder="Eladó partner"
                isClearable
              />
            </LabeledInput>

            {/* Nettó érték + valuta */}
            <LabeledInput
              label="Nettó érték:"
              className="w-full col-span-2 flex flex-col"
              contentCls="w-full">
              <Input
                element="input"
                type="text"
                id="netto_price"
                name="netto_price"
                inputcls="w-full"
                placeholder="Nettó ár"
                onChange={inputHandler}
                initialValid={true}
                initialValue={formState.inputs.netto_price?.value || 0}
              />
              <SelectInput
                name="netto_price_currency"
                options={references.currencies}
                value={references.currencies.find(
                  (c) => c.value === selected.netto_price_currency
                )}
                onChange={(val) =>
                  handleSelectChange("netto_price_currency", val)
                }
                placeholder="Valuta"
                isClearable
              />
            </LabeledInput>

            {/* Bruttó érték + valuta */}
            <LabeledInput
              label="Bruttó beszerzési érték:"
              className="w-full col-span-2 flex flex-col"
              contentCls="w-full">
              <Input
                element="input"
                type="text"
                id="price"
                name="price"
                inputcls="w-full"
                placeholder="Bruttó ár"
                onChange={inputHandler}
                initialValid={true}
                initialValue={formState.inputs.price?.value || 0}
              />
              <SelectInput
                name="price_currency"
                options={references.currencies}
                value={references.currencies.find(
                  (c) => c.value === selected.price_currency
                )}
                onChange={(val) => handleSelectChange("price_currency", val)}
                placeholder="Valuta"
                isClearable
              />
            </LabeledInput>
          </Section>

          {/* 4. Lízing & kapcsolattartó */}
          <Section
            title="Lízing & kapcsolattartó"
            description="Bérleti konstrukció és elérhetőségek."
            defaultOpen={false}>
            {/* Lízingelve? */}
            <LabeledInput
              label="Lízingelve?"
              description="Jelölje, ha eszköz bérleti konstrukcióban van."
              className="w-full col-span-2"
              contentCls="w-full">
              <label className="inline-flex items-center gap-2">
                <input
                  type="checkbox"
                  className="w-[1.25rem] h-[1.25rem]"
                  checked={!!formState.inputs?.isLeased?.value}
                  onChange={(e) =>
                    inputHandler("isLeased", e.target.checked, true)
                  }
                />
                <span>
                  {formState.inputs?.isLeased?.value ? "Igen" : "Nem"}
                </span>
              </label>
            </LabeledInput>

            {/* Kapcsolattartó személy */}
            <LabeledInput
              label="Kapcsolattartó személy"
              className="col-span-2"
              contentCls="w-full">
              <Input
                element="input"
                type="text"
                id="contact_person"
                name="contact_person"
                onChange={inputHandler}
                initialValid={true}
                initialValue={formState.inputs?.contact_person?.value || ""}
              />
            </LabeledInput>

            {/* Kapcsolattartó telefon */}
            <LabeledInput
              label="Kapcsolattartó telefon"
              className="col-span-2"
              contentCls="w-full">
              <Input
                element="input"
                type="text"
                id="contact_phone"
                name="contact_phone"
                onChange={inputHandler}
                initialValid={true}
                initialValue={formState.inputs?.contact_phone?.value || ""}
              />
            </LabeledInput>

            {/* Szerviz telefonszám */}
            <LabeledInput
              label="Szerviz telefonszám"
              className="col-span-2"
              contentCls="w-full">
              <Input
                element="input"
                type="text"
                id="service_phone"
                name="service_phone"
                onChange={inputHandler}
                initialValid={true}
                initialValue={formState.inputs?.service_phone?.value || ""}
              />
            </LabeledInput>
          </Section>

          {/* 5. Technikai adatok */}
          <Section
            title="Technikai adatok"
            description="Gyártó, modell, gyári szám, felülvizsgálat."
            defaultOpen={false}>
            {/* Felelős felhasználó */}
            <LabeledInput
              label="Felelős felhasználó:"
              description="Felhasználó, aki felelősséggel tartozik az eszközért"
              className="w-full col-span-2 flex flex-col"
              contentCls="w-full">
              <SelectInput
                name="assigned_user_id"
                options={selectOptions(references.users)}
                value={selectOptions(references.users).find(
                  (x) => x.value === formState.inputs.assigned_user_id?.value
                )}
                onChange={(val) =>
                  inputHandler("assigned_user_id", val?.value, true)
                }
                placeholder="Felhasználó"
                isClearable
              />
            </LabeledInput>

            {/* Gyártó */}
            <LabeledInput
              label="Gyártó:"
              className="w-full col-span-2 flex flex-col"
              contentCls="w-full">
              <SelectInput
                name="device_brand"
                options={selectOptions(references.deviceBrands)}
                value={selectOptions(references.deviceBrands).find(
                  (x) => x.value === formState.inputs.device_brand?.value
                )}
                onChange={(val) =>
                  inputHandler("device_brand", val?.value, true)
                }
                placeholder="Gyártó"
                isClearable
              />
            </LabeledInput>

            {/* Modell */}
            <LabeledInput
              label="Modell:"
              className="w-full col-span-2 flex flex-col"
              contentCls="w-full">
              <SelectInput
                name="device_model"
                options={selectOptions(references.deviceModels)}
                value={selectOptions(references.deviceModels).find(
                  (x) => x.value === formState.inputs.device_model?.value
                )}
                onChange={(val) =>
                  inputHandler("device_model", val?.value, true)
                }
                placeholder="Modell"
                isClearable
              />
            </LabeledInput>

            {/* Gyári szám */}
            <LabeledInput
              label="Gyári szám:"
              className="w-full col-span-4 flex flex-col"
              contentCls="w-full">
              <Input
                element="input"
                type="text"
                id="serial_number"
                name="serial_number"
                inputcls="w-full col-span-1"
                placeholder=""
                onChange={inputHandler}
                initialValid={true}
                initialValue={formState.inputs?.serial_number?.value || ""}
              />
            </LabeledInput>

            {/* Felülvizsgálat ideje */}
            <LabeledInput
              label="Felülvizsgálat ideje:"
              className="w-full col-span-2 flex flex-col"
              contentCls="w-full">
              <Input
                element="input"
                type="date"
                id="revise_date"
                name="revise_date"
                className="w-full col-span-2"
                onChange={inputHandler}
                initialValid={true}
                initialValue={
                  formState.inputs.revise_date?.value
                    ? convertGetDateString(formState.inputs.revise_date?.value)
                    : ""
                }
              />
            </LabeledInput>

            {/* Felülvizsgálat gyakorisága */}
            <LabeledInput
              label="Felülvizsgálat gyakorisága:"
              className="w-full col-span-2 flex flex-col"
              contentCls="w-full">
              <Input
                element="input"
                type="text"
                id="revision_frequency"
                name="revision_frequency"
                inputcls="w-full col-span-1"
                onChange={inputHandler}
                initialValid={true}
                initialValue={formState.inputs.revision_frequency?.value || ""}
              />
            </LabeledInput>
          </Section>

          {/* 6. Telepítés & dokumentáció (haladó) */}
          <Section
            title="Telepítési információk & leírás"
            description="Telepítési lépések, követelmények, plusz funkciók és leírás."
            defaultOpen={false}>
            {/* Telepítési lépések */}
            <LabeledInput
              label="Telepítési lépések"
              description="Leírás a telepítés menetéről."
              className="w-full col-span-4 flex flex-col"
              contentCls="w-full">
              <SuneditorComponent
                lang="en"
                name="install_steps"
                value={formState.inputs?.install_steps?.value || ""}
                placeholder="Ide gépelheti a szöveget..."
                onChange={(v) => inputHandler("install_steps", v, true)}
                options={{
                  buttonList: [
                    ["bold", "italic", "underline"],
                    ["list", "table"],
                    ["link", "image"],
                    ["undo", "redo"],
                  ],
                }}
              />
            </LabeledInput>

            {/* Telepítési követelmények */}
            <LabeledInput
              label="Telepítési követelmények"
              description="HW/SW/prereq feltételek."
              className="w-full col-span-4 flex flex-col"
              contentCls="w-full">
              <SuneditorComponent
                lang="en"
                name="install_requirements"
                value={formState.inputs?.install_requirements?.value || ""}
                placeholder="Ide gépelheti a szöveget..."
                onChange={(v) => inputHandler("install_requirements", v, true)}
                options={{
                  buttonList: [
                    ["bold", "italic", "underline"],
                    ["list", "table"],
                    ["link", "image"],
                    ["undo", "redo"],
                  ],
                }}
              />
            </LabeledInput>

            {/* Plusz funkció */}
            <LabeledInput
              label={"Plusz funkció:"}
              description={
                "Adja meg, hogy milyen plusz funkciókkal rendelkezik az eszköz."
              }
              className="w-full col-span-4 flex flex-col"
              contentCls="w-full">
              <SuneditorComponent
                lang={"en"}
                name={"extended_functions"}
                value={extendedFunctions}
                placeholder={"Ide gépelheti a szöveget..."}
                onChange={(value) => setExtendedFunctions(value)}
                options={{
                  buttonList: [
                    ["font", "fontSize", "formatBlock"],
                    [
                      "bold",
                      "underline",
                      "italic",
                      "strike",
                      "subscript",
                      "superscript",
                    ],
                    ["align", "list", "table"],
                    ["fontColor", "hiliteColor"],
                    ["outdent", "indent"],
                    ["undo", "redo"],
                    ["removeFormat"],
                    ["link", "image"],
                    ["preview", "print"],
                    ["fullScreen", "showBlocks", "codeView"],
                  ],
                  tableWidth: "100%",
                  tableStyles: {
                    border: "1px solid black",
                  },
                }}
              />
            </LabeledInput>

            {/* Leírás */}
            <LabeledInput
              label={"Leírás:"}
              description={"Adja meg szöveges tartalmat."}
              className="w-full col-span-4 flex flex-col"
              contentCls="w-full">
              <SuneditorComponent
                lang={"en"}
                name={"description"}
                value={assetDescription}
                placeholder={"Ide gépelheti a szöveget..."}
                onChange={(value) => setAssetDescription(value)}
                options={{
                  buttonList: [
                    ["font", "fontSize", "formatBlock"],
                    [
                      "bold",
                      "underline",
                      "italic",
                      "strike",
                      "subscript",
                      "superscript",
                    ],
                    ["align", "list", "table"],
                    ["fontColor", "hiliteColor"],
                    ["outdent", "indent"],
                    ["undo", "redo"],
                    ["removeFormat"],
                    ["link", "image"],
                    ["preview", "print"],
                    ["fullScreen", "showBlocks", "codeView"],
                  ],
                  tableWidth: "100%",
                  tableStyles: {
                    border: "1px solid black",
                  },
                }}
              />
            </LabeledInput>
          </Section>

          {/* 7. Státusz információk (read-only) */}
          <Section
            title="Státusz információk"
            description="Eszköztörzslap és eszközbiztosítás státusza."
            defaultOpen={false}>
            {/* Eszköztörzslap státusza */}
            <LabeledInput
              label="Eszköztörzslap státusza:"
              className="w-full col-span-2 flex flex-col"
              contentCls="w-full">
              <input
                name="device_reg_page"
                value={
                  asset.device_reg_page
                    ? boolFields[0]?.label
                    : boolFields[1]?.label
                }
                readOnly
                className="border rounded px-2 py-1 text-sm bg-gray-50"
              />
            </LabeledInput>

            {/* Eszközbiztosítás */}
            <LabeledInput
              label="Eszközbiztosítás státusza:"
              className="w-full col-span-2 flex flex-col"
              contentCls="w-full">
              <input
                name="device_insurance"
                value={
                  asset.device_reg_page
                    ? boolFields[0]?.label
                    : boolFields[1]?.label
                }
                readOnly
                className="border rounded px-2 py-1 text-sm bg-gray-50"
              />
            </LabeledInput>
          </Section>

          {/* Save button */}
          <div className="flex justify-end">
            <button
              className="bg-blue-600 text-white px-4 py-2 mt-2 rounded hover:bg-blue-700"
              onClick={handleSave}>
              Mentés
            </button>
          </div>
        </>
      )}
    </div>
  );
};

export default InventoryElement;

// import React, { useEffect, useState, useMemo } from "react";
// import Input from "@/components/shared/input/Input";
// import Select from "react-select";
// import { useForm } from "@/hooks/custom-form-hook";
// import { VALIDATOR_REQUIRE, VALIDATOR_MINLENGTH } from "@/utils/validators";
// import SuneditorComponent from "@/components/shared/texteditor/SuneditorComponent";
// import LabeledInput from "@/components/shared/labeledinput/LabeledInput";
// import SelectInput from "@/components/shared/selectinput/SelectInput";
// import ApiCall from "@/api/api-calls";
// import { toast } from "react-hot-toast";
// import { convertGetDateString } from "@/utils/converDateString";
// import { currencyDisplayHandler } from "@/utils/currencyDisplay";
// import Separator from "@/components/shared/separator/Separator";

// const Section = ({
//   title,
//   description,
//   children,
//   defaultOpen = true,
// }) => {
//   const [open, setOpen] = React.useState(defaultOpen);

//   return (
//     <div className="border rounded-lg bg-white shadow-sm mb-4">
//       <button
//         type="button"
//         className="w-full flex items-center justify-between px-3 py-2 border-b text-left"
//         onClick={() => setOpen((o) => !o)}
//       >
//         <div>
//           <div className="font-semibold">{title}</div>
//           {description && (
//             <div className="text-xs text-gray-500">{description}</div>
//           )}
//         </div>
//         <span className="text-gray-500 text-sm">
//           {open ? "▲" : "▼"}
//         </span>
//       </button>
//       {open && <div className="p-3 grid grid-cols-4 gap-3">{children}</div>}
//     </div>
//   );
// };

// const InventoryElement = ({
//   asset,
//   references,
//   onRowUpdated,
//   isCreateMode = false,
// }) => {
//   const [formState, inputHandler, setFormData] = useForm({}, false);
//   const [assetDescription, setAssetDescription] = useState(
//     asset.description || ""
//   );
//   const [extendedFunctions, setExtendedFunctions] = useState(
//     asset.extended_functions || ""
//   );
//   // const [selected, setSelected] = useState({
//   //   asset_type_id: asset.asset_type_id,
//   //   buyer_id: asset.buyer_id,
//   //   department_id: asset.department_id,
//   //   placed_at: asset.placed_at,
//   //   site_id: asset.site_id,
//   //   used_at: asset.used_at,
//   //   price_currency: asset.price_currency_id ?? null,
//   //   netto_price_currency: asset.netto_price_currency_id ?? null,
//   // });
//   const boolFields = [
//     { id: 0, value: true, label: "Dokumentum csatolva." },
//     { id: 1, value: false, label: "Nincs dokumentum." },
//   ];
//   const [selected, setSelected] = useState({
//     asset_type_id: asset.asset_type_id,
//     buyer_id: asset.buyer_id,
//     department_id: asset.department_id,
//     placed_at: asset.placed_at,
//     site_id: asset.site_id,
//     used_at: asset.used_at,
//     price_currency: asset.price_currency_id ?? null,
//     netto_price_currency: asset.netto_price_currency_id ?? null,
//     item_status_id: asset.item_status_id ?? null, // 🆕
//   });
//   const eqId = (a, b) => (a?.toString?.() ?? a) === (b?.toString?.() ?? b);
//   // 🆕 filtered lists based on selected site/department
//   const filteredDepartments = useMemo(() => {
//     const deps = references?.departments ?? [];
//     if (!selected.site_id) return deps;

//     return deps.filter((d) => eqId(d.site_id, selected.site_id));
//   }, [references, selected.site_id]);

//   const filteredLocations = useMemo(() => {
//     const locs = references?.locations ?? [];
//     if (selected.department_id) {
//       return locs.filter((l) => eqId(l.department_id, selected.department_id));
//     }
//     if (selected.site_id) {
//       return locs.filter((l) => eqId(l.site_id, selected.site_id));
//     }
//     return locs;
//   }, [references, selected.department_id, selected.site_id]);

//   useEffect(() => {
//     // if site changed and department/site no longer valid
//     if (
//       selected.department_id &&
//       !filteredDepartments.some((d) => eqId(d.id, selected.department_id))
//     ) {
//       setSelected((prev) => ({ ...prev, department_id: null }));
//     }
//   }, [filteredDepartments, selected.department_id]);

//   useEffect(() => {
//     // clear placed/used if not in filteredLocations anymore
//     const ids = new Set(filteredLocations.map((l) => l.id));
//     setSelected((prev) => ({
//       ...prev,
//       placed_at:
//         prev.placed_at && ids.has(prev.placed_at) ? prev.placed_at : null,
//       used_at: prev.used_at && ids.has(prev.used_at) ? prev.used_at : null,
//     }));
//   }, [filteredLocations]);

//   const selectOptions = (items) =>
//     Array.isArray(items) && items.length && (items[0].id || items[0].value)
//       ? items.map((x) => ({ value: x.value ?? x.id, label: x.label ?? x.name }))
//       : items || [];

//   const handleSelectChange = (field, option) => {
//     setSelected((prev) => ({ ...prev, [field]: option?.value || null }));
//   };
//   useEffect(() => {
//     if (asset) {
//       const initialData = {
//         name: { value: asset.name || "", isValid: true },
//         inventory_number: {
//           value: asset.inventory_number || "",
//           isValid: true,
//         },
//         invoice_number: { value: asset.invoice_number || "", isValid: true },
//         revision_frequency: {
//           value: asset.revision_frequency || "",
//           isValid: true,
//         },
//         serial_number: { value: asset.serial_number || "", isValid: true },
//         external_id: { value: asset.external_id || "", isValid: true },
//         description: { value: asset.description || "", isValid: true },
//         extended_functions: {
//           value: asset.extended_functions || "",
//           isValid: true,
//         },
//         moved_at_date: { value: asset.moved_at_date || null, isValid: true },
//         purchase_date: { value: asset.purchase_date || null, isValid: true },
//         revise_date: { value: asset.revise_date || null, isValid: true },
//         price: { value: parseFloat(asset.price) || 0, isValid: true },
//         netto_price: {
//           value: parseFloat(asset.netto_price) || 0,
//           isValid: true,
//         },
//         cost_center: { value: asset.cost_center || "", isValid: true },
//         seller_id: { value: asset.seller_id || "", isValid: true },

//         assigned_user_id: {
//           value: asset.assigned_user_id || "",
//           isValid: true,
//         },
//         device_brand: { value: asset.device_brand || "", isValid: true },
//         device_model: { value: asset.device_model || "", isValid: true },
//         isLeased: { value: !!asset.is_leased, isValid: true },
//         service_phone: { value: asset.service_phone || "", isValid: true },
//         contact_person: { value: asset.contact_person || "", isValid: true },
//         contact_phone: { value: asset.contact_phone || "", isValid: true },
//         install_steps: { value: asset.install_steps || "", isValid: true },
//         install_requirements: {
//           value: asset.install_requirements || "",
//           isValid: true,
//         },
//         device_isLeased: {
//           value: asset.device_is_leased || null,
//           isValid: true,
//         }, // read-only here
//       };
//       const currentData = formState.inputs;

//       const isSame =
//         JSON.stringify(initialData) === JSON.stringify(currentData);

//       if (!isSame) {
//         setFormData(initialData, true);
//       }
//     }
//   }, [asset]);

//   const handleSave = async () => {
//     const rawFormData = Object.fromEntries(
//       Object.entries(formState.inputs).map(([k, v]) => [k, v.value])
//     );

//     const payload = {
//       ...rawFormData,
//       price: rawFormData.price ? parseFloat(rawFormData.price) : null,
//       netto_price: rawFormData.netto_price
//         ? parseFloat(rawFormData.netto_price)
//         : null,
//       asset_type_id: selected.asset_type_id,
//       buyer_id: selected.buyer_id,
//       department_id: selected.department_id,
//       placed_at: selected.placed_at,
//       site_id: selected.site_id,
//       used_at: selected.used_at,
//       extended_functions: extendedFunctions,
//       description: assetDescription,
//       price_currency_id: selected.price_currency,
//       netto_price_currency_id: selected.netto_price_currency,
//       isLeased: !!rawFormData.isLeased,
//       service_phone: rawFormData.service_phone || "",
//       contact_person: rawFormData.contact_person || "",
//       contact_phone: rawFormData.contact_phone || "",
//       install_steps: rawFormData.install_steps || "",
//       install_requirements: rawFormData.install_requirements || "",
//       item_status_id: selected.item_status_id || null,
//     };
//     console.log("formState before save:", formState.inputs);
//     console.log("rawFormData:", rawFormData);
//     try {
//       if (isCreateMode) {
//         await ApiCall.inventory.createAsset(payload);
//         toast.success("Eszköz létrehozva");
//       } else {
//         await ApiCall.inventory.updateAsset(asset.id, payload);
//         toast.success("Mentés sikeres");
//       }
//       onRowUpdated?.();
//     } catch (err) {
//       toast.error(isCreateMode ? "Létrehozás sikertelen" : "Mentés sikertelen");
//     }
//   };
//   const currentStatus =
//     (references?.assetItemStatuses || []).find((s) =>
//       eqId(s.id, selected.item_status_id)
//     ) || null;

//   const statusColorChipStyle = currentStatus?.color_rgb
//     ? {
//         backgroundColor: `rgb(${currentStatus.color_rgb})`,
//       }
//     : {};
//   useEffect(() => {
//     if (!selected.price_currency && references?.currencies?.length) {
//       const defaultCurrency = references.currencies.find(
//         (c) => c.label === "HUF"
//       );
//       setSelected((prev) => ({
//         ...prev,
//         price_currency: defaultCurrency?.value,
//         netto_price_currency: defaultCurrency?.value,
//       }));
//     }
//   }, [references]);

//   useEffect(() => {
//     if (
//       !selected.site_id &&
//       selected.department_id &&
//       references?.departments?.length
//     ) {
//       const dep = references.departments.find(
//         (d) => d.id === selected.department_id
//       );
//       if (dep?.site_id) {
//         setSelected((prev) => ({ ...prev, site_id: dep.site_id }));
//       }
//     }
//   }, [selected.site_id, selected.department_id, references?.departments]);

//   return (
//     <div className="flex flex-col space-y-4 ">
//       {formState?.inputs && (
//         <div className="w-full grid gap-3 grid-cols-4">
//           {/* Eszköznév */}
//           <Input
//             element="input"
//             type="text"
//             id="name"
//             name="name"
//             className="col-span-4"
//             placeholder="Eszköz neve"
//             validators={[VALIDATOR_REQUIRE(), VALIDATOR_MINLENGTH(2)]}
//             onChange={inputHandler}
//             errorText={"Nem lehet üres"}
//             initialValid={true}
//             initialValue={formState.inputs?.name?.value || ""}
//           />
//           {/* 🆕 Eszköz státusz */}
//           <LabeledInput
//             label="Eszköz státusz:"
//             description="Válassz egy státuszt az eszközhöz."
//             className="w-full col-span-2 flex flex-col"
//             contentCls="w-full">
//             <div className="flex items-center gap-2">
//               <div className="flex-1">
//                 <SelectInput
//                   name="item_status_id"
//                   options={selectOptions(
//                     references.assetItemStatuses || []
//                   ).map((o) => ({
//                     ...o,
//                     // label = Hungarian name if provided
//                     label:
//                       (references.assetItemStatuses || []).find(
//                         (s) => s.id === (o.value ?? o.id) || s.id === o.value
//                       )?.name_hu ?? o.label,
//                   }))}
//                   value={selectOptions(references.assetItemStatuses || []).find(
//                     (x) => x.value === selected.item_status_id
//                   )}
//                   onChange={(val) =>
//                     setSelected((prev) => ({
//                       ...prev,
//                       item_status_id: val?.value || null,
//                     }))
//                   }
//                   placeholder="Válassz státuszt"
//                   isClearable
//                 />
//               </div>
//               {currentStatus && (
//                 <div className="flex items-center gap-1 text-xs text-gray-700">
//                   <span
//                     className="inline-block w-4 h-4 rounded-full border"
//                     style={statusColorChipStyle}
//                     title={currentStatus.name_hu}
//                   />
//                   <span>{currentStatus.name_hu}</span>
//                 </div>
//               )}
//             </div>
//           </LabeledInput>
//           {/* Telephely */}
//           <LabeledInput
//             label="Telephely:"
//             description="Válassz ki, hogy az eszköz melyik telephelyen kerüljön leltárba!"
//             className="col-span-4 md:col-span-2"
//             contentCls={"w-full"}>
//             <SelectInput
//               name="site"
//               options={selectOptions(references.sites)}
//               value={selectOptions(references.sites).find(
//                 (x) => x.value === selected.site_id
//               )}
//               onChange={(val) =>
//                 setSelected((prev) => ({
//                   ...prev,
//                   site_id: val?.value || null,
//                   department_id: null,
//                   placed_at: null,
//                   used_at: null,
//                 }))
//               }
//               placeholder="Telephely"
//               isClearable
//             />
//           </LabeledInput>
//           {/* Szervezeti egység */}
//           <LabeledInput
//             label="Szervezeti egység:"
//             description="Válassz ki, hogy az eszköz melyik szervezeti egységnél kerüljön leltárba!"
//             className="col-span-4 md:col-span-2"
//             contentCls={"w-full"}>
//             <SelectInput
//               name="department"
//               options={selectOptions(filteredDepartments)}
//               value={selectOptions(filteredDepartments).find(
//                 (x) => x.value === selected.department_id
//               )}
//               onChange={(val) => {
//                 const depId = val?.value || null;
//                 const dep = (references?.departments ?? []).find((d) =>
//                   eqId(d.id, depId)
//                 );
//                 setSelected((prev) => ({
//                   ...prev,
//                   department_id: depId,
//                   // keep site in sync with department; if dep has a site, use it
//                   site_id: dep?.site_id ?? prev.site_id ?? null,
//                   // clear locations because the scope changed
//                   placed_at: null,
//                   used_at: null,
//                 }));
//               }}
//               placeholder="Szervezeti egység"
//               isDisabled={!selected.site_id && filteredDepartments.length === 0}
//               isClearable
//             />
//           </LabeledInput>
//           {/* feltalálási hely */}
//           <LabeledInput
//             label="Feltalálási hely:"
//             description="Válassz ki, hogy mely szervezeti egységben található az eszköz!"
//             className="w-full col-span-2 flex flex-col"
//             contentCls="w-full">
//             <SelectInput
//               name="placed_at"
//               options={selectOptions(filteredLocations)}
//               value={selectOptions(filteredLocations).find(
//                 (x) => x.value === selected.placed_at
//               )}
//               onChange={(val) => handleSelectChange("placed_at", val)}
//               placeholder="Feltalálási hely"
//               isDisabled={!selected.site_id && filteredLocations.length === 0}
//               isClearable
//             />
//           </LabeledInput>
//           {/* felhasználási hely */}
//           <LabeledInput
//             label="Felhasználási hely:"
//             description="Válassz ki, hogy hol kerül felhasználásra az eszköz az eszköz!"
//             className="w-full col-span-2 flex flex-col"
//             contentCls="w-full">
//             <SelectInput
//               name="used_at"
//               options={selectOptions(filteredLocations)}
//               value={selectOptions(filteredLocations).find(
//                 (x) => x.value === selected.used_at
//               )}
//               onChange={(val) => handleSelectChange("used_at", val)}
//               placeholder="Felhasználási hely"
//               isDisabled={!selected.site_id && filteredLocations.length === 0}
//               isClearable
//             />
//           </LabeledInput>
//           <Separator cls="col-span-4 border-b border-2 border-b-green-500 border-dashed" />
//           {/* Beszerző cég */}
//           <LabeledInput
//             label="Beszerző cég:"
//             description="Válassz ki, hogy melyik cég végezte az eszköz beszerzést!"
//             className="w-full col-span-2"
//             contentCls="w-full">
//             <SelectInput
//               name="buyer"
//               options={selectOptions(references.ownCompanies)}
//               value={selectOptions(references.ownCompanies).find(
//                 (x) => x.value === selected.buyer_id
//               )}
//               onChange={(val) => handleSelectChange("buyer_id", val)}
//               placeholder="Beszerző cég"
//               isClearable
//             />
//           </LabeledInput>
//           {/* Költséghely */}
//           <LabeledInput
//             label="Költséghely:"
//             description="Válassz költséghelyet."
//             className="w-full col-span-2"
//             contentCls="w-full">
//             <SelectInput
//               name="cost_center"
//               options={selectOptions(references.ownCompanies)}
//               value={selectOptions(references.ownCompanies).find(
//                 (x) => x.value === formState.inputs.cost_center?.value
//               )}
//               onChange={(val) => inputHandler("cost_center", val?.value, true)}
//               placeholder="Költséghely"
//               isClearable
//             />
//           </LabeledInput>
//           {/* Leltári szám */}
//           <LabeledInput
//             label="Leltári szám:"
//             description="Adjon meg az eszközhöz leltári számot"
//             className="w-full col-span-2 flex flex-col"
//             contentCls="w-full">
//             <Input
//               element="input"
//               type="text"
//               id="inventory_number"
//               name="inventory_number"
//               inputcls="w-full col-span-2"
//               validators={[VALIDATOR_MINLENGTH(2)]}
//               onChange={inputHandler}
//               // onBlur={touchHandler}
//               errorText={"Min. 2 karakter"}
//               initialValid={true}
//               initialValue={formState.inputs?.inventory_number?.value || ""}
//             />
//           </LabeledInput>
//           {/* Számlaszám */}
//           <LabeledInput
//             label="Számlaszám:"
//             description="Adjon meg az eszközhöz számlaszámot"
//             className="w-full col-span-2 flex flex-col"
//             contentCls="w-full">
//             <Input
//               element="input"
//               type="text"
//               id="invoice_number"
//               name="invoice_number"
//               inputcls="w-full col-span-2"
//               onChange={inputHandler}
//               initialValid={true}
//               initialValue={formState.inputs.invoice_number?.value || ""}
//             />
//           </LabeledInput>
//           {/* Eladó partner */}
//           <LabeledInput
//             label="Eladó partner:"
//             description="Válassz eladó partnert."
//             className="w-full col-span-2"
//             contentCls="w-full">
//             <SelectInput
//               name="seller_id"
//               options={selectOptions(references.partnerCompanies)}
//               value={selectOptions(references.partnerCompanies).find(
//                 (x) => x.value === formState.inputs.seller_id?.value
//               )}
//               onChange={(val) => inputHandler("seller_id", val?.value, true)}
//               placeholder="Eladó partner"
//               isClearable
//             />
//           </LabeledInput>
//           {/* Beszerzés ideje */}
//           <LabeledInput
//             label="Beszerzés ideje:"
//             description="Eszköz beszerzésének napja."
//             className="w-full col-span-2"
//             contentCls="w-full">
//             <Input
//               element="input"
//               type="date"
//               id="purchase_date"
//               name="purchase_date"
//               className="w-full col-span-2"
//               onChange={inputHandler}
//               initialValid={true}
//               initialValue={
//                 formState.inputs?.purchase_date?.value
//                   ? convertGetDateString(formState.inputs?.purchase_date?.value)
//                   : ""
//               }
//             />
//           </LabeledInput>
//           {/* Nettó érték */}
//           <LabeledInput
//             label="Nettó érték:"
//             className="w-full col-span-2 flex flex-col"
//             contentCls="w-full">
//             <Input
//               element="input"
//               type="text"
//               id="netto_price"
//               name="netto_price"
//               inputcls="w-full"
//               placeholder="Nettó ár"
//               onChange={inputHandler}
//               initialValid={true}
//               initialValue={formState.inputs.netto_price?.value || 0}
//             />

//             <SelectInput
//               name="netto_price_currency"
//               options={references.currencies}
//               value={references.currencies.find(
//                 (c) => c.value === selected.netto_price_currency
//               )}
//               onChange={(val) =>
//                 handleSelectChange("netto_price_currency", val)
//               }
//               placeholder="Valuta"
//               isClearable
//             />
//           </LabeledInput>
//           {/* Bruttó beszerzési érték */}
//           <LabeledInput
//             label="Bruttó beszerzési érték:"
//             className="w-full col-span-2 flex flex-col"
//             contentCls="w-full">
//             <Input
//               element="input"
//               type="text"
//               id="price"
//               name="price"
//               inputcls="w-full"
//               placeholder="Bruttó ár"
//               onChange={inputHandler}
//               initialValid={true}
//               initialValue={formState.inputs.price?.value || 0}
//             />
//             <SelectInput
//               name="price_currency"
//               options={references.currencies}
//               value={references.currencies.find(
//                 (c) => c.value === selected.price_currency
//               )}
//               onChange={(val) => handleSelectChange("price_currency", val)}
//               placeholder="Valuta"
//               isClearable
//             />
//           </LabeledInput>
//           {/* 🆕 Lízingelve? */}
//           <LabeledInput
//             label="Lízingelve?"
//             description="Jelölje, ha eszköz bérleti konstrukcióban van."
//             className="w-full col-span-2"
//             contentCls="w-full">
//             <label className="inline-flex items-center gap-2">
//               <input
//                 type="checkbox"
//                 className="w-[1.25rem] h-[1.25rem]"
//                 checked={!!formState.inputs?.isLeased?.value}
//                 onChange={(e) =>
//                   inputHandler("isLeased", e.target.checked, true)
//                 }
//               />
//               <span>{formState.inputs?.isLeased?.value ? "Igen" : "Nem"}</span>
//             </label>
//           </LabeledInput>

//           {/* 🆕 Kapcsolattartó + telefon */}
//           <LabeledInput
//             label="Kapcsolattartó személy"
//             className="col-span-2"
//             contentCls="w-full">
//             <Input
//               element="input"
//               type="text"
//               id="contact_person"
//               name="contact_person"
//               onChange={inputHandler}
//               initialValid={true}
//               initialValue={formState.inputs?.contact_person?.value || ""}
//             />
//           </LabeledInput>
//           <LabeledInput
//             label="Kapcsolattartó telefon"
//             className="col-span-2"
//             contentCls="w-full">
//             <Input
//               element="input"
//               type="text"
//               id="contact_phone"
//               name="contact_phone"
//               onChange={inputHandler}
//               initialValid={true}
//               initialValue={formState.inputs?.contact_phone?.value || ""}
//             />
//           </LabeledInput>

//           {/* 🆕 Szerviz telefonszám */}
//           <LabeledInput
//             label="Szerviz telefonszám"
//             className="col-span-2"
//             contentCls="w-full">
//             <Input
//               element="input"
//               type="text"
//               id="service_phone"
//               name="service_phone"
//               onChange={inputHandler}
//               initialValid={true}
//               initialValue={formState.inputs?.service_phone?.value || ""}
//             />
//           </LabeledInput>

//           {/* 🆕 Telepítési lépések */}
//           <LabeledInput
//             label="Telepítési lépések"
//             description="Leírás a telepítés menetéről."
//             className="w-full col-span-4 flex flex-col"
//             contentCls="w-full">
//             <SuneditorComponent
//               lang="en"
//               name="install_steps"
//               value={formState.inputs?.install_steps?.value || ""}
//               placeholder="Ide gépelheti a szöveget..."
//               onChange={(v) => inputHandler("install_steps", v, true)}
//               options={{
//                 buttonList: [
//                   ["bold", "italic", "underline"],
//                   ["list", "table"],
//                   ["link", "image"],
//                   ["undo", "redo"],
//                 ],
//               }}
//             />
//           </LabeledInput>

//           {/* 🆕 Telepítési követelmények */}
//           <LabeledInput
//             label="Telepítési követelmények"
//             description="HW/SW/prereq feltételek."
//             className="w-full col-span-4 flex flex-col"
//             contentCls="w-full">
//             <SuneditorComponent
//               lang="en"
//               name="install_requirements"
//               value={formState.inputs?.install_requirements?.value || ""}
//               placeholder="Ide gépelheti a szöveget..."
//               onChange={(v) => inputHandler("install_requirements", v, true)}
//               options={{
//                 buttonList: [
//                   ["bold", "italic", "underline"],
//                   ["list", "table"],
//                   ["link", "image"],
//                   ["undo", "redo"],
//                 ],
//               }}
//             />
//           </LabeledInput>
//           <Separator cls="col-span-4 border-b border-2 border-b-green-500 border-dashed" />

//           {/* Eszköz csoport */}
//           <LabeledInput
//             label="Eszköz csoport:"
//             description="Válassz ki, hogy milyen eszközcsoportba tartozik az eszköz!"
//             className="w-full col-span-2 flex flex-col"
//             contentCls="w-full">
//             <SelectInput
//               name="assetTypes"
//               options={selectOptions(references.assetTypes)}
//               value={selectOptions(references.assetTypes).find(
//                 (x) => x.value === selected.asset_type_id
//               )}
//               onChange={(val) => handleSelectChange("asset_type_id", val)}
//               placeholder="Eszköz csoport"
//               isClearable
//             />
//           </LabeledInput>
//           {/* Felelős felhasználó */}
//           <LabeledInput
//             label="Felelős felhasználó:"
//             description="Felhasználó, aki felelősséggel tartozik az eszközért"
//             className="w-full col-span-2 flex flex-col"
//             contentCls="w-full">
//             <SelectInput
//               name="assigned_user_id"
//               options={selectOptions(references.users)}
//               value={selectOptions(references.users).find(
//                 (x) => x.value === formState.inputs.assigned_user_id?.value
//               )}
//               onChange={(val) =>
//                 inputHandler("assigned_user_id", val?.value, true)
//               }
//               placeholder="Felhasználó"
//               isClearable
//             />
//           </LabeledInput>

//           {/* Gyártó */}
//           <LabeledInput
//             label="Gyártó:"
//             className="w-full col-span-2 flex flex-col"
//             contentCls="w-full">
//             <SelectInput
//               name="device_brand"
//               options={selectOptions(references.deviceBrands)}
//               value={selectOptions(references.deviceBrands).find(
//                 (x) => x.value === formState.inputs.device_brand?.value
//               )}
//               onChange={(val) => inputHandler("device_brand", val?.value, true)}
//               placeholder="Gyártó"
//               isClearable
//             />
//           </LabeledInput>
//           {/*Modell  */}
//           <LabeledInput
//             label="Modell:"
//             className="w-full col-span-2 flex flex-col"
//             contentCls="w-full">
//             <SelectInput
//               name="device_model"
//               options={selectOptions(references.deviceModels)}
//               value={selectOptions(references.deviceModels).find(
//                 (x) => x.value === formState.inputs.device_model?.value
//               )}
//               onChange={(val) => inputHandler("device_model", val?.value, true)}
//               placeholder="Modell"
//               isClearable
//             />
//           </LabeledInput>
//           {/* Gyári szám */}
//           <LabeledInput
//             label="Gyári szám:"
//             className="w-full col-span-4 flex flex-col"
//             contentCls="w-full">
//             <Input
//               element="input"
//               type="text"
//               id="serial_number"
//               name="serial_number"
//               inputcls="w-full col-span-1"
//               placeholder=""
//               onChange={inputHandler}
//               initialValid={true}
//               initialValue={formState.inputs?.serial_number?.value || ""}
//             />
//           </LabeledInput>
//           {/* Felülvizsgálat ideje */}
//           <LabeledInput
//             label="Felülvizsgálat ideje:"
//             className="w-full col-span-2 flex flex-col"
//             contentCls="w-full">
//             <Input
//               element="input"
//               type="date"
//               id="revise_date"
//               name="revise_date"
//               className="w-full col-span-2"
//               onChange={inputHandler}
//               initialValid={true}
//               initialValue={
//                 formState.inputs.revise_date?.value
//                   ? convertGetDateString(formState.inputs.revise_date?.value)
//                   : ""
//               }
//             />
//           </LabeledInput>
//           {/* Felülvizsgálat gyakorisága */}
//           <LabeledInput
//             label="Felülvizsgálat gyakorisága:"
//             className="w-full col-span-2 flex flex-col"
//             contentCls="w-full">
//             <Input
//               element="input"
//               type="text"
//               id="revision_frequency"
//               name="revision_frequency"
//               inputcls="w-full col-span-1"
//               onChange={inputHandler}
//               initialValid={true}
//               initialValue={formState.inputs.revision_frequency?.value || ""}
//             />
//           </LabeledInput>
//           {/* átmozgatás dátuma */}
//           <Input
//             type="date"
//             id="moved_at_date"
//             name="moved_at_date"
//             initialValue={convertGetDateString(
//               formState.inputs.moved_at_date?.value
//             )}
//             onChange={inputHandler}
//           />

//           {/* gyártó - device brands*/}

//           {/* modell - device model*/}
//           {/* gyári szám */}

//           {/* plusz funkció */}
//           <LabeledInput
//             label={"Plusz funkció:"}
//             description={
//               "Adja meg, hogy milyen plusz funkciókkal rendelkezik az eszköz."
//             }
//             className="w-full col-span-4 flex flex-col"
//             contentCls="w-full">
//             <SuneditorComponent
//               lang={"en"}
//               name={"extended_functions"}
//               value={extendedFunctions}
//               placeholder={"Ide gépelheti a szöveget..."}
//               onChange={(value) => setExtendedFunctions(value)}
//               options={{
//                 buttonList: [
//                   ["font", "fontSize", "formatBlock"],
//                   [
//                     "bold",
//                     "underline",
//                     "italic",
//                     "strike",
//                     "subscript",
//                     "superscript",
//                   ],
//                   ["align", "list", "table"],
//                   ["fontColor", "hiliteColor"],
//                   ["outdent", "indent"],
//                   ["undo", "redo"],
//                   ["removeFormat"],
//                   ["link", "image"],
//                   ["preview", "print"],
//                   ["fullScreen", "showBlocks", "codeView"],
//                 ],
//                 tableWidth: "100%",
//                 tableStyles: {
//                   border: "1px solid black",
//                 },
//               }}
//             />
//           </LabeledInput>
//           {/* megjegyzés */}
//           <LabeledInput
//             label={"Leírás:"}
//             description={"Adja meg szöveges tartalmat."}
//             className="w-full col-span-4 flex flex-col"
//             contentCls="w-full">
//             <SuneditorComponent
//               lang={"en"}
//               name={"description"}
//               value={assetDescription}
//               placeholder={"Ide gépelheti a szöveget..."}
//               onChange={(value) => setAssetDescription(value)}
//               options={{
//                 buttonList: [
//                   ["font", "fontSize", "formatBlock"],
//                   [
//                     "bold",
//                     "underline",
//                     "italic",
//                     "strike",
//                     "subscript",
//                     "superscript",
//                   ],
//                   ["align", "list", "table"],
//                   ["fontColor", "hiliteColor"],
//                   ["outdent", "indent"],
//                   ["undo", "redo"],
//                   ["removeFormat"],
//                   ["link", "image"],
//                   ["preview", "print"],
//                   ["fullScreen", "showBlocks", "codeView"],
//                 ],
//                 tableWidth: "100%",
//                 tableStyles: {
//                   border: "1px solid black",
//                 },
//               }}
//             />
//           </LabeledInput>
//           {/* eszköztörzslap státusza */}
//           <LabeledInput
//             label="Eszköztörzslap státusza:"
//             className="w-full col-span-1 flex flex-col"
//             contentCls="w-full">
//             <input
//               name="device_reg_page"
//               value={
//                 asset.device_reg_page
//                   ? boolFields[0]?.label
//                   : boolFields[1]?.label
//               }
//               readOnly
//               placeholder="Eszköztörzslap státusza"
//             />
//           </LabeledInput>
//           {/* eszközbiztosítás */}
//           <LabeledInput
//             label="Eszközbiztosítás státusza:"
//             className="w-full col-span-1 flex flex-col"
//             contentCls="w-full">
//             <input
//               name="device_insurance"
//               options={boolFields}
//               value={
//                 asset.device_reg_page
//                   ? boolFields[0]?.label
//                   : boolFields[1]?.label
//               }
//               placeholder="Eszközbiztosítás státusza"
//               readOnly
//             />
//           </LabeledInput>

//           <button
//             className="w-full bg-blue-600 text-white px-4 py-2 mt-4 rounded hover:bg-blue-700 col-span-4"
//             onClick={handleSave}>
//             Mentés
//           </button>
//         </div>
//       )}
//     </div>
//   );
// };

// export default InventoryElement;
