import React, { useEffect, useState } from "react";
import ApiCall from "@/api/api-calls";
import { toast } from "react-hot-toast";
import { useModalManager } from "@/hooks/useModalManager";
import MediaPreview from "../shared/mediaobject/MediaPreview";
import Modal from "../shared/modal/Modal";
import Select from "react-select";

const FilingDocumentTab = ({
  filingId,
  description = "Dokumentum",
  tabLabel = "Dokumentumok",
  documentTypeIds = [],
  defaultDocumentTypeId = null,
}) => {
  const [docs, setDocs] = useState([]);
  const [documentTypes, setDocumentTypes] = useState([]);
  const [selectedDocumentType, setSelectedDocumentType] = useState(null);
  const [uploadDocumentType, setUploadDocumentType] = useState(null);
  const [file, setFile] = useState(null);
  const [loading, setLoading] = useState(false);
  const { toggleModal, isModalOpen } = useModalManager();
  const [selectedDoc, setSelectedDoc] = useState(null);
  const [docUrl, setDocUrl] = useState(null);
  const [noteDraft, setNoteDraft] = useState("");

  const fetchDocuments = async () => {
    try {
      const types = await ApiCall.documents.getDocumentTypes();
      setDocumentTypes(types.data);

      const { data } = await ApiCall.filing.listDocuments(filingId);

      const filteredDocs = documentTypeIds.length
        ? data.filter((doc) => documentTypeIds.includes(doc.document_type_id))
        : data;

      setDocs(filteredDocs);
    } catch {
      toast.error("Hiba a dokumentumok betöltésekor");
    }
  };

  useEffect(() => {
    fetchDocuments();
  }, [filingId, documentTypeIds]);

  useEffect(() => {
    setNoteDraft(selectedDoc?.note || "");
  }, [selectedDoc]);

  useEffect(() => {
    if (selectedDoc && documentTypes.length) {
      const match = documentTypes.find(
        (opt) => opt.id === selectedDoc.document_type_id
      );
      setSelectedDocumentType(match || null);
    }
  }, [selectedDoc, documentTypes]);

  const fetchDocumentUrl = async (docId) => {
    try {
      const res = await ApiCall.documents.downloadFilingDocument(docId);
      const blob = new Blob([res.data], { type: selectedDoc?.mime_type });
      const url = URL.createObjectURL(blob);
      setDocUrl(url);
    } catch {
      toast.error("Nem sikerült betölteni a dokumentumot");
    }
  };

  useEffect(() => {
    if (selectedDoc?.mime_type === "application/pdf") {
      fetchDocumentUrl(selectedDoc.id);
    } else {
      setDocUrl(null);
    }
  }, [selectedDoc]);

  const handleUpload = async () => {
    if (!file) return;
    const formData = new FormData();
    formData.append("file", file);
    formData.append("description", description);
    formData.append(
      "document_type_id",
      uploadDocumentType?.id || defaultDocumentTypeId
    );

    try {
      await ApiCall.filing.uploadDocument(filingId, formData);
      toast.success("Sikeres feltöltés");
      setFile(null);
      setUploadDocumentType(null);
      fetchDocuments();
    } catch {
      toast.error("Sikertelen feltöltés");
    }
  };

  const handleDelete = async (docId) => {
    try {
      await ApiCall.documents.removeFilingDocument(filingId, docId);
      toast.success("Dokumentum törölve");
      toggleModal("showDeleteModal");
      fetchDocuments();
    } catch {
      toast.error("Törlés sikertelen");
    }
  };

  return (
    <div className="w-full p-4">
      <h2 className="font-bold mb-2">{tabLabel}</h2>

      {/* Upload */}
      <div
        className="border-2 border-dashed border-gray-400 p-6 rounded-lg text-center cursor-pointer hover:border-blue-500 transition-colors"
        onDragOver={(e) => e.preventDefault()}
        onDrop={(e) => {
          e.preventDefault();
          if (e.dataTransfer.files.length > 0) setFile(e.dataTransfer.files[0]);
        }}
        onClick={() => document.getElementById("fileInput").click()}>
        <p className="text-gray-600">
          Húzza ide a fájlt vagy kattintson a kiválasztáshoz
        </p>
        {file && <p className="mt-2 font-medium text-sm">{file.name}</p>}
        <input
          type="file"
          id="fileInput"
          className="hidden"
          onChange={(e) => setFile(e.target.files[0])}
        />
      </div>

      {/* Optional doc type picker */}
      {documentTypeIds.length > 1 && (
        <div className="mt-2">
          <Select
            className="w-full"
            placeholder="Típus kiválasztása"
            options={documentTypes.filter((dt) =>
              documentTypeIds.includes(dt.id)
            )}
            value={uploadDocumentType}
            onChange={setUploadDocumentType}
          />
        </div>
      )}

      <button
        className="w-full bg-green-500 text-white px-4 py-2 mt-2 rounded disabled:opacity-50 disabled:bg-gray-500"
        onClick={handleUpload}
        disabled={
          !file ||
          loading ||
          (documentTypeIds.length > 1 && !uploadDocumentType)
        }>
        Feltöltés
      </button>

      {/* Document List */}
      <div className="w-full flex flex-col my-4 py-4 border-t border-b border-gray-400">
        <h2 className="font-semibold">Feltöltött dokumentumok:</h2>
        <ul className="list-disc pl-6 py-2">
          {docs.map((doc) => (
            <li key={doc.id} className="w-full flex justify-evenly mb-1">
              {doc.title || doc.file_name} ({(doc.file_size / 1024).toFixed(1)}{" "}
              KB)
              <button
                className="ml-4 text-blue-600 font-semibold cursor-pointer"
                onClick={() => {
                  setSelectedDoc(doc);
                  toggleModal("showDetailsModal");
                }}>
                Részletek
              </button>
              <button
                className="ml-auto text-red-600 font-semibold cursor-pointer"
                onClick={() => {
                  setSelectedDoc(doc);
                  toggleModal("showDeleteModal");
                }}>
                Törlés
              </button>
            </li>
          ))}
        </ul>
      </div>

      <Modal
        title={`${selectedDoc?.title || selectedDoc?.file_name || ""} törlése`}
        show={isModalOpen("showDeleteModal")}
        onClose={() => {
          toggleModal("showDeleteModal");
          setSelectedDoc(null);
        }}
        height="auto"
        width="max-content">
        <div className="p-4 w-full flex justify-evenly items-center">
          <button
            className="mr-2 text-red-500"
            onClick={() => handleDelete(selectedDoc.id)}>
            Csatolmány eltávolítása
          </button>
          <button
            className="ml-2 text-gray-500"
            onClick={() => toggleModal("showDeleteModal")}>
            Mégse
          </button>
        </div>
      </Modal>

      {/* Details Modal */}
      <Modal
        title="Dokumentum részletei"
        show={isModalOpen("showDetailsModal")}
        onClose={() => {
          toggleModal("showDetailsModal");
          setSelectedDoc(null);
          setDocUrl(null);
        }}
        height="auto"
        width="max-content">
        {selectedDoc ? (
          <div className="space-y-2 text-sm">
            <div className="w-full grid grid-cols-4 mx-2">
              {selectedDoc?.mime_type === "application/msword" && (
                <iframe
                  src={`https://docs.google.com/gview?url=${encodeURIComponent(
                    `${docUrl}`
                  )}&embedded=true`}
                  className="w-full h-auto min-h-[800px] col-span-2"
                  frameBorder="0"
                />
              )}
              {selectedDoc?.mime_type ===
                "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet" && (
                <iframe
                  src={`https://docs.google.com/gview?url=${encodeURIComponent(
                    `${docUrl}`
                  )}&embedded=true`}
                  className="w-full h-auto min-h-[800px] col-span-2"
                  frameBorder="0"
                />
              )}
              {selectedDoc?.mime_type === "application/pdf" && (
                <MediaPreview
                  data={selectedDoc}
                  index={0}
                  preview={docUrl}
                  className="w-full h-auto min-h-[800px] col-span-2"
                />
              )}

              <div className="w-full col-span-2 px-4 py-2">
                <p className="py-2">
                  <strong>Fájlnév:</strong> {selectedDoc.title}
                </p>
                <p className="py-2">
                  <strong>Feltöltve:</strong>{" "}
                  {new Date(selectedDoc.upload_date).toLocaleString()}
                </p>
                <p className="py-2">
                  <strong>Feltöltő:</strong> {selectedDoc.uploaded_by_username}
                </p>
                <Select
                  options={documentTypes}
                  className="w-full"
                  value={selectedDocumentType}
                  onChange={(val) => {
                    setSelectedDocumentType(val);
                    setSelectedDoc((prev) => ({
                      ...prev,
                      document_type_id: val ? val.id : null,
                    }));
                  }}
                />
                <div className="py-2">
                  <label className="font-semibold block mb-1">Leírás:</label>
                  <textarea
                    className="w-full border p-2 rounded"
                    rows={2}
                    value={noteDraft}
                    onChange={(e) => setNoteDraft(e.target.value)}
                  />
                </div>

                <a
                  className="text-blue-600 underline py-2 cursor-pointer"
                  onClick={async () => {
                    await fetchDocumentUrl(selectedDoc.id);
                    if (docUrl) {
                      const link = document.createElement("a");
                      link.href = docUrl;
                      link.download = selectedDoc.title || "document.pdf";
                      link.click();
                    }
                  }}>
                  Letöltés
                </a>

                <button
                  className="w-full cursor-pointer mt-4 bg-blue-600 text-white px-4 py-2 rounded"
                  onClick={async () => {
                    try {
                      await ApiCall.documents.updateFilingMetadata(
                        selectedDoc.id,
                        {
                          document_type_id:
                            selectedDocumentType?.id ||
                            selectedDoc.document_type_id,
                          description: noteDraft,
                        }
                      );
                      toast.success("Frissítve");
                      fetchDocuments();
                      toggleModal("showDetailsModal");
                    } catch {
                      toast.error("Frissítés sikertelen");
                    }
                  }}>
                  Mentés
                </button>
              </div>
            </div>
          </div>
        ) : (
          <p>Nincs kiválasztott dokumentum.</p>
        )}
      </Modal>
    </div>
  );
};

export default FilingDocumentTab;

// import React, { useEffect, useState } from "react";
// import ApiCall from "@/api/api-calls";
// import { toast } from "react-hot-toast";
// import { useModalManager } from "@/hooks/useModalManager";
// import MediaPreview from "../shared/mediaobject/MediaPreview";
// import Modal from "../shared/modal/Modal";
// import Select from "react-select";

// const FilingDocumentTab = ({
//   filingId,
//   description = "Dokumentum",
//   tabLabel = "Dokumentumok",
//   documentTypeId = null,
// }) => {
//   const [docs, setDocs] = useState([]);
//   const [documentTypes, setDocumentTypes] = useState([]);
//   const [selectedDocumentType, setSelectedDocumentType] = useState(null);
//   const [file, setFile] = useState(null);
//   const [loading, setLoading] = useState(false);
//   const { toggleModal, isModalOpen } = useModalManager();
//   const [selectedDoc, setSelectedDoc] = useState(null);
//   const [docUrl, setDocUrl] = useState(null); // <-- holds object URL
//   const [noteDraft, setNoteDraft] = useState("");
//   const fetchDocuments = async () => {
//     try {
//       const types = await ApiCall.documents.getDocumentTypes();
//       setDocumentTypes(types.data);

//       const { data } = await ApiCall.filing.listDocuments(filingId);

//       // ✅ Filter docs by the given documentTypeId, if provided
//       const filteredDocs = documentTypeId
//         ? data.filter((doc) => doc.document_type_id === documentTypeId)
//         : data;

//       setDocs(filteredDocs);
//     } catch {
//       toast.error("Hiba a dokumentumok betöltésekor");
//     }
//   };

//   useEffect(() => {
//     fetchDocuments();
//   }, [filingId, documentTypeId]);

//   useEffect(() => {
//     setNoteDraft(selectedDoc?.note || "");
//   }, [selectedDoc]);

//   useEffect(() => {
//     if (selectedDoc && documentTypes.length) {
//       const match = documentTypes.find(
//         (opt) => opt.id === selectedDoc.document_type_id
//       );
//       setSelectedDocumentType(match || null);
//     }
//   }, [selectedDoc, documentTypes]);

//   const fetchDocumentUrl = async (docId) => {
//     try {
//       const res = await ApiCall.documents.downloadFilingDocument(docId);
//       const blob = new Blob([res.data], { type: selectedDoc?.mime_type });
//       const url = URL.createObjectURL(blob);
//       setDocUrl(url);
//     } catch {
//       toast.error("Nem sikerült betölteni a dokumentumot");
//     }
//   };

//   useEffect(() => {
//     if (selectedDoc?.mime_type === "application/pdf") {
//       fetchDocumentUrl(selectedDoc.id);
//     } else {
//       setDocUrl(null);
//     }
//   }, [selectedDoc]);

//   const handleUpload = async () => {
//     if (!file) return;
//     const formData = new FormData();
//     formData.append("file", file);
//     formData.append("description", description);
//     if (documentTypeId) formData.append("document_type_id", documentTypeId);

//     try {
//       await ApiCall.filing.uploadDocument(filingId, formData);
//       toast.success("Sikeres feltöltés");
//       setFile(null);
//       fetchDocuments();
//     } catch {
//       toast.error("Sikertelen feltöltés");
//     }
//   };

//   const handleDelete = async (docId) => {
//     try {
//       await ApiCall.documents.removeFilingDocument(filingId, docId);
//       toast.success("Dokumentum törölve");
//       toggleModal("showDeleteModal");
//       fetchDocuments();
//     } catch {
//       toast.error("Törlés sikertelen");
//     }
//   };

//   return (
//     <div className="w-full p-4 ">
//       <h2 className="font-bold mb-2">{tabLabel}</h2>

//       {/* Upload */}
//       <div
//         className="border-2 border-dashed border-gray-400 p-6 rounded-lg text-center cursor-pointer hover:border-blue-500 transition-colors"
//         onDragOver={(e) => e.preventDefault()}
//         onDrop={(e) => {
//           e.preventDefault();
//           if (e.dataTransfer.files.length > 0) setFile(e.dataTransfer.files[0]);
//         }}
//         onClick={() => document.getElementById("fileInput").click()}>
//         <p className="text-gray-600">
//           Húzza ide a fájlt vagy kattintson a kiválasztáshoz
//         </p>
//         {file && <p className="mt-2 font-medium text-sm">{file.name}</p>}
//         <input
//           type="file"
//           id="fileInput"
//           className="hidden"
//           onChange={(e) => setFile(e.target.files[0])}
//         />
//       </div>

//       <button
//         className="w-full bg-green-500 text-white px-4 py-2 mt-2 rounded disabled:opacity-50 disabled:bg-gray-500"
//         onClick={handleUpload}
//         disabled={!file || loading}>
//         Feltöltés
//       </button>

//       {/* Document List */}
//       <div className="w-full flex flex-col my-4 py-4 border-t border-b border-gray-400">
//         <h2 className="font-semibold">Feltöltött dokumentumok:</h2>
//         <ul className="list-disc pl-6 py-2">
//           {docs.map((doc) => (
//             <li key={doc.id} className="w-full flex justify-evenly mb-1">
//               {doc.title || doc.file_name} ({(doc.file_size / 1024).toFixed(1)}{" "}
//               KB)
//               <button
//                 className="ml-4 text-blue-600 font-semibold cursor-pointer"
//                 onClick={() => {
//                   setSelectedDoc(doc);
//                   toggleModal("showDetailsModal");
//                 }}>
//                 Részletek
//               </button>
//               <button
//                 className="ml-auto text-red-600 font-semibold cursor-pointer"
//                 onClick={() => {
//                   setSelectedDoc(doc);
//                   toggleModal("showDeleteModal");
//                 }}>
//                 Törlés
//               </button>
//             </li>
//           ))}
//         </ul>
//       </div>

//       {/* Delete Modal */}
//       <Modal
//         title={`${selectedDoc?.title || selectedDoc?.file_name || ""} törlése`}
//         show={isModalOpen("showDeleteModal")}
//         onClose={() => {
//           toggleModal("showDeleteModal");
//           setSelectedDoc(null);
//         }}>
//         <div className="p-4 w-full flex justify-evenly items-center">
//           <button
//             className="mr-2 text-red-500"
//             onClick={() => handleDelete(selectedDoc.id)}>
//             Törlés
//           </button>
//           <button
//             className="ml-2 text-gray-500"
//             onClick={() => toggleModal("showDeleteModal")}>
//             Mégse
//           </button>
//         </div>
//       </Modal>

//       {/* Details Modal */}
//       <Modal
//         title="Dokumentum részletei"
//         show={isModalOpen("showDetailsModal")}
//         onClose={() => {
//           toggleModal("showDetailsModal");
//           setSelectedDoc(null);
//           setDocUrl(null);
//         }}
//         height="auto">
//         {selectedDoc ? (
//           <div className="space-y-2 text-sm">
//             <div className="w-full grid grid-cols-4 mx-2">
//               {selectedDoc?.mime_type === "application/msword" && (
//                 <iframe
//                   src={`https://docs.google.com/gview?url=${encodeURIComponent(
//                     `${docUrl}`
//                   )}&embedded=true`}
//                   className="w-full h-auto min-h-[800px] col-span-2"
//                   frameBorder="0"
//                 />
//               )}
//               {selectedDoc?.mime_type ===
//                 "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet" && (
//                 <iframe
//                   src={`https://docs.google.com/gview?url=${encodeURIComponent(
//                     `${docUrl}`
//                   )}&embedded=true`}
//                   className="w-full h-auto min-h-[800px] col-span-2"
//                   frameBorder="0"
//                 />
//               )}
//               {selectedDoc?.mime_type === "application/pdf" && (
//                 <MediaPreview
//                   data={selectedDoc}
//                   index={0}
//                   preview={docUrl}
//                   className="w-full h-auto min-h-[800px] col-span-2"
//                 />
//               )}

//               <div className="w-full col-span-2 px-4 py-2">
//                 <p className="py-2">
//                   <strong>Fájlnév:</strong> {selectedDoc.title}
//                 </p>
//                 <p className="py-2">
//                   <strong>Feltöltve:</strong>{" "}
//                   {new Date(selectedDoc.upload_date).toLocaleString()}
//                 </p>
//                 <p className="py-2">
//                   <strong>Feltöltő:</strong> {selectedDoc.uploaded_by_username}
//                 </p>
//                 <Select
//                   options={documentTypes}
//                   className="w-full"
//                   value={selectedDocumentType}
//                   onChange={(val) => {
//                     setSelectedDocumentType(val);
//                     setSelectedDoc((prev) => ({
//                       ...prev,
//                       document_type_id: val ? val.id : null,
//                     }));
//                   }}
//                 />
//                 <div className="py-2">
//                   <label className="font-semibold block mb-1">Leírás:</label>
//                   <textarea
//                     className="w-full border p-2 rounded"
//                     rows={2}
//                     value={noteDraft}
//                     onChange={(e) => setNoteDraft(e.target.value)}
//                   />
//                 </div>

//                 <a
//                   className="text-blue-600 underline py-2 cursor-pointer"
//                   onClick={async () => {
//                     await fetchDocumentUrl(selectedDoc.id);
//                     if (docUrl) {
//                       const link = document.createElement("a");
//                       link.href = docUrl;
//                       link.download = selectedDoc.title || "document.pdf";
//                       link.click();
//                     }
//                   }}>
//                   Letöltés
//                 </a>

//                 <button
//                   className="w-full cursor-pointer mt-4 bg-blue-600 text-white px-4 py-2 rounded"
//                   onClick={async () => {
//                     try {
//                       await ApiCall.documents.updateFilingMetadata(
//                         selectedDoc.id,
//                         {
//                           document_type_id:
//                             selectedDocumentType?.id ||
//                             selectedDoc.document_type_id,
//                           description: noteDraft,
//                         }
//                       );
//                       toast.success("Frissítve");
//                       fetchDocuments();
//                       toggleModal("showDetailsModal");
//                     } catch {
//                       toast.error("Frissítés sikertelen");
//                     }
//                   }}>
//                   Mentés
//                 </button>
//               </div>
//             </div>
//           </div>
//         ) : (
//           <p>Nincs kiválasztott dokumentum.</p>
//         )}
//       </Modal>
//     </div>
//   );
// };

// export default FilingDocumentTab;
