"use client";
import { useEffect, useRef, useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import useDynamicManager from "@/store/managers/useDynamicManager";

const getLogColor = (text) => {
  if (text.includes("✅")) return "text-green-400";
  if (text.includes("❌")) return "text-red-400";
  if (text.includes("🚀")) return "text-blue-400";
  if (text.includes("⚠️")) return "text-yellow-400";
  if (text.includes("✂️")) return "text-red-400";
  return "text-gray-300";
};

const getLogType = (text) => {
  if (text.includes("✅")) return "success";
  if (text.includes("❌")) return "error";
  if (text.includes("🚀")) return "start";
  if (text.includes("⚠️")) return "warn";
  if (text.includes("✂️")) return "delete";
  return "info";
};

const UploadLogPanel = () => {
  const { state, clearLogs } = useDynamicManager("uploadLog");
  const logs = state?.logs ?? [];

  const [isOpen, setIsOpen] = useState(true);
  const [filter, setFilter] = useState("all");

  const scrollContainerRef = useRef(null);
  const bottomRef = useRef(null);
  const [isAutoScroll, setIsAutoScroll] = useState(true);

  const filteredLogs =
    filter === "all"
      ? state?.logs
      : state?.logs.filter((log) => getLogType(log.text) === filter);

  useEffect(() => {
    const el = scrollContainerRef.current;
    if (!el) return;

    const scrollToBottom = () => {
      if (isAutoScroll) {
        el.scrollTop = el.scrollHeight;
      }
    };

    // Use requestAnimationFrame to wait until layout settles
    requestAnimationFrame(scrollToBottom);
  }, [filteredLogs.length, isOpen, isAutoScroll]);

  return (
    <div className="bg-black text-sm font-mono rounded-lg shadow-lg border border-gray-700 my-4">
      {/* Header */}
      <div className="flex justify-between items-center px-4 py-2 bg-gray-800 text-white rounded-t-lg">
        <span>📦 Feltöltési napló</span>
        <div className="flex items-center gap-2">
          <select
            className="bg-gray-700 text-white text-xs px-2 py-1 rounded"
            value={filter}
            onChange={(e) => setFilter(e.target.value)}>
            <option value="all">Összes</option>
            <option value="success">✅ Sikeres</option>
            <option value="error">❌ Hibák</option>
            <option value="start">🚀 Indítás</option>
            <option value="info">ℹ️ Info</option>
            <option value="warn">⚠️ Figyelmeztetés</option>
            <option value="delete">✂️ Törölt elem</option>
          </select>
          <button
            onClick={clearLogs}
            className="text-xs text-red-400 hover:text-red-300 underline">
            Törlés
          </button>
          <button
            onClick={() => setIsOpen(!isOpen)}
            className="text-xs underline hover:text-blue-300">
            {isOpen ? "Bezár" : "Megnyit"}
          </button>
        </div>
      </div>

      {/* Body */}
      <AnimatePresence>
        {isOpen && (
          <div
            ref={scrollContainerRef}
            className="overflow-y-auto max-h-64 px-4 py-2 transition-all duration-300">
            <AnimatePresence initial={false}>
              {filteredLogs.map((log, idx) => (
                <motion.div
                  key={idx}
                  initial={{ opacity: 0, y: 4 }}
                  animate={{ opacity: 1, y: 0 }}
                  exit={{ opacity: 0 }}
                  transition={{ duration: 0.2 }}
                  className={`${getLogColor(log.text)} break-words`}>
                  {log.text}
                </motion.div>
              ))}
            </AnimatePresence>

            {/* Always-scroll-to-bottom anchor */}
            <div ref={bottomRef} />
          </div>
        )}
      </AnimatePresence>
    </div>
  );
};

export default UploadLogPanel;
