"use client";

import { useEffect, useMemo, useState, useCallback } from "react";
import Select from "react-select";
import AsyncSelect from "react-select/async";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";

// export type EmailUpsertPayload = {
//   email: string;
//   type?: string | null;

//   handler_person?: string | null; // UUID
//   handler_segment?: string | null; // UUID
//   handler?: string | null; // optional legacy

//   user_status?: string | null; // "active" | "quit"
//   groups?: string[];
//   is_forwarded?: boolean;
//   forward_to?: string[] | null;
//   should_archive?: boolean;
//   notes?: string | null;

//   receive_status?: "allow" | "suspended" | null;
//   send_status?: "allow" | "suspended" | "hold" | null;
//   login_status?: "allow" | "suspended" | null;
// };
export type EmailUpsertPayload = {
  email: string;
  type?: string | null;

  // NEW: multiple handlers
  handler_people_ids?: string[]; // UUID[]
  handler_primary_person_id?: string | null; // UUID | null

  // keep segment if you still want it
  handler_segment?: string | null; // UUID
  handler?: string | null; // optional legacy (ha még kell)

  user_status?: string | null; // "active" | "quit"
  groups?: string[];
  is_forwarded?: boolean;
  forward_to?: string[] | null;
  should_archive?: boolean;
  notes?: string | null;

  receive_status?: "allow" | "suspended" | null;
  send_status?: "allow" | "suspended" | "hold" | null;
  login_status?: "allow" | "suspended" | null;
};

type Option = { value: string; label: string };

export default function EmailRegistryForm({
  initial,
  onSubmit,
}: {
  initial?: any; // we'll read fields defensively
  onSubmit: (payload: EmailUpsertPayload) => void | Promise<void>;
}) {
  // basic fields
  const [email, setEmail] = useState(initial?.email ?? "");
  const [type, setType] = useState<string | null>(initial?.type ?? null);

  // handler person/segment (UUIDs from backend)
  const [handlerPeople, setHandlerPeople] = useState<Option[]>(() => {
    // prefer new fields
    if (
      Array.isArray(initial?.handler_people_ids) &&
      initial.handler_people_ids.length
    ) {
      const names = Array.isArray(initial?.handler_people)
        ? initial.handler_people
        : [];
      return initial.handler_people_ids.map((id: string, i: number) => ({
        value: id,
        label: names[i] ?? id,
      }));
    }

    // fallback legacy primary
    const legacyId =
      initial?.handler_person ?? initial?.handler_person_id ?? null;
    if (legacyId)
      return [{ value: legacyId, label: initial?.handler_person ?? legacyId }];

    return [];
  });

  const [handlerPrimaryId, setHandlerPrimaryId] = useState<string | null>(
    () => {
      return (
        initial?.handler_primary_person_id ?? initial?.handler_person_id ?? null
      );
    }
  );

  const [handlerSegmentId, setHandlerSegmentId] = useState<string | null>(
    initial?.handler_segment ?? initial?.handler_segment_id ?? null
  );

  const [groupOptions, setGroupOptions] = useState<Option[]>([]);
  const [typeOptions, setTypeOptions] = useState<Option[]>([]);
  const [handlerPersonOptions, setHandlerPersonOptions] = useState<Option[]>(
    []
  );
  const [handlerSegmentOptions, setHandlerSegmentOptions] = useState<Option[]>(
    []
  );

  const [groupsSel, setGroupsSel] = useState<Option[]>(
    (initial?.groups ?? []).map((g: string) => ({ value: g, label: g }))
  );

  const [isForwarded, setIsForwarded] = useState<boolean>(
    !!initial?.is_forwarded
  );
  const [forwardTo, setForwardTo] = useState<string[]>(
    Array.isArray(initial?.forward_to)
      ? initial!.forward_to!
      : initial?.forward_to
      ? [initial!.forward_to as any]
      : []
  );

  const [receiveStatus, setReceiveStatus] = useState<"allow" | "suspended">(
    (initial?.receive_status as any) ?? "allow"
  );
  const [sendStatus, setSendStatus] = useState<"allow" | "suspended" | "hold">(
    (initial?.send_status as any) ?? "allow"
  );
  const [loginStatus, setLoginStatus] = useState<"allow" | "suspended">(
    (initial?.login_status as any) ?? "allow"
  );

  const [shouldArchive, setShouldArchive] = useState<boolean>(
    !!initial?.should_archive
  );
  const [notes, setNotes] = useState<string>(initial?.notes ?? "");

  // user_status: canonical "active"/"quit"
  const [userStatus, setUserStatus] = useState<string | null>(
    initial?.user_status ?? null
  );
  const [statusOptions, setStatusOptions] = useState<Option[]>([]);

  const receiveOptions: Option[] = [
    { value: "allow", label: "Engedélyezve" },
    { value: "suspended", label: "Felfüggesztve" },
  ];
  const sendOptions: Option[] = [
    { value: "allow", label: "Engedélyezve" },
    { value: "suspended", label: "Felfüggesztve" },
    { value: "hold", label: "Visszatartva" },
  ];
  const loginOptions: Option[] = [
    { value: "allow", label: "Engedélyezve" },
    { value: "suspended", label: "Felfüggesztve" },
  ];

  // Load static options (types, groups, statuses, handler persons/segments)
  useEffect(() => {
    (async () => {
      const [typesRes, groupsRes, statusesRes, personsRes, segmentsRes] =
        await Promise.all([
          ApiCall.emailRegistry.getTypes(),
          ApiCall.emailRegistry.getGroups(),
          ApiCall.emailRegistry.getHandlerStatuses(),
          ApiCall.emailRegistry.getHandlerPersons(),
          ApiCall.emailRegistry.getHandlerSegments(),
        ]);

      // 🔹 types: use value/label from backend
      setTypeOptions(
        (typesRes.data?.items ?? []).map((t: any) => ({
          value: t.value, // id from backend
          label: t.label, // human name
        }))
      );

      // 🔹 groups
      setGroupOptions(
        (groupsRes.data?.items ?? []).map((g: any) => ({
          value: g.value,
          label: g.label,
        }))
      );

      // 🔹 statuses: backend already sends canonical + HU label
      setStatusOptions(
        (statusesRes.data?.items ?? []).map((s: any) => ({
          value: s.value, // "active"|"quit"
          label: s.label, // "Aktív"/"Kilépett"
        }))
      );

      // 🔹 handler persons: UUID + display name
      setHandlerPersonOptions(
        (personsRes.data?.items ?? []).map((p: any) => ({
          value: p.value, // UUID
          label: p.label, // display_name
        }))
      );

      // 🔹 handler segments: UUID + name
      setHandlerSegmentOptions(
        (segmentsRes.data?.items ?? []).map((s: any) => ({
          value: s.value, // UUID
          label: s.label, // name
        }))
      );
    })();
  }, []);

  // Hydrate from initial when record changes
  useEffect(() => {
    if (!initial) return;

    setEmail(initial.email ?? "");
    setType(initial.type ?? null);

    // handlers hydrate (prefer new)
    if (
      Array.isArray(initial?.handler_people_ids) &&
      initial.handler_people_ids.length
    ) {
      const names = Array.isArray(initial?.handler_people)
        ? initial.handler_people
        : [];
      const opts = initial.handler_people_ids.map((id: string, i: number) => ({
        value: id,
        label: names[i] ?? id,
      }));
      setHandlerPeople(opts);
      setHandlerPrimaryId(
        initial?.handler_primary_person_id ?? opts?.[0]?.value ?? null
      );
    } else {
      const legacyId =
        initial?.handler_person ?? initial?.handler_person_id ?? null;
      setHandlerPeople(
        legacyId
          ? [{ value: legacyId, label: initial?.handler_person ?? legacyId }]
          : []
      );
      setHandlerPrimaryId(
        initial?.handler_primary_person_id ?? legacyId ?? null
      );
    }

    setHandlerSegmentId(
      initial.handler_segment ?? initial.handler_segment_id ?? null
    );

    setGroupsSel(
      (initial.groups ?? []).map((g: string) => ({ value: g, label: g }))
    );
    setIsForwarded(!!initial.is_forwarded);

    const arr = Array.isArray(initial.forward_to)
      ? initial.forward_to
      : initial?.forward_to
      ? [initial.forward_to as any]
      : [];
    setForwardTo(arr);

    setShouldArchive(!!initial.should_archive);
    setNotes(initial.notes ?? "");

    const norm = (v: any) =>
      String(v ?? "")
        .trim()
        .toLowerCase();

    const rcv = norm(initial.receive_status);
    setReceiveStatus(rcv === "suspended" ? "suspended" : "allow");

    const snd = norm(initial.send_status);
    setSendStatus(
      snd === "hold" ? "hold" : snd === "suspended" ? "suspended" : "allow"
    );

    const lgn = norm(initial.login_status);
    setLoginStatus(lgn === "suspended" ? "suspended" : "allow");

    const us = norm(initial.user_status);
    setUserStatus(us === "quit" ? "quit" : us === "active" ? "active" : null);
  }, [initial]);

  // loader for AsyncSelect (forward_to)
  const loadForwardOptions = useCallback(async (inputValue: string) => {
    const q = inputValue?.trim() ?? "";
    const res = await ApiCall.emailRegistry.emailOptions(q);
    return (res?.data?.items ?? []).map((o: any) => ({
      value: o.value,
      label: o.label,
    }));
  }, []);

  const canSubmit = useMemo(() => email.trim().length > 3, [email]);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    const payload: EmailUpsertPayload = {
      email,
      type: type ?? undefined,

      handler_people_ids: handlerPeople.map((x) => x.value),
      handler_primary_person_id: handlerPrimaryId ?? null,
      handler_segment: handlerSegmentId ?? null,

      // canonical user status ("active"/"quit") or undefined
      user_status: userStatus ?? undefined,

      groups: groupsSel.map((g) => g.value),
      is_forwarded: isForwarded,
      forward_to: isForwarded ? forwardTo : [],
      should_archive: shouldArchive,
      notes: notes || null,
      receive_status: receiveStatus,
      send_status: sendStatus,
      login_status: loginStatus,
    };
    await onSubmit(payload);
  };

  return (
    <form
      onSubmit={handleSubmit}
      className="grid grid-cols-1 md:grid-cols-2 gap-3">
      <div className="flex flex-col">
        <label className="text-xs text-gray-500">E-mail cím</label>
        <input
          className="border rounded px-3 py-2"
          placeholder="pl. support@example.com"
          value={email}
          onChange={(e) => setEmail(e.target.value)}
          required
        />
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">
          E-mail felhasználó státusza
        </label>
        <Select
          classNamePrefix="er"
          options={statusOptions}
          value={
            userStatus
              ? statusOptions.find((o) => o.value === userStatus) ?? null
              : null
          }
          onChange={(v) => setUserStatus((v as any)?.value ?? null)}
          isClearable
          placeholder="Válassz státuszt (Aktív/Kilépett)"
        />
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">E-mail cím típusa</label>
        <Select
          classNamePrefix="er"
          options={typeOptions}
          value={
            type ? typeOptions.find((o) => o.value === type) ?? null : null
          }
          onChange={(v) => setType((v as any)?.value ?? null)}
          isClearable
          placeholder="Válassz típust"
        />
      </div>

      {/* Kezelő személy & csoport as two React-Selects */}
      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Kezelő személy(ek)</label>
        <Select
          classNamePrefix="er"
          options={handlerPersonOptions}
          value={handlerPeople}
          onChange={(v) => setHandlerPeople((v as any) ?? [])}
          isClearable
          isMulti
          placeholder="Válassz kezelő személy(eke)t"
        />
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Fő kezelő</label>
        <Select
          classNamePrefix="er"
          options={handlerPeople}
          value={
            handlerPeople.find((o) => o.value === handlerPrimaryId) ?? null
          }
          onChange={(v) => setHandlerPrimaryId((v as any)?.value ?? null)}
          isClearable
          placeholder="Válaszd ki a fő kezelőt (opcionális)"
          noOptionsMessage={() => "Előbb válassz kezelő személy(eke)t"}
        />
        <span className="text-[11px] text-gray-500 mt-1">
          Ha üresen hagyod, a backend az első kiválasztott személyt veszi fő
          kezelőnek.
        </span>
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Kezelő csoport</label>
        <Select
          classNamePrefix="er"
          options={handlerSegmentOptions}
          value={
            handlerSegmentId
              ? handlerSegmentOptions.find(
                  (o) => o.value === handlerSegmentId
                ) ?? null
              : null
          }
          onChange={(v) => setHandlerSegmentId((v as any)?.value ?? null)}
          isClearable
          placeholder="Válassz kezelő csoportot"
        />
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Cégcsoport(ok)</label>
        <Select
          isMulti
          classNamePrefix="er"
          options={groupOptions}
          value={groupsSel}
          onChange={(v) => setGroupsSel((v as any) ?? [])}
          placeholder="Válassz cégcsoportot"
        />
      </div>

      {/* <div className="col-span-2 border-b border-b-1 border-b-gray-600 pb-4">
        <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
          <h4 className="text-sm text-gray-500 font-semibold col-span-2">
            Korlátozások
          </h4>
          <div className="flex flex-col">
            <label className="text-xs text-gray-500">
              Bejövő levél fogadása
            </label>
            <Select
              classNamePrefix="er"
              options={receiveOptions}
              value={receiveOptions.find((o) => o.value === receiveStatus)}
              onChange={(v) => setReceiveStatus((v as any)?.value ?? "allow")}
            />
          </div>

          <div className="flex flex-col">
            <label className="text-xs text-gray-500">Levél küldése</label>
            <Select
              classNamePrefix="er"
              options={sendOptions}
              value={sendOptions.find((o) => o.value === sendStatus)}
              onChange={(v) => setSendStatus((v as any)?.value ?? "allow")}
            />
          </div>

          <div className="flex flex-col">
            <label className="text-xs text-gray-500">
              Bejelentkezés postafiókba
            </label>
            <Select
              classNamePrefix="er"
              options={loginOptions}
              value={loginOptions.find((o) => o.value === loginStatus)}
              onChange={(v) => setLoginStatus((v as any)?.value ?? "allow")}
            />
          </div>
        </div>
      </div> */}

      <div className="col-span-2">
        <h4 className="text-sm text-gray-500 font-semibold col-span-2">
          Átirányítás
        </h4>
        <div className="flex flex-col">
          <div className="flex items-center gap-2 my-2">
            <input
              id="is_forwarded"
              className="w-[1.25rem] h-[1.25rem]"
              type="checkbox"
              checked={isForwarded}
              onChange={(e) => setIsForwarded(e.target.checked)}
            />
            <label htmlFor="is_forwarded" className="text-sm">
              Átirányítás bekapcsolása
            </label>
          </div>

          <div className="flex flex-col">
            <label className="text-xs text-gray-500">
              Átirányítva e-mail címre
            </label>
            <AsyncSelect
              classNamePrefix="er"
              isDisabled={!isForwarded}
              isMulti
              cacheOptions
              defaultOptions
              loadOptions={loadForwardOptions}
              value={forwardTo.map((v) => ({ value: v, label: v }))}
              onChange={(opts) =>
                setForwardTo(((opts as any[]) ?? []).map((o) => o.value))
              }
              placeholder={
                isForwarded ? "Válassz címeket…" : "Kapcsold be az átirányítást"
              }
              isClearable
              noOptionsMessage={() => "Nincs találat"}
            />
          </div>
        </div>
      </div>

      <div className="flex items-center gap-2 mt-2">
        <input
          id="should_archive"
          className="w-[1.25rem] h-[1.25rem]"
          type="checkbox"
          checked={shouldArchive}
          onChange={(e) => setShouldArchive(e.target.checked)}
        />
        <label htmlFor="should_archive" className="text-sm">
          Archiválni kell
        </label>
      </div>

      <div className="flex flex-col md:col-span-2">
        <label className="text-xs text-gray-500">Megjegyzés</label>
        <textarea
          className="border rounded px-3 py-2 min-h-24"
          value={notes}
          onChange={(e) => setNotes(e.target.value)}
        />
      </div>

      <div className="md:col-span-2 flex justify-end gap-2">
        <button
          type="submit"
          disabled={!canSubmit}
          className="px-4 py-2 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white disabled:opacity-50">
          Mentés
        </button>
      </div>
    </form>
  );
}
