import React, { useState, useRef, useCallback } from "react";
import { initialData } from "@/data/data";
import Select from "react-select";
import dynamic from "next/dynamic";
import "react-quill/dist/quill.snow.css";
import { useSelector, useDispatch } from "react-redux";
import {
  selectedElement,
  updateContainerContent,
} from "@/store/slices/pageEditorSlice";
import icons from "@/assets/icons/icons";
import useDynamicManager from "@/store/managers/useDynamicManager";

const ColorPicker = dynamic(() => import("react-best-gradient-color-picker"), {
  ssr: false,
});
import "suneditor/dist/css/suneditor.min.css";
import ModalButton from "@/components/shared/modal/ModalButton";
const SunEditor = dynamic(() => import("suneditor-react"), {
  ssr: false,
});

const Text = ({ toggleModal, modalName, type }) => {
  const { state, selectedElement, addcurrentElementToColumn } =
    useDynamicManager(type);
  const content = state?.current?.content?.[0]?.content?.[0] ?? {};

  const [color, setColor] = useState(
    content?.fontColor ?? "rgba(33, 33, 33, 1)"
  );
  const [backgroundColor, setBackgroundColor] = useState(
    content?.backgroundColor ?? "rgb(255, 255, 255, 0)"
  );

  const dispatch = useDispatch();
  const [textContent, setTextContent] = useState(content?.textContent ?? "");

  const inputRefs = useRef([]);
  let index = 0;

  const submitText = useCallback(() => {
    const content = {
      type: "text",
      title: "Szöveg",
      textContent: textContent,
      fontColor: color,
      backgroundColor: backgroundColor,
    };

    inputRefs.current.forEach((ref) => {
      if (!ref) return;
      if (ref.props?.name) {
        const value = Array.isArray(ref.props.value)
          ? ref.props.value[0]?.value
          : ref.props.value?.value;
        content[ref.props.name] = value;
      } else if (ref.name) {
        content[ref.name] = ref.value;
      }
    });

    selectedElement({
      content: content,
    });

    addcurrentElementToColumn();
    toggleModal(modalName);
    toggleModal("showSelectElementModal");
  }, [textContent, color, backgroundColor, inputRefs]);

  return (
    <div className="relative">
      <div className="text-wrapper flex flex-col w-full p-4">
        {/* textContent */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"textContent"}
            className="w-1/4 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap  pr-2">
            <p className="w-full font-bold">Szöveg: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Adjon meg tartalmat.
            </span>
          </label>
          <SunEditor
            lang="en"
            name="textContent"
            defaultValue={textContent}
            width="100%"
            height="100%"
            placeholder="Ide gépelheti a szöveget..."
            autoFocus={false}
            onChange={(value) => setTextContent(value)}
            setOptions={{
              buttonList: [
                ["font", "fontSize", "formatBlock"],
                [
                  "bold",
                  "underline",
                  "italic",
                  "strike",
                  "subscript",
                  "superscript",
                ],
                ["align", "list", "table"],
                ["fontColor", "hiliteColor"],
                ["outdent", "indent"],
                ["undo", "redo"],
                ["removeFormat"],
                ["link", "image"],
                ["preview", "print"],
                ["fullScreen", "showBlocks", "codeView"],
              ],
              tableWidth: "100%",
              tableStyles: {
                border: "1px solid black",
              },
            }}
          />
        </div>
        {/* text animation type */}
        <div className="flex flex-row w-full mb-4 p-4 border-b">
          <label
            htmlFor={"animationTypes"}
            className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Szöveg animációja: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Válassza ki a szöveg elem animációját. (alapértelmezett: Nincs
              animáció)
            </span>
          </label>

          <Select
            options={initialData.animation}
            className="w-3/4"
            menuPosition={"fixed"}
            placeholder={"Válasszon animáció típust"}
            backspaceRemovesValue={false}
            name={"animationTypes"}
            ref={(r) => (inputRefs.current[index++] = r)}
            defaultValue={
              content?.animationTypes
                ? initialData.animation.filter(
                    (x) => x.value === content?.animationTypes
                  )
                : initialData.animation[0]
            }
            onChange={(val, e) => {
              // second parameter shows action triggered
              // dispatch(selectParentPage({ parent: val }));
            }}
            required
          />
        </div>
        {/* text animation delay */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"animationDelay"}
            className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Szöveg animáció késleltetése: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Megadhatja, hogy a szöveg elemnek mennyi legyen az animáció
              késleltetése. Alapértelmezett: 1s
            </span>
          </label>
          <div className="flex flex-row flex-nowrap justify-evenly items-center">
            <input
              type="text"
              name="animationDelay"
              id="animationDelay"
              ref={(r) => (inputRefs.current[index++] = r)}
              className="w-3/4 border h-max p-2"
              defaultValue={content?.animationDelay ?? 1}
            />
          </div>
        </div>
        {/* text animation duration */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"animationDuration"}
            className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Szöveg animáció időtartama: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Megadhatja, hogy a szöveg elemnek mennyi legyen az animáció
              időtartama. (pl.: 0.5, 2, 4) Alapértelmezett: 3s
            </span>
          </label>
          <div className="flex flex-row flex-nowrap justify-evenly items-center">
            <input
              type="text"
              name="animationDuration"
              id="animationDuration"
              ref={(r) => (inputRefs.current[index++] = r)}
              className="w-3/4 border h-max p-2"
              defaultValue={content?.animationDuration ?? 3}
            />
          </div>
        </div>
        {/* font size xs-2xl */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"textType"}
            className="w-1/4 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Szöveg mérete: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Válassza ki, hogy milyen méretű legyen a szöveg elem tartalma -
              (alapértelmezett érték: Normál).
            </span>
          </label>
          <Select
            options={initialData.editor.text.textTypes}
            className="w-3/4"
            placeholder={"Válasszon szöveg méretet"}
            backspaceRemovesValue={false}
            name={"textType"}
            ref={(r) => (inputRefs.current[index++] = r)}
            defaultValue={
              content?.textType
                ? initialData.editor.text.textTypes.filter(
                    (x) => x.value === content?.textType
                  )
                : initialData.editor.text.textTypes[2]
            }
            onChange={(val, e) => {
              // second parameter shows action triggered
            }}
            required
          />
        </div>
        {/* text line height */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"textLineHeight"}
            className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Szöveg elem sortáv: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Adjon meg szöveg elem sortávot. Pl.: 28px vagy 1em, 2rem, stb.
            </span>
          </label>
          <input
            type="text"
            name="textLineHeight"
            id="textLineHeight"
            ref={(r) => (inputRefs.current[index++] = r)}
            className="w-3/4 border h-max p-2"
            defaultValue={content?.textLineHeight ?? "28px"}
          />
        </div>
        {/* text padding */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"textPadding"}
            className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Szöveg eltartás: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Megadhatja, hogy a szöveg elemnek mekkora eltartása legyen px-ben.
              Alapértelmezett: 0 0 0 0
            </span>
          </label>

          <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
            <span className="fill-gray-600 w-[2rem] h-[2rem]">{icons.up}</span>
            <input
              type="text"
              name="textPaddingTop"
              id="textPaddingTop"
              ref={(r) => (inputRefs.current[index++] = r)}
              className="w-3/4 border h-max p-2"
              defaultValue={content?.textPaddingTop ?? 0}
            />
          </div>
          <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
            <span className="fill-gray-600 w-[2rem] h-[2rem]">
              {icons.right}
            </span>
            <input
              type="text"
              name="textPaddingRight"
              id="textPaddingRight"
              ref={(r) => (inputRefs.current[index++] = r)}
              className="w-3/4 border h-max p-2"
              defaultValue={content?.textPaddingRight ?? 0}
            />
          </div>
          <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
            <span className="fill-gray-600 w-[2rem] h-[2rem]">
              {icons.down}
            </span>
            <input
              type="text"
              name="textPaddingBottom"
              id="textPaddingBottom"
              ref={(r) => (inputRefs.current[index++] = r)}
              className="w-3/4 border h-max p-2"
              defaultValue={content?.textPaddingBottom ?? 0}
            />
          </div>
          <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
            <span className="fill-gray-600 w-[2rem] h-[2rem]">
              {icons.left}
            </span>
            <input
              type="text"
              name="textPaddingLeft"
              id="textPaddingLeft"
              ref={(r) => (inputRefs.current[index++] = r)}
              className="w-3/4 border h-max p-2"
              defaultValue={content?.textPaddingLeft ?? 0}
            />
          </div>
        </div>
        {/* text content order */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"textOrder"}
            className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Szöveg tartalom elrendezés: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Válassza ki, hogy sor vagy oszlop legyen a tartalmi elrendezés.
              (alapértelmezett: Oszlop)
            </span>
          </label>
          <Select
            options={initialData?.editor?.text?.textOrder}
            className="w-3/4"
            menuPosition={"fixed"}
            placeholder={
              "Válassza ki, hogy sor vagy oszlop legyen a tartalmi elrendezés."
            }
            backspaceRemovesValue={false}
            name={"textOrder"}
            ref={(r) => (inputRefs.current[index++] = r)}
            defaultValue={
              content?.textOrder
                ? initialData?.editor?.text?.textOrder?.filter(
                    (x) => x.value === content?.textOrder
                  )
                : initialData?.editor?.text?.textOrder[1]
            }
            onChange={(val, e) => {
              // second parameter shows action triggered
              // dispatch(selectParentPage({ parent: val }));
            }}
            required
          />
        </div>
        {/* text alignment */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"textAlignment"}
            className="w-1/4 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap  pr-2">
            <p className="w-full font-bold">Szöveg elrendezés: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Válassza ki, hogy milyen elrendezésben legyen a szöveg elem -
              (alapértelmezett érték: Balra zárt).
            </span>
          </label>

          <Select
            options={initialData.editor.text.textAlignment}
            className="w-3/4"
            placeholder={"Válasszon szöveg elrendezést"}
            backspaceRemovesValue={false}
            name={"textAlignment"}
            ref={(r) => (inputRefs.current[index++] = r)}
            defaultValue={
              content?.textAlignment
                ? initialData.editor.text.textAlignment.filter(
                    (x) => x.value === content?.textAlignment
                  )
                : initialData.editor.text.textAlignment[0]
            }
            onChange={(val, e) => {
              // second parameter shows action triggered
              // dispatch(selectParentPage({ parent: val }));
            }}
            required
          />
        </div>
        {/* text height*/}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"textHeight"}
            className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Szöveg magasság: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Megadhatja, hogy a szöveg elemnek mekkora magassága legyen.
              Alapértelmezett: inherit. Lehetséges értékek: auto, fit-content,
              inherit, initial, max-content, min-content, revert, unset, px, vh,
              rem, em
            </span>
          </label>

          <div className="flex flex-row flex-nowrap justify-evenly items-center">
            <div className="flex flex-row flex-nowrap w-3/4 justify-evenly items-center max-h-10">
              <span className="fill-gray-600 w-[2rem] h-[2rem]">
                {icons.up}
                {icons.down}
              </span>
              <input
                type="text"
                name="textHeight"
                id="textHeight"
                ref={(r) => (inputRefs.current[index++] = r)}
                className="w-3/4 border h-max p-2"
                defaultValue={content?.textHeight ?? "inherit"}
              />
            </div>
          </div>
        </div>
        {/* text max height*/}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"maxHeight"}
            className="w-1/4 text-normal text-gray-400 whitespace-pre-line flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Szöveg maximum magasság: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Megadhatja, hogy a Szöveg elemnek mekkora maximum magassága
              legyen. Alapértelmezett: inherit. Lehetséges értékek: auto,
              fit-content, inherit, initial, max-content, min-content, revert,
              unset, px, vh, rem, em
            </span>
          </label>

          <div className="flex flex-row flex-nowrap justify-evenly items-center">
            <div className="flex flex-row flex-nowrap w-3/4 justify-evenly items-center max-h-10">
              <span className="fill-gray-600 w-[2rem] h-[2rem]">
                {icons.up}
                {icons.down}
              </span>
              <input
                type="text"
                name="maxHeight"
                id="maxHeight"
                ref={(r) => (inputRefs.current[index++] = r)}
                className="w-3/4 border h-max p-2"
                defaultValue={content?.maxHeight ?? "inherit"}
              />
            </div>
          </div>
        </div>
        {/* font family */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"fontFamily"}
            className="w-1/4 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap  pr-2">
            <p className="w-full font-bold">Betűtípus: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Válassza ki, hogy milyen betűtípust használjon a szöveg elem -
              (alapértelmezett érték: Open Sans).
            </span>
          </label>

          <Select
            options={initialData.editor.title.fontFamily}
            className="w-3/4"
            placeholder={"Válasszon betűtípust"}
            backspaceRemovesValue={false}
            name={"fontFamily"}
            ref={(r) => (inputRefs.current[index++] = r)}
            defaultValue={
              content?.fontFamily
                ? initialData.editor.title.fontFamily.filter(
                    (x) => x.value === content?.fontFamily
                  )
                : initialData.editor.title.fontFamily[0]
            }
            onChange={(val, e) => {
              // second parameter shows action triggered
              // dispatch(selectParentPage({ parent: val }));
            }}
            required
          />
        </div>
        {/* font color */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"fontColor"}
            className="w-1/4 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Betű szín: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Válassza ki, hogy milyen színű legyen a szöveg elem -
              (alapértelmezett érték: #212121 - Fekete).
            </span>
          </label>

          <div className="w-3/4 flex flex-row justify-center align-center">
            <ColorPicker value={color} onChange={setColor} />
          </div>
        </div>
        {/* background color */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"fontBackgroundColor"}
            className="w-1/4 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Háttér szín: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Válassza ki, hogy milyen színű legyen a cím elem háttere -
              (alapértelmezett érték: #fff - Fehér).
            </span>
          </label>

          <div className="w-3/4 flex flex-row justify-center align-center">
            <ColorPicker
              value={backgroundColor}
              onChange={setBackgroundColor}
            />
          </div>
        </div>
        {/* text position */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"textPosition"}
            className="w-1/4 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Szöveg elhelyezkedése: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Válassza ki, hogy a szöveg elem a tartalmazó dobozban milyen
              pozícióban helyezkedjen el - (alapértelmezett érték: Relatív).
            </span>
          </label>

          <Select
            options={initialData.editor.text.textPosition}
            className="w-3/4"
            placeholder={"Válasszon szöveg pozíciót"}
            backspaceRemovesValue={false}
            name={"textPosition"}
            menuPosition={"fixed"}
            ref={(r) => (inputRefs.current[index++] = r)}
            defaultValue={
              content?.textPosition
                ? initialData.editor.text.textPosition.filter(
                    (x) => x.value === content?.textPosition
                  )
                : initialData.editor.text.textPosition[0]
            }
            onChange={(val, e) => {
              // second parameter shows action triggered
              // dispatch(selectParentPage({ parent: val }));
            }}
            required
          />
        </div>
        {/* text top right bottom left - % */}
        <div className="flex flex-row w-full mb-4 pb-4 border-b">
          <label
            htmlFor={"textDistance"}
            className="w-1/4 text-normal text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
            <p className="w-full font-bold">Szöveg távolsága: </p>
            <span className="w-full text-sm text-gray-600 whitespace-pre-line">
              Válassza ki, hogy a szöveg elem a tartalmazó dobozban - a doboz
              négy sarkától milyen távolságra helyezkedjen el %-ban - ABSZOLÚT
              POZÍCIÓ ESETÉN (alapértelmezett érték: 0 0 0 0).
            </span>
          </label>

          <div className="flex flex-row flex-nowrap justify-evenly items-center">
            <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
              <span className="fill-gray-600 w-[2rem] h-[2rem]">
                {icons.up}
              </span>
              <input
                type="text"
                name="textPositionTop"
                id="textPositionTop"
                ref={(r) => (inputRefs.current[index++] = r)}
                className="w-3/4 border h-max p-2"
                defaultValue={content?.textPositionTop ?? 0}
              />
            </div>
            <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
              <span className="fill-gray-600 w-[2rem] h-[2rem]">
                {icons.right}
              </span>
              <input
                type="text"
                name="textPositionRight"
                id="textPositionRight"
                ref={(r) => (inputRefs.current[index++] = r)}
                className="w-3/4 border h-max p-2"
                defaultValue={content?.textPositionRight ?? 0}
              />
            </div>
            <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
              <span className="fill-gray-600 w-[2rem] h-[2rem]">
                {icons.down}
              </span>
              <input
                type="text"
                name="textPositionBottom"
                id="textPositionBottom"
                ref={(r) => (inputRefs.current[index++] = r)}
                className="w-3/4 border h-max p-2"
                defaultValue={content?.textPositionBottom ?? 0}
              />
            </div>
            <div className="flex flex-row flex-nowrap w-1/4 justify-evenly items-center max-h-10">
              <span className="fill-gray-600 w-[2rem] h-[2rem]">
                {icons.left}
              </span>
              <input
                type="text"
                name="textPositionLeft"
                id="textPositionLeft"
                ref={(r) => (inputRefs.current[index++] = r)}
                className="w-3/4 border h-max p-2"
                defaultValue={content?.textPositionLeft ?? 0}
              />
            </div>
          </div>
        </div>
      </div>
      <ModalButton
        disabled={!textContent}
        btntext={"Mentés"}
        callback={submitText}
      />
    </div>
  );
};

export default Text;
