import React, { useMemo, useState, useEffect, useCallback } from "react";
import Cookies from "js-cookie";
import dynamic from "next/dynamic";
import { findLastIndexOfChar } from "@/utils/findLastIndexOfChar";
import { useSelector, useDispatch } from "react-redux";
import { currencyDisplayHandler } from "@/utils/currencyDisplay";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import Loader from "@/components/shared/loader/global-loading";
import TableSelectOptions from "@/components/shared/datatable/TableSelectOptions";
import ApiCall from "@/api/api-calls";
import useDynamicManager from "@/store/managers/useDynamicManager";
import useTableDebounce from "@/hooks/useTableDebounce";
import Autocomplete from "@mui/material/Autocomplete";
import { TextField } from "@mui/material";

const ProductList = ({ showSelectProductModal, type }) => {
  const [data, setData] = useState([]);
  const [page, setPage] = useState(1);
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(25);
  const { state, selectSingleProduct } = useDynamicManager(type);
  const [sortField, setSortField] = useState("createdAt");
  const [sortDirection, setSortDirection] = useState("desc");
  const [searchTerm, setSearchTerm] = useState("");
  const [totalRecords, setTotalRecords] = useState(0);
  const [error, setError] = useState(false);
  const [message, setMessage] = useState(false);
  const [loading, setLoading] = useState(false);

  const [selectedRows, setSelectedRows] = useState([]);
  const [toggleCleared, setToggleCleared] = useState(false);
  const [inputValue, setInputValue] = useState("");
  const debouncedSearchTerm = useTableDebounce(inputValue, 500);

  const query = `?page=${page}&pageSize=${pageSize}&sortField=${sortField}&sortDirection=${sortDirection}&searchTerm=${debouncedSearchTerm.trim()}`;

  const fetchData = async () => {
    setLoading(true);
    const response = await ApiCall.productGroups.getAll(query);

    setData(response?.data);
    setTotalRecords(response?.data.totalRecords);
    setLoading(false);
  };

  useEffect(() => {
    fetchData();
  }, [query]);

  useEffect(() => {
    fetchData();
  }, [searchTerm, pageSize, currentPage]);

  const tableSelectOptions = [
    {
      text: "Kiválasztás",
      type: "select",
      className: "rounded bg-blue-500/75 px-4 py-2 mr-4 text-white",
    },
  ];

  const columns = useMemo(
    () => [
      {
        id: "productFeaturedImage",
        name: "Termék kép",
        selector: (row) => (
          <img
            width={72}
            height={72}
            src={
              row.productFeaturedImage !== "noimage"
                ? `${process.env.NEXT_PUBLIC_PATH}/api/admin/i/2/2/0/` +
                  findLastIndexOfChar.before(row.productFeaturedImage, ".") +
                  ".webp"
                : "/noimage.png"
            }
            className="object-cover"
            alt={
              row.productFeaturedImage ? row.productFeaturedImage : "no image"
            }
            style={{
              width: "72px",
              height: "72px",
              display: "block",
              objectFit: "cover",
            }}
            effect="blur"
            loading="lazy"
          />
        ),
        sortable: true,
      },
      {
        id: "productItemNumberIdentifier",
        name: "Termék leltári száma",
        selector: (row) => {
          return (
            <p className="font-bold text-sm">
              {row.productItemNumberIdentifier}
            </p>
          );
        },
        sortable: true,
      },
      {
        id: "productGroupTitle",
        name: "Termék neve",
        selector: (row) => {
          return <p className="font-bold text-sm">{row.productGroupTitle}</p>;
        },
        sortable: true,
        style: {
          overflow: "hidden",
          textOverflow: "ellipsis",
          whiteSpace: "normal", // Use "normal" to allow text to wrap within the cell
        },
      },
      {
        id: "productGroupCategory",
        name: "Termék kategória",
        selector: (row) => {
          return row.productGroupCategory?.map((r, i) => {
            return (
              <li
                key={i}
                className="w-full text-xs px-2 py-1 bg-gray-700 text-white my-1">
                {r}
              </li>
            );
          });
        },
        sortable: true,
      },
      {
        id: "productPieceWebshopPrice",
        name: "Webshop ár (EUR/db)",
        selector: (row) => {
          return (
            <li className="w-full text-sm text-center">
              {currencyDisplayHandler(row.productPieceWebshopPrice)}
              /db
            </li>
          );
        },
        sortable: true,
      },

      {
        id: "createdAt",
        name: "Létrehozva",
        selector: (row) => row.createdAt,
        sortable: true,
      },
      {
        id: "updatedAt",
        name: "Frissítve",
        selector: (row) => row.updatedAt,
        sortable: true,
      },
    ],
    []
  );
  const handleSelect = async () => {
    selectSingleProduct({ product: selectedRows[0] });
    setSelectedRows([]);
    showSelectProductModal(false);
  };

  const subHeaderComponent = (
    <div className="flex flex-wrap w-full justify-between items-center gap-4">
      <Autocomplete
        freeSolo
        id="subHeaderSearch"
        className="w-72"
        disableClearable
        options={[]}
        inputValue={inputValue}
        onInputChange={(event, newInputValue) => {
          setInputValue(newInputValue); // ✅ Don't trim here
        }}
        renderInput={(params) => (
          <TextField
            {...params}
            label="Keresés"
            type="search"
            variant="outlined"
          />
        )}
      />
      {selectedRows.length === 1 && (
        <div className="flex items-center gap-2 bg-gray-100 px-4 py-2 border rounded">
          <span className="text-sm text-gray-600">
            {selectedRows.length} kiválasztva
          </span>
          <TableSelectOptions
            data={tableSelectOptions}
            selectedRows={selectedRows}
            handleSelect={handleSelect}
          />
        </div>
      )}
    </div>
  );

  return (
    <div>
      {error && (
        <p className="bg-red-500 text-white p-5 text-base font-bold">{error}</p>
      )}
      {message && (
        <p className="bg-green-500 text-white p-5 text-base font-bold">
          {message}
        </p>
      )}
      {data && data.message && (
        <p className="bg-red-500 text-white p-5 text-base font-bold">
          {data.message}
        </p>
      )}

      {data && !data.message && (
        <DataTableWrapper
          tableTitle="Termék Csoportok"
          data={data?.tableData}
          columns={columns}
          progressPending={loading}
          paginationDefaultPage={page}
          defaultSortFieldId="createdAt"
          tableSelectOptions={tableSelectOptions}
          subHeaderComponent={subHeaderComponent}
          selectPath="product-groups/select-product/"
          paginationServer
          // contextActions={contextActions}
          onChangePage={(pageNumber) => {
            setPage(pageNumber);
          }}
          // progressComponent={<Loader />}
          paginationTotalRows={totalRecords}
          paginationPerPage={pageSize}
          onSort={(sortId, direction) => {
            setSortField(sortId.id);
            setSortDirection(direction);
            setPage(1); // reset to first page when sorting
          }}
          onChangeRowsPerPage={(size) => {
            setPageSize(size);
            setPage(1); // reset page
          }}
          selectableRows={true}
          onRowSelected={setSelectedRows}
          clearSelectedRows={toggleCleared}
          onFilterClear={() => fetchData()}
          subHeader={true}
        />
      )}
    </div>
  );
};

export default ProductList;
