import React, { useMemo, useState, useEffect, useCallback } from "react";
import Cookies from "js-cookie";
import dynamic from "next/dynamic";
import { findLastIndexOfChar } from "@/utils/findLastIndexOfChar";
import { useSelector, useDispatch } from "react-redux";
import ApiCall from "@/api/api-calls";

const SelectArticleList = dynamic(() => import("./SelectArticleList"), {
  ssr: false,
});

const ArticleList = ({ showSelectArticleModal }) => {
  const token = Cookies.get("token");
  const [data, setData] = useState([]);
  const [page, setPage] = useState(1);
  const [currentPage, setCurrentPage] = useState(1);

  const [pageSize, setPageSize] = useState(25);
  const [sortField, setSortfield] = useState("createdAt");
  const [sortDirection, setSortDirection] = useState("desc");
  const [searchTerm, setSearchTerm] = useState("");
  const [totalRecords, setTotalRecords] = useState(0);
  const [error, setError] = useState(false);
  const [message, setMessage] = useState(false);
  const [loading, setLoading] = useState(false);
  const dispatch = useDispatch();

  const query = `?page=${currentPage}&pageSize=${pageSize}&sortField=${sortField}&sortDirection=${sortDirection}&searchTerm=${searchTerm}`;

  const fetchData = async () => {
    setLoading(true);
    const response = await ApiCall.articles.getAll(query);

    setData(response);
    setTotalRecords(response.totalRecords);
    setLoading(false);
  };

  useEffect(() => {
    fetchData();
  }, []);

  useEffect(() => {
    fetchData();
  }, [searchTerm, pageSize, currentPage]);

  const tableSelectOptions = [
    {
      text: "Kiválasztás",
      type: "select",
      className: "rounded bg-blue-500/75 px-4 py-2 mr-4 text-white",
    },
  ];

  const columns = useMemo(
    () => [
      {
        id: "featuredImage",
        name: "Cikk kép",
        selector: (row) => (
          <img
            width={72}
            height={72}
            src={
              row.featuredImage !== "noimage"
                ? `${process.env.NEXT_PUBLIC_PATH}/api/admin/i/2/2/0/` +
                  findLastIndexOfChar.before(row?.featuredImage, ".") +
                  ".webp"
                : "/noimage.png"
            }
            className="object-cover"
            alt={row.featuredImage ?? "no image"}
            style={{
              width: "72px",
              height: "72px",
              display: "block",
              objectFit: "cover",
            }}
            effect="blur"
            loading="lazy"
          />
        ),
        sortable: true,
      },
      {
        id: "articleTitle",
        name: "Cikk neve",
        selector: (row) => {
          return <p className="font-bold text-sm">{row?.articleTitle}</p>;
        },
        sortable: true,
        style: {
          overflow: "hidden",
          textOverflow: "ellipsis",
          whiteSpace: "normal", // Use "normal" to allow text to wrap within the cell
        },
      },
      {
        id: "articleColumn",
        name: "Cikk rovat",
        selector: (row) => {
          return row?.articleColumn?.map((r, i) => {
            return (
              <li
                key={i}
                className="w-full text-xs px-2 py-1 bg-gray-700 text-white my-1">
                {r?.columnTitle?.value}
              </li>
            );
          });
        },
        sortable: true,
      },
      {
        id: "createdAt",
        name: "Létrehozva",
        selector: (row) => row?.createdAt,
        sortable: true,
      },
      {
        id: "updatedAt",
        name: "Frissítve",
        selector: (row) => row?.updatedAt,
        sortable: true,
      },
    ],
    []
  );

  const handleChangePage = (page) => dispatch(setCurrentPage({ page: page }));

  return (
    <div>
      {error && (
        <p className="bg-red-500 text-white p-5 text-base font-bold">{error}</p>
      )}
      {message && (
        <p className="bg-green-500 text-white p-5 text-base font-bold">
          {message}
        </p>
      )}
      {data && data.message && (
        <p className="bg-red-500 text-white p-5 text-base font-bold">
          {data.message}
        </p>
      )}

      {data && !data.message && (
        <SelectArticleList
          tableTitle="Cikkek"
          data={data}
          token={token}
          columns={columns}
          tableSelectOptionsList={tableSelectOptions}
          selectPath="articles/select-article/"
          defaultSortFieldId="createdAt"
          paginationDefaultPage={currentPage}
          progressPending={loading}
          paginationTotalRows={totalRecords}
          paginationPerPage={pageSize}
          onSort={(sortId, direction) => {
            setSortfield(sortId.id);
            setSortDirection(direction);
          }}
          onSearch={setSearchTerm}
          onChangePage={handleChangePage}
          onChangeRowsPerPage={(pageSize) => setPageSize(pageSize)}
          onFilterClear={() => fetchData()}
          showSelectArticleModal={showSelectArticleModal}
        />
      )}
    </div>
  );
};

export default ArticleList;
