// DomainRegistryForm.tsx
"use client";

import { useEffect, useMemo, useState, useCallback } from "react";
import AsyncSelect from "react-select/async";
import ApiCall from "@/api/api-calls";

export type DomainUpsertPayload = {
  domain_name: string;
  current_company_id?: string | null;
  new_biller_id?: string | null;
  registrar_id?: string | null;
  expires_on?: string | null; // "YYYY-MM-DD"
  auto_renew?: boolean;
  cancel_flag?: boolean;
  notes?: string | null;
};

// NEVER use toISOString() for dates that represent a calendar day.
// Accepts ISO-with-time, or already-YYYY-MM-DD, and returns YYYY-MM-DD.
const toYMD = (v: unknown): string => {
  if (!v) return "";
  const s = String(v).trim();

  // already good
  if (/^\d{4}-\d{2}-\d{2}$/.test(s)) return s;

  // ISO-ish -> just take the date part safely if present
  const m = s.match(/^(\d{4}-\d{2}-\d{2})[T ]/);
  if (m) return m[1];

  // last resort: try parsing, but format using local getters (not UTC)
  const d = new Date(s);
  if (Number.isNaN(d.getTime())) return "";
  const pad = (n: number) => (n < 10 ? `0${n}` : String(n));
  return `${d.getFullYear()}-${pad(d.getMonth() + 1)}-${pad(d.getDate())}`;
};

export default function DomainRegistryForm({
  initial,
  onSubmit,
}: {
  initial?: any;
  onSubmit: (payload: DomainUpsertPayload) => void | Promise<void>;
}) {
  const [domain, setDomain] = useState(initial?.domain_name ?? "");

  const [currentCompanyId, setCurrentCompanyId] = useState<string | null>(
    initial?.current_company_id ?? null
  );
  const [currentCompanyLabel, setCurrentCompanyLabel] = useState<string>(
    initial?.current_company_name ?? initial?.current_company ?? ""
  );

  const [newBillerId, setNewBillerId] = useState<string | null>(
    initial?.new_biller_id ?? null
  );
  const [newBillerLabel, setNewBillerLabel] = useState<string>(
    initial?.new_biller_name ?? initial?.new_biller ?? ""
  );

  const [registrarId, setRegistrarId] = useState<string | null>(
    initial?.registrar_id ?? null
  );
  const [registrarLabel, setRegistrarLabel] = useState<string>(
    initial?.registrar_name ?? initial?.registrar ?? ""
  );

  // Keep as "YYYY-MM-DD" string
  const [expiresOn, setExpiresOn] = useState<string | null>(
    initial?.expires_on ? toYMD(initial.expires_on) : null
  );

  const [autoRenew, setAutoRenew] = useState<boolean>(!!initial?.auto_renew);
  const [cancelFlag, setCancelFlag] = useState<boolean>(!!initial?.cancel_flag);
  const [notes, setNotes] = useState<string>(initial?.notes ?? "");

  useEffect(() => {
    if (!initial) return;
    setDomain(initial.domain_name ?? "");
    setCurrentCompanyId(initial.current_company_id ?? null);
    setCurrentCompanyLabel(
      initial?.current_company_name ?? initial?.current_company ?? ""
    );
    setNewBillerId(initial.new_biller_id ?? null);
    setNewBillerLabel(initial?.new_biller_name ?? initial?.new_biller ?? "");
    setRegistrarId(initial?.registrar_id ?? null);
    setRegistrarLabel(initial?.registrar_name ?? initial?.registrar ?? "");

    // normalize any incoming shape to "YYYY-MM-DD"
    setExpiresOn(initial.expires_on ? toYMD(initial.expires_on) : null);

    setAutoRenew(!!initial.auto_renew);
    setCancelFlag(!!initial.cancel_flag);
    setNotes(initial.notes ?? "");
  }, [initial]);

  const loadOwnOptions = useCallback(async (input: string) => {
    const r = await ApiCall.domainRegistry.companyOptions({
      kind: "own",
      q: input || "",
    });
    return (r.data?.items ?? []).map((o: any) => ({
      value: o.value,
      label: o.label,
    }));
  }, []);

  const loadRegistrarOptions = useCallback(async (input: string) => {
    const r = await ApiCall.domainRegistry.companyOptions({
      kind: "registrar",
      q: input || "",
    });
    return (r.data?.items ?? []).map((o: any) => ({
      value: o.value,
      label: o.label,
    }));
  }, []);

  const canSubmit = useMemo(() => domain.trim().length > 3, [domain]);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    const payload: DomainUpsertPayload = {
      domain_name: domain.trim().toLowerCase(),
      current_company_id: currentCompanyId ?? null,
      new_biller_id: newBillerId ?? null,
      registrar_id: registrarId ?? null,
      // send YYYY-MM-DD as string, not a Date
      expires_on: expiresOn || null,
      auto_renew: autoRenew,
      cancel_flag: cancelFlag,
      notes: notes || null,
    };
    await onSubmit(payload);
  };

  return (
    <form
      onSubmit={handleSubmit}
      className="grid grid-cols-1 md:grid-cols-2 gap-3">
      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Domain név</label>
        <input
          className="border rounded px-3 py-2"
          placeholder="pl. example.com"
          value={domain}
          onChange={(e) => setDomain(e.target.value)}
          required
        />
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Jelenlegi cég</label>
        <AsyncSelect
          cacheOptions
          defaultOptions
          loadOptions={loadOwnOptions}
          value={
            currentCompanyId
              ? { value: currentCompanyId, label: currentCompanyLabel || "—" }
              : currentCompanyLabel
              ? { value: "", label: currentCompanyLabel }
              : null
          }
          onChange={(opt: any) => {
            setCurrentCompanyId(opt?.value ?? null);
            setCurrentCompanyLabel(opt?.label ?? "");
          }}
          isClearable
          placeholder="Válassz céget"
        />
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Új számlafizető</label>
        <AsyncSelect
          cacheOptions
          defaultOptions
          loadOptions={loadOwnOptions}
          value={
            newBillerId
              ? { value: newBillerId, label: newBillerLabel || "—" }
              : newBillerLabel
              ? { value: "", label: newBillerLabel }
              : null
          }
          onChange={(opt: any) => {
            setNewBillerId(opt?.value ?? null);
            setNewBillerLabel(opt?.label ?? "");
          }}
          isClearable
          placeholder="Válassz céget"
        />
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Regisztrátor</label>
        <AsyncSelect
          cacheOptions
          defaultOptions
          loadOptions={loadRegistrarOptions}
          value={
            registrarId
              ? { value: registrarId, label: registrarLabel || "—" }
              : registrarLabel
              ? { value: "", label: registrarLabel }
              : null
          }
          onChange={(opt: any) => {
            setRegistrarId(opt?.value || null);
            setRegistrarLabel(opt?.label || "");
          }}
          isClearable
          placeholder="Válassz regisztrátort"
        />
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Lejár</label>
        <input
          type="date"
          className="border rounded px-3 py-2"
          value={expiresOn ?? ""} // must be "YYYY-MM-DD" or ""
          onChange={(e) => setExpiresOn(e.target.value || null)} // keep string
        />
      </div>

      <div className="flex items-center gap-2 mt-2">
        <input
          id="auto_renew"
          className="w-[1.25rem] h-[1.25rem]"
          type="checkbox"
          checked={autoRenew}
          onChange={(e) => setAutoRenew(e.target.checked)}
        />
        <label htmlFor="auto_renew" className="text-sm">
          Automatikus megújítás
        </label>
      </div>

      <div className="flex items-center gap-2 mt-2">
        <input
          id="cancel_flag"
          className="w-[1.25rem] h-[1.25rem]"
          type="checkbox"
          checked={cancelFlag}
          onChange={(e) => setCancelFlag(e.target.checked)}
        />
        <label htmlFor="cancel_flag" className="text-sm">
          Megszüntetni
        </label>
      </div>

      <div className="flex flex-col md:col-span-2">
        <label className="text-xs text-gray-500">Megjegyzés</label>
        <textarea
          className="border rounded px-3 py-2 min-h-24"
          value={notes}
          onChange={(e) => setNotes(e.target.value)}
        />
      </div>

      <div className="md:col-span-2 flex justify-end gap-2">
        <button
          type="submit"
          disabled={!canSubmit}
          className="px-4 py-2 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white disabled:opacity-50">
          Mentés
        </button>
      </div>
    </form>
  );
}
