import React, { useCallback, useEffect, useState } from "react";
import ApiCall from "@/api/api-calls";
import { toast } from "react-hot-toast";
import { useForm } from "@/hooks/custom-form-hook";
import {
  VALIDATOR_REQUIRE,
  VALIDATOR_MINLENGTH,
  VALIDATOR_MAXLENGTH,
} from "@/utils/validators";
import Input from "@/components/shared/input/Input";

const ExpandedDocumentTypeRow = ({ id, onRowUpdated }) => {
  const [data, setData] = useState(null);
  const [loading, setLoading] = useState(false);
  const [formState, handleInput, handleSetInputData] = useForm([], false);

  useEffect(() => {
    setLoading(true);
    if (!id) return;
    const fetchRow = async () => {
      try {
        const rowElement = await ApiCall.documents.getEditDocumentType(id);

        if (rowElement.status === 200) {
          setData(rowElement.data);
          toast.success("Leltár sor sikeresen betöltve");
        }
      } catch (error) {
        console.error("Error fetching row data:", error);
        toast.error("Hiba a sor betöltésekor: " + error.message);
      } finally {
        setLoading(false);
      }
    };
    fetchRow();
  }, [id]);

  const handleUpdateDocumentType = useCallback(() => {
    const sendData = async () => {
      const updateData = {
        display_name: formState.inputs.display_name?.value,
        description: formState.inputs.description?.value,
        name: data?.name, // reuse original value or add an editable input
        versioned: data?.versioned, // same here
        required: data?.required,
      };

      try {
        const res = await ApiCall.documents.editDocumentType(id, updateData);
        if (res.status === 200) {
          toast.success("Dokumentum típus sikeresen frissítve");
          onRowUpdated?.(); // callback to refresh parent if needed
        }
      } catch (err) {
        toast.error("Frissítés sikertelen");
        console.error(err);
      }
    };
    sendData();
  }, [formState, id, data, onRowUpdated]);

  return (
    <>
      {data && (
        <div className="w-full my-4">
          <Input
            element="input"
            id="display_name"
            type="text"
            name="display_name"
            placeholder="Dokumentum típus neve"
            inputcls="w-full border-2 text-center p-2 my-2 outline-blue-500"
            className="w-100"
            validators={[
              VALIDATOR_REQUIRE(),
              VALIDATOR_MINLENGTH(2),
              VALIDATOR_MAXLENGTH(254),
            ]}
            errorText={"Nem lehet üres"}
            onInput={handleInput}
            initialValid={true}
            value={data?.display_name}
          />
          <Input
            element="textarea"
            id="description"
            type="text"
            name="description"
            placeholder="Dokumentum leírás"
            inputcls="w-full border-2 text-center p-2 my-2 outline-blue-500"
            className="w-100"
            validators={[VALIDATOR_REQUIRE(), VALIDATOR_MINLENGTH(2)]}
            errorText={"Nem lehet üres"}
            onInput={handleInput}
            initialValid={true}
            value={data?.description}
          />
          <button
            onClick={handleUpdateDocumentType}
            className="w-full cursor-pointer mt-4 bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700">
            Mentés
          </button>
        </div>
      )}
    </>
  );
};

export default ExpandedDocumentTypeRow;
