"use client";

import { useEffect, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import DocTemplateKindForm, {
  DocTemplateKindPayload,
} from "./DocTemplateKindForm";

export default function ExpandedDocTemplateKind({
  id,
  onSaved,
  onDeleted,
}: {
  id: string;
  onSaved?: () => void;
  onDeleted?: () => void;
}) {
  const [loading, setLoading] = useState(true);
  const [item, setItem] = useState<any>(null);

  const load = async () => {
    setLoading(true);
    try {
      const r = await ApiCall.docTemplateKinds.getOne(id);
      setItem(r.data ?? null);
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ?? "Sablon típus betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (id) load();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [id]);

  const handleSave = async (payload: DocTemplateKindPayload) => {
    try {
      await ApiCall.docTemplateKinds.upsertItem({ id, ...payload });
      toast.success("Sablon típus frissítve");
      onSaved?.();
      await load();
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ?? "Sablon típus mentése sikertelen"
      );
    }
  };

  const handleDelete = async () => {
    if (!confirm("Biztosan törli ezt a típust?")) return;
    try {
      await ApiCall.docTemplateKinds.remove(id);
      toast.success("Típus törölve");
      onDeleted?.();
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ?? "Sablon típus törlése sikertelen"
      );
    }
  };

  if (loading)
    return <div className="p-3 text-sm text-gray-500">Betöltés…</div>;
  if (!item)
    return <div className="p-3 text-sm text-red-600">Típus nem található.</div>;

  return (
    <div className="rounded border p-3 bg-white/50 space-y-3 max-w-[80vw]">
      <DocTemplateKindForm initial={item} onSubmit={handleSave} />

      <div className="flex items-center justify-between">
        <span className="text-xs text-gray-500">
          Utoljára módosítva: {item.updated_at ?? "-"}
        </span>
        <button
          onClick={handleDelete}
          className="px-3 py-2 bg-red-600/80 hover:bg-red-600 text-white rounded">
          Törlés
        </button>
      </div>
    </div>
  );
}
