"use client";

import { useEffect, useMemo, useState } from "react";

export type DocTemplateKindPayload = {
  code: string;
  label: string;
  description?: string | null;
  is_active?: boolean;
  sort_order?: number;
};

export default function DocTemplateKindForm({
  initial,
  onSubmit,
}: {
  initial?: any;
  onSubmit: (payload: DocTemplateKindPayload) => void | Promise<void>;
}) {
  const [code, setCode] = useState(initial?.code ?? "");
  const [label, setLabel] = useState(initial?.label ?? "");
  const [description, setDescription] = useState(initial?.description ?? "");
  const [isActive, setIsActive] = useState<boolean>(initial?.is_active ?? true);
  const [sortOrder, setSortOrder] = useState<number>(
    initial?.sort_order ?? 100
  );

  useEffect(() => {
    if (!initial) return;
    setCode(initial.code ?? "");
    setLabel(initial.label ?? "");
    setDescription(initial.description ?? "");
    setIsActive(initial.is_active ?? true);
    setSortOrder(initial.sort_order ?? 100);
  }, [initial]);

  const canSubmit = useMemo(
    () => code.trim().length > 0 && label.trim().length > 0,
    [code, label]
  );

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    const payload: DocTemplateKindPayload = {
      code: code.trim(),
      label: label.trim(),
      description: description || null,
      is_active: isActive,
      sort_order: sortOrder || 100,
    };
    await onSubmit(payload);
  };

  return (
    <form
      onSubmit={handleSubmit}
      className="grid grid-cols-1 md:grid-cols-2 gap-3">
      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Kód</label>
        <input
          className="border rounded px-3 py-2"
          value={code}
          onChange={(e) => setCode(e.target.value)}
          placeholder="pl. faq"
          required
        />
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Megnevezés</label>
        <input
          className="border rounded px-3 py-2"
          value={label}
          onChange={(e) => setLabel(e.target.value)}
          placeholder="pl. FAQ"
          required
        />
      </div>

      <div className="flex flex-col md:col-span-2">
        <label className="text-xs text-gray-500">Leírás</label>
        <textarea
          className="border rounded px-3 py-2 min-h-20 text-sm"
          value={description}
          onChange={(e) => setDescription(e.target.value)}
        />
      </div>

      <div className="flex items-center gap-2 mt-2">
        <input
          id="is_active"
          type="checkbox"
          className="w-[1.25rem] h-[1.25rem]"
          checked={isActive}
          onChange={(e) => setIsActive(e.target.checked)}
        />
        <label htmlFor="is_active" className="text-sm">
          Aktív
        </label>
      </div>

      <div className="flex flex-col">
        <label className="text-xs text-gray-500">Sorrend</label>
        <input
          type="number"
          className="border rounded px-3 py-2"
          value={sortOrder}
          onChange={(e) => setSortOrder(Number(e.target.value) || 0)}
        />
      </div>

      <div className="md:col-span-2 flex justify-end gap-2">
        <button
          type="submit"
          disabled={!canSubmit}
          className="px-4 py-2 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white disabled:opacity-50">
          Mentés
        </button>
      </div>
    </form>
  );
}
