"use client";

import { useEffect, useState } from "react";
import ApiCall from "@/api/api-calls";
import MediaGallery from "@/components/media-store/MediaGallery";
import toast from "react-hot-toast";
import apiClient from "@/lib/apiClient";

type DocTemplate = {
  id?: string;
  name: string;
  kind: string;
  header_html: string;
  footer_html: string;
  header_image_name?: string | null;
  footer_image_name?: string | null;
  is_active?: boolean;
};
type DocTemplateKind = {
  code: string;
  label: string;
  description?: string;
};
async function blobToPngDataUrl(blob: Blob): Promise<string> {
  const img = await new Promise<HTMLImageElement>((resolve, reject) => {
    const url = URL.createObjectURL(blob);
    const imageEl = new Image();
    imageEl.onload = () => {
      URL.revokeObjectURL(url);
      resolve(imageEl);
    };
    imageEl.onerror = (err) => {
      URL.revokeObjectURL(url);
      reject(err);
    };
    imageEl.src = url;
  });

  const canvas = document.createElement("canvas");
  canvas.width = img.naturalWidth || img.width;
  canvas.height = img.naturalHeight || img.height;
  const ctx = canvas.getContext("2d");
  if (!ctx) throw new Error("Canvas 2D context not available");

  ctx.drawImage(img, 0, 0);

  // 👉 data:image/png;base64,...
  return canvas.toDataURL("image/png");
}
export default function DocTemplateEditor({
  id,
  onSaved,
}: {
  id?: string;
  onSaved?: (tpl: DocTemplate) => void;
}) {
  const [tpl, setTpl] = useState<DocTemplate>({
    name: "",
    kind: "generic",
    header_html: "",
    footer_html: "",
    header_image_name: null,
    footer_image_name: null,
    is_active: true,
  });

  const [loading, setLoading] = useState<boolean>(!!id);
  const [showHeaderPicker, setShowHeaderPicker] = useState(false);
  const [showFooterPicker, setShowFooterPicker] = useState(false);
  const [kinds, setKinds] = useState<DocTemplateKind[]>([]);
  const [kindsLoading, setKindsLoading] = useState(false);

  useEffect(() => {
    if (!id) {
      setLoading(false);
      return;
    }
    (async () => {
      try {
        const res = await ApiCall.docTemplates.getItem(id);
        setTpl(res.data);
      } catch (e) {
        console.error(e);
        toast.error("Sablon betöltése sikertelen");
      } finally {
        setLoading(false);
      }
    })();
  }, [id]);

  useEffect(() => {
    (async () => {
      try {
        setKindsLoading(true);
        const res = await ApiCall.docTemplateKinds.getList();
        setKinds(res.data ?? []);
      } catch (e) {
        console.error(e);
        // optional toast
      } finally {
        setKindsLoading(false);
      }
    })();
  }, []);

  const handleSave = async () => {
    try {
      const res = await ApiCall.docTemplates.upsertItem(tpl);
      const saved = res.data;
      setTpl(saved);
      toast.success("Sablon mentve");

      onSaved?.(saved); // 👈 notify parent
    } catch (e) {
      console.error(e);
      toast.error("Sablon mentése sikertelen");
    }
  };

  const handleHeaderImageSelected = async (image: any) => {
    try {
      const filename = image.original_name;

      // ask backend for a header-friendly PNG (e.g. 16/3/1)
      const imgUrl = await ApiCall.mediaStore.getResized("/16/3/1", filename, {
        format: "png",
      });

      const res = await apiClient.get(imgUrl, { responseType: "blob" });
      if (res.status !== 200) {
        throw new Error(`HTTP ${res.status} when fetching image`);
      }

      const blob = res.data as Blob;
      const base64DataUrl = await blobToPngDataUrl(blob);

      const html = `<div style="text-align:center;">
  <img src="${base64DataUrl}" style="width:100%; height:auto; display:block;" />
</div>`;

      setTpl((prev) => ({
        ...prev,
        header_html: html,
        header_image_name: filename, // 👈 store original_name
      }));
      setShowHeaderPicker(false);
    } catch (err) {
      console.error(err);
      toast.error("Nem sikerült a fejléc-kép betöltése");
    }
  };
  const handleFooterImageSelected = async (image: any) => {
    try {
      const filename = image.original_name;

      const imgUrl = await ApiCall.mediaStore.getResized("/10/1/0", filename, {
        format: "png",
      });

      const res = await apiClient.get(imgUrl, { responseType: "blob" });
      if (res.status !== 200) {
        throw new Error(`HTTP ${res.status} when fetching image`);
      }

      const blob = res.data as Blob;
      const base64DataUrl = await blobToPngDataUrl(blob);

      const html = `<div style="margin:0; padding:0; line-height:1;">
  <img src="${base64DataUrl}" style="display:block; width:100%; height:100%; object-fit:cover; margin:0; padding:0;" />
</div>`;

      setTpl((prev) => ({
        ...prev,
        footer_html: html,
        footer_image_name: filename, // 👈 store original_name
      }));
      setShowFooterPicker(false);
    } catch (err) {
      console.error(err);
      toast.error("Nem sikerült a lábléc-kép betöltése");
    }
  };

  if (loading) return <p className="p-4">Betöltés…</p>;

  return (
    <div className="max-w-3xl mx-auto p-4 space-y-4">
      <div className="flex gap-2 items-center">
        <input
          className="border rounded px-3 py-2 flex-1"
          placeholder="Sablon neve"
          value={tpl.name}
          onChange={(e) => setTpl({ ...tpl, name: e.target.value })}
        />
        <select
          className="border rounded px-2 py-2 text-sm"
          value={tpl.kind}
          onChange={(e) => setTpl({ ...tpl, kind: e.target.value })}>
          {kinds.map((k) => (
            <option key={k.code} value={k.code}>
              {k.label}
            </option>
          ))}
        </select>
        {!kindsLoading && kinds.length === 0 && (
          <span className="ml-2 text-xs text-red-500">
            Nincsenek elérhető sablon típusok
          </span>
        )}
      </div>

      {/* Header HTML + pick button */}
      <div className="space-y-2">
        <div className="flex justify-between items-center">
          <span className="text-sm font-medium">Fejléc HTML</span>
          <button
            onClick={() => setShowHeaderPicker(true)}
            className="px-2 py-1 text-xs rounded border">
            Kép választása
          </button>
        </div>
        <textarea
          className="border rounded w-full text-xs p-2 min-h-[80px]"
          value={tpl.header_html}
          onChange={(e) =>
            setTpl((prev) => ({ ...prev, header_html: e.target.value }))
          }
        />
        {/* Under the header_html textarea */}
        {tpl.header_html && (
          <div className="border rounded p-2 mt-2">
            <div className="text-xs text-gray-500 mb-1">Fejléc előnézet</div>
            <div dangerouslySetInnerHTML={{ __html: tpl.header_html }} />
          </div>
        )}
      </div>

      {/* Footer HTML + pick button */}
      <div className="space-y-2">
        <div className="flex justify-between items-center">
          <span className="text-sm font-medium">Lábléc HTML</span>
          <button
            onClick={() => setShowFooterPicker(true)}
            className="px-2 py-1 text-xs rounded border">
            Kép választása
          </button>
        </div>
        <textarea
          className="border rounded w-full text-xs p-2 min-h-[80px]"
          value={tpl.footer_html}
          onChange={(e) =>
            setTpl((prev) => ({ ...prev, footer_html: e.target.value }))
          }
        />
      </div>

      {/* Under the footer_html textarea */}
      {tpl.footer_html && (
        <div className="border rounded p-2 mt-2">
          <div className="text-xs text-gray-500 mb-1">Lábléc előnézet</div>
          <div dangerouslySetInnerHTML={{ __html: tpl.footer_html }} />
        </div>
      )}

      <button
        onClick={handleSave}
        className="px-4 py-2 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white text-sm">
        Mentés
      </button>

      {/* Header image picker modal */}
      {showHeaderPicker && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50">
          <div className="bg-white rounded shadow-lg max-w-7xl w-full max-h-[90vh] overflow-x-hidden">
            <div className="flex justify-between items-center px-4 py-2 border-b">
              <h2 className="text-sm font-semibold">Fejléc kép kiválasztása</h2>
              <button
                onClick={() => setShowHeaderPicker(false)}
                className="text-xs px-2 py-1">
                Bezárás
              </button>
            </div>
            <div className="p-4">
              <MediaGallery
                mode="picker"
                action="editorImage"
                onSelect={() => {}}
                onConfirmSelection={async (itemOrArray: any) => {
                  const image = Array.isArray(itemOrArray)
                    ? itemOrArray[0]
                    : itemOrArray;
                  if (image) {
                    await handleHeaderImageSelected(image);
                  }
                }}
                onClose={() => setShowHeaderPicker(false)}
              />
            </div>
          </div>
        </div>
      )}

      {/* Footer image picker modal */}
      {showFooterPicker && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50">
          <div className="bg-white rounded shadow-lg max-w-7xl w-full max-h-[90vh] overflow-x-hidden">
            <div className="flex justify-between items-center px-4 py-2 border-b">
              <h2 className="text-sm font-semibold">Lábléc kép kiválasztása</h2>
              <button
                onClick={() => setShowFooterPicker(false)}
                className="text-xs px-2 py-1">
                Bezárás
              </button>
            </div>
            <div className="p-4">
              <MediaGallery
                mode="picker"
                action="editorImage"
                onSelect={() => {}} // 👈 and here too
                onConfirmSelection={(itemOrArray: any) => {
                  const image = Array.isArray(itemOrArray)
                    ? itemOrArray[0]
                    : itemOrArray;
                  if (image) {
                    handleFooterImageSelected(image);
                  }
                }}
                onClose={() => setShowFooterPicker(false)}
              />
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
