"use client";

import { useEffect, useMemo, useState } from "react";
import ApiCall from "@/api/api-calls";
import type { LookupUser } from "@/api/routes/lookups";
import type { ActionItemAssigneeRole } from "@/api/routes/action-items";
import { Chip, TextInput, Select, Button } from "@/components/ui/CardBits";

type Sel = { user: LookupUser; role: ActionItemAssigneeRole };

export default function AssigneesPicker(props: {
  value: Sel[];
  onChange: (v: Sel[]) => void;
}) {
  const { value, onChange } = props;

  const [q, setQ] = useState("");
  const [hits, setHits] = useState<LookupUser[]>([]);
  const [loading, setLoading] = useState(false);

  const selectedIds = useMemo(
    () => new Set(value.map((x) => x.user.id)),
    [value]
  );

  useEffect(() => {
    const t = setTimeout(async () => {
      const term = q.trim();
      if (!term) {
        setHits([]);
        return;
      }
      setLoading(true);
      try {
        const r = await ApiCall.lookups.users({ q: term, limit: 10 });
        setHits((r.data.users ?? []).filter((u) => !selectedIds.has(u.id)));
      } finally {
        setLoading(false);
      }
    }, 250);

    return () => clearTimeout(t);
  }, [q, selectedIds]);

  function add(u: LookupUser) {
    onChange([...value, { user: u, role: "contributor" }]);
    setQ("");
    setHits([]);
  }

  function remove(id: string) {
    onChange(value.filter((x) => x.user.id !== id));
  }

  function setRole(id: string, role: ActionItemAssigneeRole) {
    onChange(value.map((x) => (x.user.id === id ? { ...x, role } : x)));
  }

  return (
    <div style={{ display: "flex", flexDirection: "column", gap: 10 }}>
      <div style={{ display: "flex", gap: 10, flexWrap: "wrap" }}>
        {value.map((x) => (
          <div
            key={x.user.id}
            style={{
              display: "flex",
              gap: 8,
              alignItems: "center",
              border: "1px solid rgba(148,163,184,.35)",
              borderRadius: 12,
              padding: "6px 8px",
            }}>
            <Chip tone={x.role === "owner" ? "good" : "neutral"}>
              {x.user.name ?? x.user.email ?? x.user.id}
            </Chip>
            <Select
              value={x.role}
              onChange={(e) => setRole(x.user.id, e.target.value as any)}>
              <option value="owner">owner</option>
              <option value="contributor">contributor</option>
              <option value="reviewer">reviewer</option>
              <option value="watcher">watcher</option>
            </Select>
            <Button onClick={() => remove(x.user.id)}>×</Button>
          </div>
        ))}
      </div>

      <div style={{ display: "flex", gap: 8, alignItems: "center" }}>
        <TextInput
          value={q}
          onChange={(e) => setQ(e.target.value)}
          placeholder="Felhasználó keresés (név/email/username)…"
          style={{ flex: 1, minWidth: 260 }}
        />
        <Chip tone="neutral">{loading ? "..." : `${hits.length} találat`}</Chip>
      </div>

      {hits.length ? (
        <div style={{ display: "flex", gap: 8, flexWrap: "wrap" }}>
          {hits.map((u) => (
            <Button key={u.id} onClick={() => add(u)}>
              + {u.name ?? u.email ?? u.username ?? u.id}
            </Button>
          ))}
        </div>
      ) : null}
    </div>
  );
}
