// components/audit-log/ExpandedAuditLog.tsx
"use client";

import { useEffect, useState } from "react";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";

type AuditDetail = {
  id: string;
  at: string;
  actor_id: string | null;
  action: string;
  object_type: string;
  object_id: string | null;
  route: string | null;
  ip: string | null;
  payload_hash: string | null;
  meta: any;
};

export default function ExpandedAuditLog({ id }: { id: string }) {
  const [loading, setLoading] = useState(true);
  const [item, setItem] = useState<AuditDetail | null>(null);

  const load = async () => {
    setLoading(true);
    try {
      const r = await ApiCall.auditLog.getOne(id);
      setItem(r.data?.body ?? null);
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ?? "Audit bejegyzés betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (id) load();
  }, [id]);

  if (loading) {
    return <div className="text-sm text-gray-500">Betöltés…</div>;
  }
  if (!item) {
    return <div className="text-sm text-red-600">Nem található.</div>;
  }

  return (
    <div className="rounded border bg-white/60 p-3 space-y-3 max-w-[80vw] text-sm">
      <div className="grid grid-cols-1 md:grid-cols-2 gap-x-6 gap-y-1">
        <div>
          <span className="font-semibold">Időpont:</span> {item.at}
        </div>
        <div>
          <span className="font-semibold">Felhasználó ID:</span>{" "}
          {item.actor_id ?? "—"}
        </div>
        <div>
          <span className="font-semibold">Akció:</span> {item.action}
        </div>
        <div>
          <span className="font-semibold">Objektum típus:</span>{" "}
          {item.object_type}
        </div>
        <div>
          <span className="font-semibold">Objektum ID:</span>{" "}
          {item.object_id ?? "—"}
        </div>
        <div>
          <span className="font-semibold">Útvonal:</span> {item.route ?? "—"}
        </div>
        <div>
          <span className="font-semibold">IP:</span> {item.ip ?? "—"}
        </div>
        <div>
          <span className="font-semibold">Payload hash:</span>{" "}
          {item.payload_hash ?? "—"}
        </div>
      </div>

      <div>
        <div className="font-semibold mb-1">Meta (JSON)</div>
        <pre className="max-h-80 overflow-auto rounded bg-gray-900 text-gray-100 text-xs p-2">
          {JSON.stringify(item.meta ?? {}, null, 2)}
        </pre>
      </div>
    </div>
  );
}
