import React, { useEffect, useState, useCallback } from "react";
import OrderStatusCheckboxGroup from "./OrderStatusCheckboxGroup";
import ApiCall from "@/api/api-calls";
import { currencyDisplayHandler } from "@/utils/currencyDisplay";
import toast from "react-hot-toast";

const OrderItem = ({ item, index }) => {
  return (
    <div
      className={`w-full flex flex-row flex-wrap justify-evenly items-center ${
        index % 2 === 0 ? "bg-white" : "bg-gray-200"
      } p-4`}>
      <p className="flex flex-col text-[0.85rem] text-gray-400 flex-1">
        Azonosító:
        <span className="text-[0.95rem] text-black flex-1">
          {item?.item?.productItemNumber?.value}
        </span>
      </p>
      <p className="flex flex-col text-[0.85rem] text-gray-400 flex-1">
        Terméknév:
        <span className="text-[0.95rem] text-black flex-1">
          {item?.item?.productParentGroup?.productGroupTitle?.value}
        </span>
      </p>
      <p className="flex flex-col text-[0.85rem] text-gray-400 flex-1">
        Darabszám:{" "}
        <span className="text-[0.95rem] text-black flex-1">{item.qty} db</span>
      </p>
      {item?.item?.productColor?.[0] && (
        <p className="flex flex-col text-[0.85rem] text-gray-400 flex-1">
          Szín:{" "}
          <span className="text-[0.95rem] text-black flex-1">
            {item?.item?.productColor?.[0]?.colorTitle?.value}
          </span>
        </p>
      )}
      {item?.item?.productSize?.[0] && (
        <p className="flex flex-col text-[0.85rem] text-gray-400 flex-1">
          Méret:{" "}
          <span className="text-[0.95rem] text-black flex-1">
            {item?.item?.productSize?.[0]?.sizeTitle?.value}
          </span>
        </p>
      )}
      <p className="flex flex-col text-[0.85rem] text-gray-400 flex-1">
        Nettó:{" "}
        <span className="text-[0.95rem] text-black flex-1">
          {item.netto} Ft
        </span>
      </p>
      <p className="flex flex-col text-[0.85rem] text-gray-400 flex-1">
        Bruttó:{" "}
        <span className="text-[0.95rem] text-black flex-1">
          {item.brutto} Ft
        </span>
      </p>
      {item?.item?.productDiscount?.value !== 0 && (
        <p className="flex flex-col text-[0.85rem] text-gray-400 flex-1">
          Kedvezmény:{" "}
          <span className="text-[0.95rem] text-black flex-1">
            {item?.item?.productDiscount?.value} %
          </span>
        </p>
      )}
    </div>
  );
};

const ExpandedAdminOrder = ({ id, data }) => {
  const [orderData, setOrderData] = useState(null);
  const [hasUserChangedCheckbox, setHasUserChangedCheckbox] = useState(false);
  const [checkBoxState, setCheckboxState] = useState(false);

  useEffect(() => {
    const fetchData = async () => {
      const response = await ApiCall.adminOrders.getById(id);
      if (response.status === 200) {
        toast.success("Rendelés betöltve!");
        setOrderData(response.data);
        setCheckboxState({
          isCompleted: response?.data?.isCompleted,
          isPacking: response?.data?.isPacking,
          isUnderDelivery: response?.data?.isUnderDelivery,
          isOuterStore: response?.data?.isOuterStore,
          isDeleted: response?.data?.isDeleted,
        });
      } else {
        toast.error("Hiba a rendelés betöltése közben.");
      }
    };
    fetchData();
  }, []);

  const handleCheckboxChange = useCallback((id, field) => {
    // setCheckboxState((prevState) => ({
    //   ...prevState,
    //   [field]: !prevState[field], // ✅ Toggle the actual checkbox value
    // }));
    // setHasUserChangedCheckbox(true);
    setCheckboxState((prevState) => ({
      isCompleted: field === "isCompleted",
      isPacking: field === "isPacking",
      isUnderDelivery: field === "isUnderDelivery",
      isOuterStore: field === "isOuterStore",
      isDeleted: field === "isDeleted",
    }));
    setHasUserChangedCheckbox(true);
  }, []);

  const handleSendCheckboxChange = useCallback(async () => {
    if (!hasUserChangedCheckbox) return; // Only proceed if the user has changed the checkbox

    const response = await ApiCall.adminOrders.updateDeliveryStatus(
      orderData?._id,
      checkBoxState
    );
    if (response.status === 200 && response.data.status) {
      toast.success(response.data.message);
      if (
        response?.data?.status &&
        (response?.data?.status.isCompleted !== checkBoxState.isCompleted ||
          response?.data.status.isPacking !== checkBoxState.isPacking ||
          response?.data.status.isUnderDelivery !==
            checkBoxState.isUnderDelivery ||
          response?.data.status.isOuterStore !== checkBoxState.isOuterStore ||
          response?.data.status.isDeleted !== checkBoxState.isDeleted)
      ) {
        setCheckboxState({
          isCompleted: response?.data.status.isCompleted,
          isPacking: response?.data.status.isPacking,
          isUnderDelivery: response?.data.status.isUnderDelivery,
          isOuterStore: response?.data.status.isOuterStore,
          isDeleted: response?.data.status.isDeleted,
        });
      }
      setHasUserChangedCheckbox(false); // Reset the user change flag after API call
    }
  }, [orderData?._id, hasUserChangedCheckbox]);

  useEffect(() => {
    if (!hasUserChangedCheckbox) return;

    const timeout = setTimeout(() => {
      handleSendCheckboxChange();
    }, 500); // debounce time

    return () => clearTimeout(timeout);
  }, [checkBoxState, hasUserChangedCheckbox]);

  return (
    <>
      {data && orderData && (
        <div className="w-full flex flex-col bg-[#E3F8FC] p-4">
          <div className="w-full flex flex-row flex-wrap justify-between items-start pb-4 border-b border-blue-600/50">
            <p className="flex flex-col text-[0.85rem] text-gray-400 flex-1">
              Rendelés ideje:{" "}
              <span className="text-[0.95rem] text-gray-600">
                {data.createdAt}
              </span>
            </p>

            <p className="flex flex-col text-[0.85rem] text-gray-400 flex-1">
              Megrendelő e-mail:
              <span className="text-[0.95rem] text-gray-600">
                {data?.orderedBy.username.value}
              </span>
            </p>
            <p className="flex flex-col text-[0.85rem] text-gray-400 flex-1">
              Megrendelő telefon:{" "}
              <span className="text-[0.95rem] text-gray-600">
                {data?.orderedBy.telefon.value}
              </span>
            </p>
            <p className="flex flex-col text-[0.85rem] text-gray-400 flex-1">
              Szállítási cím:{" "}
              <span className="text-[0.95rem] text-gray-600">
                {data?.deliveryAddress}
              </span>
            </p>
            <p className="flex flex-col text-[0.85rem] text-gray-400 flex-1">
              Rendelés összege:{" "}
              <span className="text-[0.95rem] text-gray-600">
                {currencyDisplayHandler(data.orderBrutto, "hu-HU", "HUF", 2)}
              </span>
            </p>
            <p className="flex flex-col text-[0.85rem] text-gray-400 flex-1">
              Szállítás típusa{" "}
              <span className="text-[0.95rem] text-gray-600">
                {data?.deliveryMethod}
              </span>
            </p>
          </div>
          <div className="w-full flex flex-row flex-wrap justify-between items-start pb-4 border-b border-blue-600/50">
            {orderData?.orderData?.map(
              (d, i) => (
                <OrderItem item={d} index={i} key={i} />
              )
              // return <p>{d?.item?.productParentGroup?.productGroupTitle?.value}</p>;
            )}
          </div>
          <div className="w-full flex flex-col flex-wrap my-4">
            <h5 className="text-center text-sm font-semibold pb-2">
              Megrendelés státusza
            </h5>
            {checkBoxState && (
              <OrderStatusCheckboxGroup
                state={checkBoxState}
                onChange={(field) => handleCheckboxChange(orderData._id, field)}
              />
            )}
          </div>
        </div>
      )}
    </>
  );
};

export default ExpandedAdminOrder;
