"use client";

import { useEffect, useMemo, useState, useCallback } from "react";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ApiCall from "@/api/api-calls";
import Loader from "@/components/shared/loader/global-loading";
import { TextField } from "@mui/material";
import Autocomplete from "@mui/material/Autocomplete";
import Link from "next/link";
import useTableDebounce from "@/hooks/useTableDebounce";
import { useRouter } from "next/navigation";

const PageEditorsList = () => {
  const router = useRouter();
  const [data, setData] = useState([]);
  const [page, setPage] = useState(1);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");
  const [pageSize, setPageSize] = useState(25);
  const [sortField, setSortField] = useState("createdAt");
  const [sortDirection, setSortDirection] = useState("asc");
  const [totalRecords, setTotalRecords] = useState(0);
  const [inputValue, setInputValue] = useState("");
  const [selectedRows, setSelectedRows] = useState([]);
  const [toggleCleared, setToggleCleared] = useState(false);
  const debouncedSearchTerm = useTableDebounce(inputValue, 500); // ✅ Use custom hook

  const query = `?page=${page}&pageSize=${pageSize}&sortField=${sortField}&sortDirection=${sortDirection}&searchTerm=${debouncedSearchTerm.trim()}`;

  const fetchData = async () => {
    setLoading(true);
    try {
      const res = await ApiCall.pages.getAll(query);
      setData(res?.data?.tableData ?? []);
      setTotalRecords(res?.data?.totalRecords ?? 0);
    } catch (err) {
      console.error(err);
    }
    setLoading(false);
  };

  useEffect(() => {
    fetchData();
  }, [query]);

  const handleSelectAction = async (actionType, rowsOrRow) => {
    const selected = Array.isArray(rowsOrRow) ? rowsOrRow : [rowsOrRow];
    const ids = selected.map((r) => r._id);

    if (actionType === "delete") {
      const confirmDelete = confirm(
        `Biztosan törölni szeretnéd a következő elemeket?\n${ids.join(", ")}`
      );
      if (confirmDelete) {
        try {
          await ApiCall.pages.deletePage(ids);
          fetchData();
          setToggleCleared(!toggleCleared);
        } catch (err) {
          console.error("Delete failed:", err);
        }
      }
    }

    if (actionType === "edit") {
      if (selected.length !== 1) {
        alert("Csak egy elem szerkeszthető egyszerre.");
        return;
      }
      const id = selected[0]._id;
      router.push(`/admin/static-pages/${id}`);
    }
  };

  const tableSelectOptions = [
    {
      text: "Szerkesztés",
      type: "edit",
      className: "bg-blue-500 text-white",
    },
    {
      text: "Törlés",
      type: "delete",
      className: "bg-red-500 text-white",
    },
  ];

  const columns = useMemo(
    () => [
      {
        id: "_id",
        name: "Oldal azonosító",
        selector: (row) => row._id,
        sortable: true,
      },
      {
        id: "pageTitle",
        name: "Oldal neve",
        selector: (row) => row.pageTitle,
        sortable: true,
      },
      {
        id: "pageLink",
        name: "Oldal link",
        selector: (row) => row.pageLink,
        sortable: true,
      },
      {
        id: "isPublished",
        name: "Státusz",
        selector: (row) => (row.isPublished ? "Publikálva" : "Vázlat"),
        sortable: true,
      },
      {
        id: "createdAt",
        name: "Létrehozva",
        selector: (row) => row.createdAt,
        sortable: true,
      },
      {
        id: "updatedAt",
        name: "Frissítve",
        selector: (row) => row.updatedAt,
        sortable: true,
      },
      {
        name: "Műveletek",
        cell: (row) => (
          <div className="flex space-x-2">
            <Link
              href={`/admin/static-pages/${row._id}`}
              className="rounded bg-blue-500/75 px-3 py-1 text-sm text-white hover:bg-blue-600">
              Szerkesztés
            </Link>
            <button
              onClick={() => handleSelectAction("delete", row)}
              className="rounded bg-red-500/75 px-3 py-1 text-sm text-white hover:bg-red-600">
              Törlés
            </button>
          </div>
        ),
      },
    ],
    []
  );

  const subHeaderComponent = (
    <div className="flex flex-wrap w-full justify-between items-center gap-4">
      <Autocomplete
        freeSolo
        id="subHeaderSearch"
        className="w-72"
        disableClearable
        options={[]}
        inputValue={inputValue}
        onInputChange={(event, newInputValue) => {
          setInputValue(newInputValue);
        }}
        renderInput={(params) => (
          <TextField
            {...params}
            label="Keresés"
            type="search"
            variant="outlined"
          />
        )}
      />
      {selectedRows.length > 0 && (
        <div className="flex items-center gap-2 bg-gray-100 px-4 py-2 border rounded">
          <span className="text-sm text-gray-600">
            {selectedRows.length} kiválasztva
          </span>
          {tableSelectOptions.map((option) => (
            <button
              key={option.text}
              onClick={() => {
                handleSelectAction(option.type, selectedRows);
              }}
              className={`rounded px-3 py-1 text-sm ${option.className}`}>
              {option.text}
            </button>
          ))}
        </div>
      )}
    </div>
  );

  return (
    <DataTableWrapper
      tableTitle="Oldal szerkesztők"
      data={data}
      columns={columns}
      progressPending={loading}
      paginationDefaultPage={page}
      paginationServer
      onChangePage={(pageNumber) => setPage(pageNumber)}
      progressComponent={<Loader />}
      paginationTotalRows={totalRecords}
      paginationPerPage={pageSize}
      onSort={(col, dir) => {
        setSortField(col.id);
        setSortDirection(dir);
        setPage(1);
      }}
      onChangeRowsPerPage={(size) => {
        setPageSize(size);
        setPage(1);
      }}
      onFilterClear={fetchData}
      subHeader
      subHeaderComponent={subHeaderComponent}
      subHeaderAlign="right"
      subHeaderWrap
      tableOptionsPath="static-pages/create"
      selectableRows={true}
      onRowSelected={setSelectedRows}
      toggleCleared={toggleCleared}
    />
  );
};

export default PageEditorsList;
