// Template for a list view component based on the OwnCompanyList pattern
// This example is for seller_types. You can replicate it for other reference tables

"use client";

import { useEffect, useMemo, useState } from "react";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ApiCall from "@/api/api-calls";
import Loader from "@/components/shared/loader/global-loading";
import { TextField } from "@mui/material";
import Autocomplete from "@mui/material/Autocomplete";
import toast from "react-hot-toast";
import { Pencil, Trash2 } from "lucide-react";
import Link from "next/link";
import useTableDebounce from "@/hooks/useTableDebounce";

const SellerTypeList = () => {
  const [data, setData] = useState([]);
  const [page, setPage] = useState(1);
  const [loading, setLoading] = useState(true);
  const [pageSize, setPageSize] = useState(25);
  const [sortField, setSortField] = useState("name");
  const [sortDirection, setSortDirection] = useState("asc");
  const [totalRecords, setTotalRecords] = useState(0);
  const [inputValue, setInputValue] = useState("");
  const [selectedRows, setSelectedRows] = useState([]);
  const [toggleCleared, setToggleCleared] = useState(false);
  const debouncedSearchTerm = useTableDebounce(inputValue, 500);

  const query = useMemo(() => {
    return `?page=${page}&pageSize=${pageSize}&sortField=${sortField}&sortDirection=${sortDirection}&searchTerm=${debouncedSearchTerm.trim()}`;
  }, [page, pageSize, sortField, sortDirection, debouncedSearchTerm]);

  const fetchData = async () => {
    setLoading(true);
    try {
      const response = await ApiCall.sellerTypes.getList(query);
      setData(response?.data?.tableData ?? []);
      setTotalRecords(response?.data?.totalRecords ?? 0);
    } catch (err) {
      console.error(err);
    }
    setLoading(false);
  };

  useEffect(() => {
    fetchData();
  }, [query]);

  const handleSelectAction = async (actionType, rowsOrRow) => {
    const selected = Array.isArray(rowsOrRow) ? rowsOrRow : [rowsOrRow];
    const ids = selected.map((r) => r.id);

    if (actionType === "delete") {
      const confirmDelete = confirm(`Confirm delete:\n${ids.join(", ")}`);
      if (confirmDelete) {
        try {
          await ApiCall.sellerTypes.delete(ids);
          fetchData();
          setToggleCleared(!toggleCleared);
          toast.success("Deleted successfully");
        } catch (err) {
          console.error("Delete failed:", err);
          toast.error("Delete failed");
        }
      }
    }

    if (actionType === "edit") {
      if (selected.length !== 1) {
        alert("Only one item can be edited at a time.");
        return;
      }
      const id = selected[0]._id;
      window.location.href = `/admin/seller-types/edit/${id}`;
    }
  };

  const columns = useMemo(
    () => [
      { id: "id", name: "id", selector: (row) => row.id, sortable: true },
      {
        id: "name",
        name: "Eladó típusa",
        selector: (row) => row.name,
        sortable: true,
      },
      {
        name: "Műveletek",
        cell: (row) => (
          <div className="flex space-x-2">
            <Link
              href={`/admin/seller-types/edit/${row._id}`}
              className="rounded bg-blue-500/75 px-3 py-1 text-sm text-white hover:bg-blue-600">
              <Pencil className="w-4 h-4" />
            </Link>
            <button
              onClick={() => handleSelectAction("delete", row)}
              className="rounded bg-red-500/75 px-3 py-1 text-sm text-white hover:bg-red-600">
              <Trash2 className="w-4 h-4" />
            </button>
          </div>
        ),
      },
    ],
    []
  );

  const subHeaderComponent = (
    <div className="flex flex-wrap w-full justify-between items-center gap-4">
      <Autocomplete
        freeSolo
        id="subHeaderSearch"
        className="w-72"
        disableClearable
        options={[]}
        inputValue={inputValue}
        onInputChange={(event, newInputValue) => {
          setInputValue(newInputValue);
        }}
        renderInput={(params) => (
          <TextField
            {...params}
            label="Keresés"
            type="search"
            variant="outlined"
          />
        )}
      />
    </div>
  );

  return (
    <DataTableWrapper
      tableTitle="Eladó típusok"
      data={data}
      columns={columns}
      progressPending={loading}
      paginationDefaultPage={page}
      defaultSortFieldId="name"
      paginationServer
      onChangePage={(pageNumber) => setPage(pageNumber)}
      progressComponent={<Loader />}
      paginationTotalRows={totalRecords}
      paginationPerPage={pageSize}
      onSort={(col, dir) => {
        setSortField(col.id);
        setSortDirection(dir);
        setPage(1);
      }}
      onChangeRowsPerPage={(size) => {
        setPageSize(size);
        setPage(1);
      }}
      onFilterClear={fetchData}
      subHeader
      subHeaderComponent={subHeaderComponent}
      subHeaderAlign="right"
      subHeaderWrap
      tableOptionsPath="seller-types/create"
      selectableRows={true}
      onRowSelected={setSelectedRows}
      toggleCleared={toggleCleared}
    />
  );
};

export default SellerTypeList;
