"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import StatusBadge from "@/components/qms/StatusBadge";
import Chips from "@/components/qms/Chips";
import Modal from "@/components/shared/modal/Modal";
import useModalMap from "@/hooks/useModalMap";
import WorkItemModal from "@/components/qms/WorkItemModal";
import WorkItemCreateModal from "@/components/qms/WorkItemCreateModal";

type WorkItemRow = {
  id: string;
  title: string;
  status: string;
  due_at: string | null;
  notes: string | null;

  object_type: string;
  object_id: string;

  created_at: string | null;
  updated_at: string | null;

  createdByName?: string | null;
  updatedByName?: string | null;

  assignees?: { subjectType: string; subjectId: string; displayName: string }[];
  comment_count?: number;
};

export default function QmsWorkItemsPage() {
  const { isModalOpen, toggleModal } = useModalMap();

  const [items, setItems] = useState<WorkItemRow[]>([]);
  const [loading, setLoading] = useState(false);

  const [mineOnly, setMineOnly] = useState(true);
  const [status, setStatus] = useState<string>("");
  const [selectedId, setSelectedId] = useState<string | null>(null);

  const load = useCallback(async () => {
    setLoading(true);
    try {
      const params: any = {};
      if (mineOnly) params.mine = true;
      if (status) params.status = status;

      const r = await ApiCall.qms.listWorkItems(params);
      setItems((r.data?.items ?? []) as WorkItemRow[]);
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ?? "Work itemek betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  }, [mineOnly, status]);

  useEffect(() => {
    load();
  }, [load]);

  const selected = useMemo(
    () => items.find((x) => x.id === selectedId) ?? null,
    [items, selectedId]
  );

  return (
    <main className="p-4 space-y-3">
      <div className="flex items-center gap-2 flex-wrap">
        <h1 className="text-lg font-semibold">Work items</h1>

        <div className="ml-auto flex items-center gap-2">
          <label className="text-sm flex items-center gap-2">
            <input
              type="checkbox"
              checked={mineOnly}
              onChange={(e) => setMineOnly(e.target.checked)}
            />
            Csak az enyém
          </label>

          <select
            className="border rounded px-2 py-1"
            value={status}
            onChange={(e) => setStatus(e.target.value)}>
            <option value="">(összes)</option>
            <option value="open">open</option>
            <option value="in_progress">in_progress</option>
            <option value="blocked">blocked</option>
            <option value="done">done</option>
          </select>
          <button
            className="px-2 py-1 rounded bg-emerald-600 text-white"
            onClick={() => {
              setSelectedId(null);
              toggleModal("workItemCreateModal", true);
            }}>
            + Work item
          </button>
          <button className="px-2 py-1 rounded border" onClick={load}>
            Frissít
          </button>
        </div>
      </div>

      {loading ? (
        <div className="py-6 text-sm text-gray-500">Betöltés…</div>
      ) : (
        <div className="overflow-auto border rounded">
          <table className="min-w-full text-sm">
            <thead className="text-left border-b bg-gray-50">
              <tr>
                <th className="py-2 px-3">Cím</th>
                <th className="py-2 px-3">Státusz</th>
                <th className="py-2 px-3">Határidő</th>
                <th className="py-2 px-3">Assignees</th>
                <th className="py-2 px-3">Komment</th>
                <th className="py-2 px-3">Utoljára</th>
                <th className="py-2 px-3 text-right">Művelet</th>
              </tr>
            </thead>
            <tbody>
              {items.map((x) => (
                <tr key={x.id} className="border-b hover:bg-gray-50">
                  <td className="py-2 px-3">
                    <div className="font-medium">{x.title}</div>
                    <div className="text-xs text-gray-500">
                      {x.object_type}:{x.object_id}
                    </div>
                    <div className="text-xs text-gray-500">
                      Created: {x.createdByName ?? "—"} • Updated:{" "}
                      {x.updatedByName ?? "—"}
                    </div>
                  </td>
                  <td className="py-2 px-3">
                    <StatusBadge value={x.status} />
                  </td>
                  <td className="py-2 px-3">{x.due_at ?? "—"}</td>
                  <td className="py-2 px-3">
                    {x.assignees?.length ? (
                      <Chips
                        items={x.assignees.map((a) => ({
                          displayName: a.displayName,
                        }))}
                      />
                    ) : (
                      <span className="text-xs text-gray-500">—</span>
                    )}
                  </td>
                  <td className="py-2 px-3">{x.comment_count ?? 0}</td>
                  <td className="py-2 px-3">
                    {x.updated_at ?? x.created_at ?? "—"}
                  </td>
                  <td className="py-2 px-3">
                    <div className="flex justify-end gap-2">
                      <button
                        className="px-2 py-1 rounded bg-gray-800 text-white"
                        onClick={() => {
                          setSelectedId(x.id);
                          toggleModal("workItemModal", true);
                        }}>
                        Megnyit
                      </button>
                    </div>
                  </td>
                </tr>
              ))}

              {!items.length && (
                <tr>
                  <td colSpan={7} className="py-8 text-center text-gray-500">
                    Nincs work item.
                  </td>
                </tr>
              )}
            </tbody>
          </table>
        </div>
      )}

      <Modal
        show={isModalOpen("workItemModal")}
        onClose={() => toggleModal("workItemModal", false)}
        title={selected ? `Work item • ${selected.title}` : "Work item"}
        height={"auto"}
        width={"860px"}>
        {selectedId ? (
          <WorkItemModal
            id={selectedId}
            onClose={() => toggleModal("workItemModal", false)}
            onChanged={load}
          />
        ) : null}
      </Modal>
      <Modal
        show={isModalOpen("workItemCreateModal")}
        onClose={() => toggleModal("workItemCreateModal", false)}
        title="Új work item"
        height="auto"
        width="720px">
        <WorkItemCreateModal
          onClose={() => toggleModal("workItemCreateModal", false)}
          onCreated={load}
        />
      </Modal>
    </main>
  );
}
