"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import toast from "react-hot-toast";
import { useRouter } from "next/navigation";
import ApiCall from "@/api/api-calls";
import StatusBadge from "@/components/qms/StatusBadge";

type ApprovalRow = {
  id: string;
  document_id: string;
  version_id: string;
  status: "pending" | "approved" | "denied" | "cancelled" | string;

  document_title?: string;
  document_code?: string | null;
  version_no?: number | null;

  requested_at?: string | null;
  requested_by_name?: string | null;
  completed_by_name?: string | null;

  active_step_sequence_no?: number | null;
  active_step_pending_count?: number | null;

  i_am_pending?: boolean;
};

export default function QmsApprovalsPage() {
  const router = useRouter();

  const [items, setItems] = useState<ApprovalRow[]>([]);
  const [loading, setLoading] = useState(false);

  const [mineOnly, setMineOnly] = useState(true);
  const [status, setStatus] = useState<string>("pending");

  const load = useCallback(async () => {
    setLoading(true);
    try {
      const params: any = {};
      if (mineOnly) params.mine = true;
      if (status) params.status = status;

      const r = await ApiCall.qms.listApprovalRequests(params);
      setItems((r.data?.items ?? []) as ApprovalRow[]);
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ?? "Jóváhagyások betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  }, [mineOnly, status]);

  useEffect(() => {
    load();
  }, [load]);

  const pendingCount = useMemo(
    () => items.filter((x) => x.status === "pending").length,
    [items]
  );

  return (
    <main className="p-4 space-y-3">
      <div className="flex items-center gap-2 flex-wrap">
        <h1 className="text-lg font-semibold">Jóváhagyások</h1>

        <div className="ml-auto flex items-center gap-2">
          <label className="text-sm flex items-center gap-2">
            <input
              type="checkbox"
              checked={mineOnly}
              onChange={(e) => setMineOnly(e.target.checked)}
            />
            Csak az enyém
          </label>

          <select
            className="border rounded px-2 py-1"
            value={status}
            onChange={(e) => setStatus(e.target.value)}
            title="Státusz szűrő">
            <option value="">(összes)</option>
            <option value="pending">pending</option>
            <option value="approved">approved</option>
            <option value="denied">denied</option>
            <option value="cancelled">cancelled</option>
          </select>

          <button className="px-2 py-1 rounded border" onClick={load}>
            Frissít
          </button>
        </div>
      </div>

      <div className="text-sm text-gray-600">
        Találatok: <b>{items.length}</b> • Pending: <b>{pendingCount}</b>
      </div>

      {loading ? (
        <div className="py-6 text-sm text-gray-500">Betöltés…</div>
      ) : (
        <div className="overflow-auto border rounded">
          <table className="min-w-full text-sm">
            <thead className="text-left border-b bg-gray-50">
              <tr>
                <th className="py-2 px-3">Dokumentum</th>
                <th className="py-2 px-3">Verzió</th>
                <th className="py-2 px-3">Státusz</th>
                <th className="py-2 px-3">Kérte</th>
                <th className="py-2 px-3">Aktív lépés</th>
                <th className="py-2 px-3 text-right">Művelet</th>
              </tr>
            </thead>

            <tbody>
              {items.map((x) => (
                <tr key={x.id} className="border-b hover:bg-gray-50">
                  <td className="py-2 px-3">
                    <div className="font-medium">
                      {(x.document_code ? `${x.document_code} • ` : "") +
                        (x.document_title ?? "—")}
                    </div>
                    <div className="text-xs text-gray-500">
                      {x.requested_at ? String(x.requested_at) : "—"}
                    </div>
                  </td>
                  <td className="py-2 px-3">
                    {x.version_no != null ? `v${x.version_no}` : "—"}
                  </td>
                  <td className="py-2 px-3">
                    <StatusBadge value={x.status} />
                    {x.i_am_pending ? (
                      <span className="ml-2 text-xs text-amber-700">
                        (rád vár)
                      </span>
                    ) : null}
                  </td>
                  <td className="py-2 px-3">{x.requested_by_name ?? "—"}</td>
                  <td className="py-2 px-3">
                    {x.status === "pending" &&
                    x.active_step_sequence_no != null ? (
                      <span className="text-xs">
                        #{x.active_step_sequence_no} • pending:{" "}
                        <b>{x.active_step_pending_count ?? "—"}</b>
                      </span>
                    ) : (
                      <span className="text-xs text-gray-500">—</span>
                    )}
                  </td>
                  <td className="py-2 px-3">
                    <div className="flex justify-end gap-2">
                      <button
                        className="px-2 py-1 rounded bg-gray-800 text-white"
                        onClick={() =>
                          router.push(`/admin/qms/approvals/${x.id}`)
                        }>
                        Megnyit
                      </button>
                      <button
                        className="px-2 py-1 rounded border"
                        onClick={() =>
                          router.push(`/admin/qms/document/${x.document_id}`)
                        }>
                        Dokumentum
                      </button>
                    </div>
                  </td>
                </tr>
              ))}

              {!items.length && (
                <tr>
                  <td colSpan={6} className="py-8 text-center text-gray-500">
                    Nincs találat.
                  </td>
                </tr>
              )}
            </tbody>
          </table>
        </div>
      )}
    </main>
  );
}
