// File: app/admin/own-company/create/page.tsx
"use client";

import { useEffect, useState, useCallback } from "react";
import Input from "@/components/shared/input/Input";
import { useForm } from "@/hooks/custom-form-hook";
import { useRouter } from "next/navigation";
import Select from "react-select";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import {
  VALIDATOR_REQUIRE,
  VALIDATOR_MAXLENGTH,
  VALIDATOR_MINLENGTH,
  VALIDATOR_EMAIL,
  VALIDATOR_PHONE_HU,
  VALIDATOR_TAXNUMBER_HU,
} from "@/utils/validators";

const CreateOwnCompany = () => {
  const router = useRouter();

  const [companyOptions, setCompanyOptions] = useState([]);
  const [formState, handleInput, handleSetInputData] = useForm({}, true);
  const [selectedGroup, setSelectedGroup] = useState(null);
  const [companyData, setCompanyData] = useState(null);
  const [shortCodes, setShortCodes] = useState([]);
  const [selectedShortCode, setSelectedShortCode] = useState(null);

  useEffect(() => {
    const fetchData = async () => {
      try {
        const [optionsRes, shortCodesRes] = await Promise.all([
          ApiCall.ownCompanies.getAllGroupNames(),
          ApiCall.ownCompanies.getAllShortNames(),
        ]);
        console.log("optionsRes", optionsRes, "shortCodesRes", shortCodesRes);

        if (optionsRes.status === 200) {
          setCompanyOptions(optionsRes.data.groups);
        }
        if (shortCodesRes.status === 200) {
          setShortCodes(shortCodesRes.data.shortCodes);
        }
      } catch (err) {
        toast.error("Hiba történt az adatok betöltése közben");
      }
    };

    fetchData();
  }, []);
  console.log(formState);
  const handleSubmit = useCallback(async () => {
    const payload = Object.fromEntries(
      Object.entries(formState.inputs).map(([k, v]) => [k, v.value])
    );
    payload.company_group_id = selectedGroup?.value || null;
    payload.short_code_id = selectedShortCode?.value || null;
    try {
      const res = await ApiCall.ownCompanies.postCreate(payload);
      if (res.status === 200) {
        toast.success(res.data.message || "Cég mentve");
        router.push("/admin/own-company/list");
      } else {
        toast.error(res.data.error || "Hiba történt mentés közben");
      }
    } catch (err) {
      console.error("Submit failed:", err);
      toast.error("Mentés nem sikerült");
    }
  }, [formState, selectedGroup, selectedShortCode]);

  return (
    <div className="p-6 max-w-4xl mx-auto">
      <h2 className="text-xl font-bold mb-4">Saját cég létrehozása</h2>
      <Input
        id="name"
        name="name"
        element="input"
        placeholder="Cégnév"
        validators={[VALIDATOR_REQUIRE(), VALIDATOR_MINLENGTH(2)]}
        errorText="Kötelező mező"
        initialValid={false}
        onInput={handleInput}
        initialValue={""}
      />
      <div className="mb-4">
        <label className="text-sm text-gray-600 mb-1 block">
          Cég névkód (opcionális):
        </label>
        <Select
          id="short_code_id"
          name="short_code_id"
          options={shortCodes}
          isClearable
          placeholder="Válassz egy névkódot"
          initialValue={selectedShortCode}
          onChange={(selected) => {
            setSelectedShortCode(selected);
            handleInput("short_code_id", selected?.value || "", true);
          }}
        />
      </div>
      <Input
        id="address"
        name="address"
        element="input"
        placeholder="Cég címe"
        validators={[VALIDATOR_MAXLENGTH(254)]}
        errorText="Min. 2 karakter"
        initialValid={true}
        onInput={handleInput}
        initialValue={""}
      />
      <Input
        id="tax_number"
        name="tax_number"
        element="input"
        placeholder="Adószám"
        validators={[VALIDATOR_TAXNUMBER_HU]}
        errorText="Kötelező mező! Nem megfelelő formátum!"
        initialValid={true}
        onInput={handleInput}
        initialValue={""}
      />
      <Input
        id="company_registration_number"
        name="company_registration_number"
        element="input"
        placeholder="Cégjegyzékszám"
        initialValid={true}
        onInput={handleInput}
        initialValue={""}
      />
      <Input
        id="email"
        name="email"
        element="input"
        placeholder="Email"
        validators={[VALIDATOR_EMAIL]}
        errorText="Kötelező mező! Nem megfelelő formátum!"
        initialValid={true}
        onInput={handleInput}
        initialValue={""}
      />
      <Input
        id="phone"
        name="phone"
        element="input"
        placeholder="Telefonszám"
        validators={[VALIDATOR_PHONE_HU]}
        errorText=" Nem megfelelő formátum!"
        initialValid={true}
        onInput={handleInput}
        initialValue={""}
      />
      <Input
        id="notes"
        name="notes"
        element="textarea"
        placeholder="Megjegyzés"
        initialValid={true}
        onInput={handleInput}
        initialValue={""}
      />

      <div className="mb-4">
        <label className="text-sm text-gray-600 mb-1 block">
          Cégcsoport (opcionális):
        </label>
        <Select
          id="company_group_id"
          name="company_group_id"
          options={companyOptions}
          isClearable
          placeholder="Válassz egy cégcsoportot"
          onChange={(selected) => {
            handleInput("company_group_id", selected?.value || "", true);
          }}
        />
      </div>

      <button
        className={`mt-4 px-4 py-2  ${
          !formState.isValid
            ? "bg-gray-500 hover:bg-red-700"
            : "bg-green-600 hover:bg-green-700"
        } text-white rounded `}
        onClick={handleSubmit}
        disabled={!formState.isValid}>
        Mentés
      </button>
    </div>
  );
};

export default CreateOwnCompany;
