// app/(admin)/org-structure/page.tsx
"use client";

import { useEffect, useState } from "react";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";
import OrgTree from "@/components/org-structure/OrgTree";
import OrgNodeForm from "@/components/org-structure/OrgNodeForm";
import { Plus } from "lucide-react";

export default function OrgStructurePage() {
  const [tree, setTree] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);

  // drawer/modal state
  const [openForm, setOpenForm] = useState(false);
  const [mode, setMode] = useState<"create" | "edit">("create");
  const [nodeType, setNodeType] = useState<
    "site" | "department" | "location" | null
  >(null);
  const [context, setContext] = useState<any>(null);

  const loadTree = async () => {
    setLoading(true);
    try {
      const res = await ApiCall.orgApi.getTree(true);
      setTree(res.data || []);
    } catch {
      toast.error("Sikertelen betöltés");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    loadTree();
  }, []);

  const handleCreateClick = (
    type: "site" | "department" | "location",
    parentCtx: any = null
  ) => {
    setMode("create");
    setNodeType(type);
    setContext(parentCtx);
    setOpenForm(true);
  };

  const handleEditClick = (
    type: "site" | "department" | "location",
    node: any
  ) => {
    setMode("edit");
    setNodeType(type);
    setContext(node);
    setOpenForm(true);
  };

  const handleDeleteClick = async (
    type: "site" | "department" | "location",
    node: any
  ) => {
    const confirmed = confirm(`Biztosan törli? (${node.name})`);
    if (!confirmed) return;

    try {
      if (type === "site") await ApiCall.orgApi.deleteSite(node.id);
      if (type === "department") await ApiCall.orgApi.deleteDepartment(node.id);
      if (type === "location") await ApiCall.orgApi.deleteLocation(node.id);

      toast.success("Törlés sikeres");
      loadTree();
    } catch (e: any) {
      toast.error(e?.response?.data?.error || "Törlés nem sikerült");
    }
  };

  // ---- NEW: DnD handlers ----
  const moveDepartment = async (
    departmentId: string,
    toSiteId: string,
    beforeId?: string
  ) => {
    try {
      await ApiCall.orgApi.moveDepartment({
        department_id: departmentId,
        to_site_id: toSiteId,
        before_id: beforeId,
      });
      await loadTree();
      toast.success("Szervezeti egység áthelyezve");
    } catch (e: any) {
      toast.error(e?.response?.data?.error || "Áthelyezés nem sikerült");
    }
  };

  const moveLocation = async (
    locationId: string,
    toDepartmentId: string,
    beforeId?: string
  ) => {
    try {
      await ApiCall.orgApi.moveLocation({
        location_id: locationId,
        to_department_id: toDepartmentId,
        before_id: beforeId,
      });
      await loadTree();
      toast.success("Hely áthelyezve");
    } catch (e: any) {
      toast.error(e?.response?.data?.error || "Áthelyezés nem sikerült");
    }
  };

  const reorderDepartmentsInSite = async (
    siteId: string,
    orderedIds: string[]
  ) => {
    try {
      await ApiCall.orgApi.reorderSiblings({
        type: "department",
        parent_id: siteId,
        ordered_ids: orderedIds,
      });
      await loadTree();
    } catch {
      toast.error("Sorrend mentése nem sikerült");
    }
  };

  const reorderLocationsInDepartment = async (
    departmentId: string,
    orderedIds: string[]
  ) => {
    try {
      await ApiCall.orgApi.reorderSiblings({
        type: "location",
        parent_id: departmentId,
        ordered_ids: orderedIds,
      });
      await loadTree();
    } catch {
      toast.error("Sorrend mentése nem sikerült");
    }
  };

  return (
    <div className="p-6 space-y-4">
      <div className="flex items-center justify-between">
        <h1 className="text-xl font-semibold">Szervezeti struktúra</h1>
        <button
          onClick={() => handleCreateClick("site")}
          className="inline-flex items-center gap-2 rounded bg-blue-600 px-4 py-2 text-white hover:bg-blue-700">
          <Plus className="w-4 h-4" /> Új telephely
        </button>
      </div>

      <div className="rounded border bg-white p-3">
        <OrgTree
          data={tree}
          loading={loading}
          onCreate={handleCreateClick}
          onEdit={handleEditClick}
          onDelete={handleDeleteClick}
          // NEW
          onMoveDepartment={moveDepartment}
          onMoveLocation={moveLocation}
          onReorderDepartments={reorderDepartmentsInSite}
          onReorderLocations={reorderLocationsInDepartment}
        />
      </div>

      {openForm && (
        <OrgNodeForm
          mode={mode}
          type={nodeType as any}
          context={context}
          onClose={() => setOpenForm(false)}
          onSaved={() => {
            setOpenForm(false);
            loadTree();
          }}
        />
      )}
    </div>
  );
}
