// global meetings page + new meeting

"use client";

import { useEffect, useMemo, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import type { Meeting } from "@/api/routes/meetings";
import {
  Page,
  Card,
  ToolbarRow,
  TextInput,
  Button,
  SecondaryButton,
  GridCards,
  CardTitle,
  CardMeta,
  Chip,
  SectionTitle,
  Select,
  Divider,
  EmptyState,
  Field,
} from "@/components/ui/CardBits";
import { useRouter } from "next/navigation";
import { Tabs } from "@/components/ui/Tabs";
import { Drawer } from "@/components/ui/Drawer";

function fmt(iso?: string | null) {
  if (!iso) return "-";
  try {
    return new Date(iso).toLocaleString();
  } catch {
    return iso ?? "-";
  }
}

type ViewTab = "upcoming" | "past" | "all";

function toIsoSafe(v: string) {
  // datetime-local -> ISO
  if (!v) return "";
  try {
    return new Date(v).toISOString();
  } catch {
    return "";
  }
}

export default function MeetingsPage() {
  const router = useRouter();

  const [items, setItems] = useState<Meeting[]>([]);
  const [loading, setLoading] = useState(false);

  // view / filters
  const [tab, setTab] = useState<ViewTab>("upcoming");
  const [q, setQ] = useState("");
  const [showArchived, setShowArchived] = useState(false);

  // optional date filter (compact)
  const [fromLocal, setFromLocal] = useState("");
  const [toLocal, setToLocal] = useState("");

  // create drawer
  const [createOpen, setCreateOpen] = useState(false);
  const [title, setTitle] = useState("");
  const [startsAt, setStartsAt] = useState(() =>
    new Date().toISOString().slice(0, 16)
  );
  const [endsAt, setEndsAt] = useState("");
  const [creating, setCreating] = useState(false);

  const nowIso = useMemo(() => new Date().toISOString(), []);

  async function load() {
    setLoading(true);
    try {
      // Tab alapú időszűrés
      // upcoming: from=now
      // past: to=now
      // all: semmi
      const base: any = {
        includeArchived: showArchived ? 1 : 0,
        q: q.trim() || undefined,
      };

      if (tab === "upcoming")
        base.from = fromLocal ? toIsoSafe(fromLocal) : nowIso;
      if (tab === "past") base.to = toLocal ? toIsoSafe(toLocal) : nowIso;

      // ha all tab, de a user megadott from/to-t, akkor azokat tiszteletben tartjuk
      if (tab === "all") {
        if (fromLocal) base.from = toIsoSafe(fromLocal);
        if (toLocal) base.to = toIsoSafe(toLocal);
      }

      const r = await ApiCall.meetings.list(base);
      setItems(r.data.meetings ?? []);
    } catch (e: any) {
      toast.error(
        e?.response?.data?.message ?? "Meetingek betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  }

  useEffect(() => {
    void load();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [tab, showArchived]);

  async function onCreate() {
    if (!title.trim()) return;
    setCreating(true);
    try {
      const r = await ApiCall.meetings.create({
        title: title.trim(),
        startsAt: new Date(startsAt).toISOString(),
        endsAt: endsAt ? new Date(endsAt).toISOString() : undefined,
      });
      toast.success("Meeting létrehozva");
      setCreateOpen(false);
      setTitle("");
      setEndsAt("");
      router.push(`/admin/meetings/${r.data.code ?? r.data.id}`);
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Létrehozás sikertelen");
    } finally {
      setCreating(false);
    }
  }

  const stats = useMemo(() => {
    const total = items.length;
    const archived = items.filter((m) => !!m.archived_at).length;
    const upcoming = items.filter(
      (m) => new Date(m.starts_at).getTime() >= Date.now()
    ).length;
    return { total, archived, upcoming };
  }, [items]);

  return (
    <Page
      title="Meetingek"
      subtitle={`összes: ${stats.total} · upcoming a listában: ${stats.upcoming} · archived a listában: ${stats.archived}`}
      actions={
        <>
          <SecondaryButton onClick={() => setCreateOpen(true)}>
            + Új meeting
          </SecondaryButton>
          <SecondaryButton onClick={() => void load()} disabled={loading}>
            {loading ? "..." : "Frissít"}
          </SecondaryButton>
        </>
      }>
      {/* Top control bar: tabs + search + archived toggle */}
      <div className="mb-4 grid gap-3">
        <div className="flex flex-wrap items-center justify-between gap-3">
          <Tabs<ViewTab>
            value={tab}
            onChange={setTab}
            tabs={[
              { id: "upcoming", label: "Upcoming" },
              { id: "past", label: "Past" },
              { id: "all", label: "All" },
            ]}
          />

          <div className="flex flex-wrap items-center gap-2">
            <TextInput
              value={q}
              onChange={(e) => setQ(e.target.value)}
              placeholder="Keresés (cím / kód)…"
              style={{ minWidth: 260 }}
              onKeyDown={(e: any) => {
                if (e.key === "Enter") void load();
              }}
            />

            <Select
              value={showArchived ? "1" : "0"}
              onChange={(e) => setShowArchived(e.target.value === "1")}>
              <option value="0">Csak aktív</option>
              <option value="1">Archiváltal együtt</option>
            </Select>

            <Button onClick={() => void load()} disabled={loading}>
              Keres
            </Button>
          </div>
        </div>

        {/* Compact date range */}
        <Card>
          <ToolbarRow>
            <Field label="From (opcionális)">
              <input
                type="datetime-local"
                value={fromLocal}
                onChange={(e) => setFromLocal(e.target.value)}
                className="w-full rounded-xl border border-slate-200 bg-white px-3 py-2 text-sm shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-600/20 focus:border-blue-600/40"
                style={{ minWidth: 220 }}
              />
            </Field>

            <Field label="To (opcionális)">
              <input
                type="datetime-local"
                value={toLocal}
                onChange={(e) => setToLocal(e.target.value)}
                className="w-full rounded-xl border border-slate-200 bg-white px-3 py-2 text-sm shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-600/20 focus:border-blue-600/40"
                style={{ minWidth: 220 }}
              />
            </Field>

            <SecondaryButton
              onClick={() => {
                setFromLocal("");
                setToLocal("");
              }}>
              Töröl
            </SecondaryButton>
          </ToolbarRow>
        </Card>
      </div>

      <SectionTitle>
        {tab === "upcoming"
          ? "Közelgő meetingek"
          : tab === "past"
          ? "Korábbi meetingek"
          : "Meeting lista"}
      </SectionTitle>

      {items.length ? (
        <GridCards>
          {items.map((m) => (
            <Card
              key={m.id}
              style={{ cursor: "pointer" }}
              onClick={() => router.push(`/admin/meetings/${m.code ?? m.id}`)}>
              <CardTitle>{m.title}</CardTitle>
              <CardMeta>
                {m.code ? `kód: ${m.code} · ` : null}
                {fmt(m.starts_at)}
                {m.ends_at ? ` → ${fmt(m.ends_at)}` : ""}
              </CardMeta>

              <Divider />

              <div className="flex flex-wrap gap-2">
                {m.archived_at ? (
                  <Chip tone="warn">archived</Chip>
                ) : (
                  <Chip>active</Chip>
                )}
                <Chip tone="info">id: {m.id.slice(0, 8)}…</Chip>
              </div>
            </Card>
          ))}
        </GridCards>
      ) : loading ? (
        <Card>
          <CardMeta>Betöltés…</CardMeta>
        </Card>
      ) : (
        <EmptyState
          title="Nincs meeting"
          hint="Próbálj keresést / másik tabot, vagy hozz létre újat."
        />
      )}

      {/* Create drawer */}
      <Drawer
        open={createOpen}
        onClose={() => setCreateOpen(false)}
        title="Új meeting"
        widthClass="max-w-lg">
        <div className="grid gap-3">
          <Field label="Cím" hint="kötelező">
            <TextInput
              value={title}
              onChange={(e) => setTitle(e.target.value)}
              placeholder="Meeting címe"
            />
          </Field>

          <div className="grid gap-3 md:grid-cols-2">
            <Field label="Kezdés">
              <input
                type="datetime-local"
                value={startsAt}
                onChange={(e) => setStartsAt(e.target.value)}
                className="w-full rounded-xl border border-slate-200 bg-white px-3 py-2 text-sm shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-600/20 focus:border-blue-600/40"
              />
            </Field>

            <Field label="Vége" hint="opcionális">
              <input
                type="datetime-local"
                value={endsAt}
                onChange={(e) => setEndsAt(e.target.value)}
                className="w-full rounded-xl border border-slate-200 bg-white px-3 py-2 text-sm shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-600/20 focus:border-blue-600/40"
              />
            </Field>
          </div>

          <ToolbarRow>
            <Button
              onClick={() => void onCreate()}
              disabled={creating || !title.trim()}>
              {creating ? "..." : "Létrehoz"}
            </Button>
            <SecondaryButton
              onClick={() => setCreateOpen(false)}
              disabled={creating}>
              Mégse
            </SecondaryButton>
          </ToolbarRow>

          <div className="text-xs text-slate-500">
            Case-hez kötött meetinghez a case oldalról indítsd:{" "}
            <b>Cases → + Új meeting</b>
          </div>
        </div>
      </Drawer>
    </Page>
  );
}

// "use client";

// import { useEffect, useState } from "react";
// import toast from "react-hot-toast";
// import ApiCall from "@/api/api-calls";
// import type { Meeting } from "@/api/routes/meetings";
// import {
//   Page,
//   Card,
//   ToolbarRow,
//   TextInput,
//   Button,
//   SecondaryButton,
//   GridCards,
//   CardTitle,
//   CardMeta,
//   Chip,
//   SectionTitle,
//   Select,
// } from "@/components/ui/CardBits";
// import { useRouter } from "next/navigation";

// function fmt(iso?: string | null) {
//   if (!iso) return "-";
//   try {
//     return new Date(iso).toLocaleString();
//   } catch {
//     return iso;
//   }
// }

// export default function MeetingsPage() {
//   const router = useRouter();
//   const [items, setItems] = useState<Meeting[]>([]);
//   const [loading, setLoading] = useState(false);

//   const [title, setTitle] = useState("");
//   const [startsAt, setStartsAt] = useState(() =>
//     new Date().toISOString().slice(0, 16)
//   );
//   const [endsAt, setEndsAt] = useState("");
//   const [creating, setCreating] = useState(false);

//   const [showArchived, setShowArchived] = useState(false);

//   async function load() {
//     setLoading(true);
//     try {
//       const r = await ApiCall.meetings.list({
//         includeArchived: showArchived ? 1 : 0,
//       } as any);
//       setItems(r.data.meetings ?? []);
//     } catch (e: any) {
//       toast.error(
//         e?.response?.data?.message ?? "Meetingek betöltése sikertelen"
//       );
//     } finally {
//       setLoading(false);
//     }
//   }

//   useEffect(() => {
//     void load();
//   }, [showArchived]);

//   async function onCreate() {
//     if (!title.trim()) return;
//     setCreating(true);
//     try {
//       const r = await ApiCall.meetings.create({
//         title: title.trim(),
//         startsAt: new Date(startsAt).toISOString(),
//         endsAt: endsAt ? new Date(endsAt).toISOString() : undefined,
//       });
//       toast.success("Meeting létrehozva");
//       router.push(`/admin/meetings/${r.data.code ?? r.data.id}`);
//     } catch (e: any) {
//       toast.error(e?.response?.data?.message ?? "Létrehozás sikertelen");
//     } finally {
//       setCreating(false);
//     }
//   }

//   return (
//     <Page
//       title="Meetingek"
//       subtitle="Agenda + csatolmányok + feladatok egy helyen."
//       actions={
//         <>
//           <SecondaryButton onClick={() => void load()} disabled={loading}>
//             {loading ? "..." : "Frissít"}
//           </SecondaryButton>
//           <Select
//             value={showArchived ? "1" : "0"}
//             onChange={(e) => setShowArchived(e.target.value === "1")}>
//             <option value="0">Csak aktív</option>
//             <option value="1">Archiváltal együtt</option>
//           </Select>
//         </>
//       }>
//       <SectionTitle>Új meeting</SectionTitle>
//       <Card>
//         <ToolbarRow>
//           <TextInput
//             value={title}
//             onChange={(e) => setTitle(e.target.value)}
//             placeholder="Meeting címe"
//             style={{ flex: 1, minWidth: 260 }}
//           />
//           <input
//             type="datetime-local"
//             value={startsAt}
//             onChange={(e) => setStartsAt(e.target.value)}
//             style={{
//               padding: "10px 12px",
//               borderRadius: 10,
//               border: "1px solid #e5e7eb",
//             }}
//           />
//           <input
//             type="datetime-local"
//             value={endsAt}
//             onChange={(e) => setEndsAt(e.target.value)}
//             style={{
//               padding: "10px 12px",
//               borderRadius: 10,
//               border: "1px solid #e5e7eb",
//             }}
//           />
//           <Button
//             onClick={() => void onCreate()}
//             disabled={creating || !title.trim()}>
//             {creating ? "..." : "Létrehoz"}
//           </Button>
//         </ToolbarRow>
//       </Card>

//       <SectionTitle>Legutóbbi meetingek</SectionTitle>
//       <GridCards>
//         {items.map((m) => (
//           <Card
//             key={m.id}
//             style={{ cursor: "pointer" }}
//             onClick={() => router.push(`/admin/meetings/${m.code ?? m.id}`)}>
//             <CardTitle>{m.title}</CardTitle>
//             <CardMeta>
//               {m.code ? `kód: ${m.code} · ` : null}
//               {fmt(m.starts_at)}
//               {m.ends_at ? ` → ${fmt(m.ends_at)}` : ""}
//             </CardMeta>
//             <div
//               style={{
//                 display: "flex",
//                 gap: 8,
//                 flexWrap: "wrap",
//                 marginTop: 10,
//               }}>
//               {m.archived_at ? (
//                 <Chip tone="warn">archived</Chip>
//               ) : (
//                 <Chip>active</Chip>
//               )}
//             </div>
//           </Card>
//         ))}
//       </GridCards>
//       {!items.length && !loading ? (
//         <div style={{ fontSize: 12, opacity: 0.7, marginTop: 10 }}>
//           Nincs meeting.
//         </div>
//       ) : null}
//     </Page>
//   );
// }

// // "use client";

// // import { useEffect, useState } from "react";
// // import toast from "react-hot-toast";
// // import ApiCall from "@/api/api-calls";
// // import type { Meeting } from "@/api/routes/meetings";
// // import {
// //   Page,
// //   Card,
// //   ToolbarRow,
// //   TextInput,
// //   Button,
// //   SecondaryButton,
// //   GridCards,
// //   CardTitle,
// //   CardMeta,
// //   Chip,
// //   SectionTitle,
// // } from "@/components/ui/CardBits";
// // import { useRouter } from "next/navigation";

// // function fmt(iso?: string | null) {
// //   if (!iso) return "-";
// //   try {
// //     return new Date(iso).toLocaleString();
// //   } catch {
// //     return iso;
// //   }
// // }

// // export default function MeetingsPage() {
// //   const router = useRouter();
// //   const [items, setItems] = useState<Meeting[]>([]);
// //   const [loading, setLoading] = useState(false);

// //   const [title, setTitle] = useState("");
// //   const [startsAt, setStartsAt] = useState(() =>
// //     new Date().toISOString().slice(0, 16)
// //   );
// //   const [endsAt, setEndsAt] = useState("");
// //   const [creating, setCreating] = useState(false);

// //   async function load() {
// //     setLoading(true);
// //     try {
// //       const r = await ApiCall.meetings.list();
// //       setItems(r.data.meetings ?? []);
// //     } catch (e: any) {
// //       toast.error(
// //         e?.response?.data?.message ?? "Meetingek betöltése sikertelen"
// //       );
// //     } finally {
// //       setLoading(false);
// //     }
// //   }

// //   useEffect(() => {
// //     void load();
// //   }, []);

// //   async function onCreate() {
// //     if (!title.trim()) return;
// //     setCreating(true);
// //     try {
// //       const r = await ApiCall.meetings.create({
// //         title: title.trim(),
// //         startsAt: new Date(startsAt).toISOString(),
// //         endsAt: endsAt ? new Date(endsAt).toISOString() : undefined,
// //       });
// //       toast.success("Meeting létrehozva");
// //       router.push(`/admin/meetings/${r.data.id}`);
// //     } catch (e: any) {
// //       toast.error(e?.response?.data?.message ?? "Létrehozás sikertelen");
// //     } finally {
// //       setCreating(false);
// //     }
// //   }

// //   return (
// //     <Page
// //       title="Meetingek"
// //       subtitle="Agenda + csatolmányok + feladatok egy helyen."
// //       actions={
// //         <>
// //           <SecondaryButton onClick={() => void load()} disabled={loading}>
// //             {loading ? "..." : "Frissít"}
// //           </SecondaryButton>
// //         </>
// //       }>
// //       <SectionTitle>Új meeting</SectionTitle>
// //       <Card>
// //         <ToolbarRow>
// //           <TextInput
// //             value={title}
// //             onChange={(e) => setTitle(e.target.value)}
// //             placeholder="Meeting címe"
// //             style={{ flex: 1, minWidth: 260 }}
// //           />
// //           <input
// //             type="datetime-local"
// //             value={startsAt}
// //             onChange={(e) => setStartsAt(e.target.value)}
// //             style={{
// //               padding: "10px 12px",
// //               borderRadius: 10,
// //               border: "1px solid #e5e7eb",
// //             }}
// //           />
// //           <input
// //             type="datetime-local"
// //             value={endsAt}
// //             onChange={(e) => setEndsAt(e.target.value)}
// //             style={{
// //               padding: "10px 12px",
// //               borderRadius: 10,
// //               border: "1px solid #e5e7eb",
// //             }}
// //           />
// //           <Button
// //             onClick={() => void onCreate()}
// //             disabled={creating || !title.trim()}>
// //             {creating ? "..." : "Létrehoz"}
// //           </Button>
// //         </ToolbarRow>
// //       </Card>

// //       <SectionTitle>Legutóbbi meetingek</SectionTitle>
// //       <GridCards>
// //         {items.map((m) => (
// //           <Card
// //             key={m.id}
// //             style={{ cursor: "pointer" }}
// //             onClick={() => router.push(`/admin/meetings/${m.id}`)}>
// //             <CardTitle>{m.title}</CardTitle>
// //             <CardMeta>
// //               {fmt(m.starts_at)}
// //               {m.ends_at ? ` → ${fmt(m.ends_at)}` : ""}
// //             </CardMeta>
// //             <div
// //               style={{
// //                 display: "flex",
// //                 gap: 8,
// //                 flexWrap: "wrap",
// //                 marginTop: 10,
// //               }}>
// //               <Chip>id: {m.id.slice(0, 8)}…</Chip>
// //             </div>
// //           </Card>
// //         ))}
// //       </GridCards>
// //       {!items.length && !loading ? (
// //         <div style={{ fontSize: 12, opacity: 0.7, marginTop: 10 }}>
// //           Nincs meeting.
// //         </div>
// //       ) : null}
// //     </Page>
// //   );
// // }

// // // "use client";

// // // import { useEffect, useState } from "react";
// // // import ApiCall from "@/api/api-calls";
// // // import type { Meeting } from "@/api/routes/meetings";
// // // import {
// // //   Card,
// // //   CardTitle,
// // //   CardMeta,
// // //   Chip,
// // //   GridCards,
// // //   ToolbarRow,
// // //   TextInput,
// // //   Button,
// // //   SecondaryButton,
// // //   SectionTitle,
// // // } from "@/components/ui/CardBits";
// // // import { useRouter } from "next/navigation";

// // // function fmtDate(iso?: string | null) {
// // //   if (!iso) return "-";
// // //   try {
// // //     return new Date(iso).toLocaleString();
// // //   } catch {
// // //     return iso ?? "-";
// // //   }
// // // }

// // // export default function MeetingsPage() {
// // //   const router = useRouter();
// // //   const [items, setItems] = useState<Meeting[]>([]);
// // //   const [loading, setLoading] = useState(false);

// // //   const [title, setTitle] = useState("");
// // //   const [startsAt, setStartsAt] = useState(() =>
// // //     new Date().toISOString().slice(0, 16)
// // //   );
// // //   const [endsAt, setEndsAt] = useState("");
// // //   const [creating, setCreating] = useState(false);

// // //   async function load() {
// // //     setLoading(true);
// // //     try {
// // //       const r = await ApiCall.meetings.list();
// // //       setItems(r.data.meetings ?? []);
// // //     } finally {
// // //       setLoading(false);
// // //     }
// // //   }

// // //   useEffect(() => {
// // //     void load();
// // //   }, []);

// // //   async function onCreate() {
// // //     const t = title.trim();
// // //     if (!t) return;
// // //     setCreating(true);
// // //     try {
// // //       const r = await ApiCall.meetings.create({
// // //         title: t,
// // //         startsAt: new Date(startsAt).toISOString(),
// // //         endsAt: endsAt ? new Date(endsAt).toISOString() : undefined,
// // //       });
// // //       setTitle("");
// // //       router.push(`/admin/meetings/${r.data.id}`);
// // //     } finally {
// // //       setCreating(false);
// // //     }
// // //   }

// // //   return (
// // //     <div style={{ padding: 16 }}>
// // //       <ToolbarRow>
// // //         <div style={{ fontWeight: 900, fontSize: 18 }}>Meetings</div>
// // //         <SecondaryButton onClick={() => void load()} disabled={loading}>
// // //           {loading ? "..." : "Frissít"}
// // //         </SecondaryButton>
// // //       </ToolbarRow>

// // //       <SectionTitle>Új meeting</SectionTitle>
// // //       <Card>
// // //         <div style={{ display: "grid", gap: 10 }}>
// // //           <TextInput
// // //             value={title}
// // //             onChange={(e) => setTitle(e.target.value)}
// // //             placeholder="Meeting címe"
// // //           />
// // //           <div style={{ display: "flex", gap: 10, flexWrap: "wrap" }}>
// // //             <input
// // //               type="datetime-local"
// // //               value={startsAt}
// // //               onChange={(e) => setStartsAt(e.target.value)}
// // //               style={{
// // //                 padding: "10px 12px",
// // //                 borderRadius: 10,
// // //                 border: "1px solid #e5e7eb",
// // //               }}
// // //             />
// // //             <input
// // //               type="datetime-local"
// // //               value={endsAt}
// // //               onChange={(e) => setEndsAt(e.target.value)}
// // //               style={{
// // //                 padding: "10px 12px",
// // //                 borderRadius: 10,
// // //                 border: "1px solid #e5e7eb",
// // //               }}
// // //             />
// // //             <Button
// // //               onClick={() => void onCreate()}
// // //               disabled={creating || !title.trim()}>
// // //               {creating ? "Létrehozás..." : "Létrehoz"}
// // //             </Button>
// // //           </div>
// // //         </div>
// // //       </Card>

// // //       <SectionTitle>Meetingek</SectionTitle>
// // //       <GridCards>
// // //         {items.map((m) => (
// // //           <Card
// // //             key={m.id}
// // //             style={{ cursor: "pointer" }}
// // //             onClick={() => router.push(`/admin/meetings/${m.id}`)}>
// // //             <CardTitle>{m.title}</CardTitle>
// // //             <CardMeta>
// // //               {fmtDate(m.starts_at)}
// // //               {m.ends_at ? ` → ${fmtDate(m.ends_at)}` : ""}
// // //             </CardMeta>
// // //             <div
// // //               style={{
// // //                 display: "flex",
// // //                 gap: 8,
// // //                 flexWrap: "wrap",
// // //                 marginTop: 10,
// // //               }}>
// // //               <Chip>id: {m.id.slice(0, 8)}…</Chip>
// // //             </div>
// // //           </Card>
// // //         ))}
// // //       </GridCards>

// // //       {!items.length && !loading ? (
// // //         <div style={{ marginTop: 12, fontSize: 12, opacity: 0.7 }}>
// // //           Nincs meeting.
// // //         </div>
// // //       ) : null}
// // //     </div>
// // //   );
// // // }
